#!/usr/bin/php
<?

/*

SMS Archiver for Siemens Phones v0.1
(c) Richard Low 2004
www.wentnet.com
richard@wentnet.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

/*
  convert.php

  This file converts smo and smi files from a Siemens phone to a human readable format for message archiving.

  To use, modify the path above to where your php binary is located.
  
  To convert a file foo.smi, simply run

    /path/to/convert.php foo.smi
  
  or for >1 files,
  
    /path/to/convert.php foo1.smi foo2.smi

  or

    /path/to/convert.php *

  for all files in the current directory.  The script will output the files to the current directory.
*/

require("vcard.php");

$debug = false;

for ($i = 1; $i < $argc; $i++)
{
  if (strtolower(substr($argv[$i], strlen($argv[$i])-3)) == "smo")
    convertsmo($argv[$i]);
  else if (strtolower(substr($argv[$i], strlen($argv[$i])-3)) == "smi")
    convertsmi($argv[$i]);
}

function convertsmo($filename)
{
  global $debug;

  echo "converting $filename...\n";

  $data = file_get_contents($filename);
  $number = convertPhoneNumber($data, 28);
  $name = getName($number);
  $numberOfMessages = ord(substr($data, 5, 1));

  if ($debug)
    echo "Number of messages: $numberOfMessages\n";

  if ($numberOfMessages > 1)
  {
    $message = "";
    for ($m = 0; $m < $numberOfMessages; $m++)
    {
      if ($m == 0)
        $start = 28+5+ceil(getPhoneNumberLength($data, 28)/2);
      else if ($m == 1)
        $start = 193 +16+ceil(getPhoneNumberLength($data, 28)/2);
      else
        $start += 176;
      if ($debug)
        echo "$start\n";
      // lose first 8 characters for some reason...
      $message .= substr(convertMessage($data, $start), 8);
      if ($debug)
        echo "Message: $message\n";
    }
  }
  else
  {
    $message = convertMessage($data, 28+5+ceil(getPhoneNumberLength($data, 28)/2));
  }

  writeFile($name, $number, $date, $message, false);
}

function convertsmi($filename)
{
  global $debug;

  echo "converting $filename...\n";

  $data = file_get_contents($filename);
  $number = convertPhoneNumber($data, 27);
  $name = getName($number);
  $numberOfMessages = ord(substr($data, 5, 1));

  $date = convertDate($data, 31+ceil(getPhoneNumberLength($data, 27)/2));

  if ($debug)
    echo "Number of messages: $numberOfMessages\n";

  if ($numberOfMessages > 1)
  {
    $message = "";
    for ($m = 0; $m < $numberOfMessages; $m++)
    {
      if ($m == 0)
        $start = 38+ceil(getPhoneNumberLength($data, 27)/2);
      else if ($m == 1)
        $start = 214+ceil(getPhoneNumberLength($data, 27)/2);
      else
        $start += 176;
      if ($debug)
        echo "$start\n";
      // lose first 7 characters for some reason...
      $message .= substr(convertMessage($data, $start), 7);
      if ($debug)
        echo "Message: $message\n";
    }
  }
  else
  {
    $message = convertMessage($data, 38+ceil(getPhoneNumberLength($data, 27)/2));
  }

  writeFile($name, $number, $date, $message, true);
}

function getPhoneNumberLength($data, $start)
{
  global $debug;

  $length_s = substr($data, $start, 1);
  return ord($length_s);
}

function convertPhoneNumber($data, $start)
{
  global $debug;

  $length = getPhoneNumberLength($data, $start);

  if ($debug)
    echo "Phone number length: $length\n";

  // ignore number type byte at pos $start+1
  $phoneNumber_s = substr($data, $start+2, ceil($length/2));

  $number = "";

  for ($i = 0; $i < $length/2; $i++)
  {
    $char = substr($phoneNumber_s, $i, 1);
    $number_s = dechex(ord($char));
    if (strlen($number_s) == 1)
      $number_s = '0' . $number_s;
    $number .= strrev($number_s);
  }

  if ($length % 2 != 0)
     $number = substr($number, 0, $length);

   return $number;
}

function convertMessage($data, $start)
{
  global $debug;

  $length_s = substr($data, $start, 1);
  $length = ord($length_s);
  $length_octets = ceil($length*7/8);

  if ($debug)
    echo "Message Length: $length\n";

  $message_s = substr($data, $start+1, $length_octets);

  $bin = "";

  for ($i = 0; $i < $length_octets; $i++)
  {
    $char = substr($data, $start+1+$i, 1);
    $bin .= makeEightBit(decbin(ord($char)));
  }

  if ($debug)
    echo "Binary: $bin\n";

  $message = "";

  $start_front = 1;
  $length_front = 7;
  $start_end = -8;
  $length_end = 0; 
  for ($i = 0; $i < $length; $i++)
  {
    if ($debug)
      echo "$start_front, $length_front + $start_end, $length_end\n";
    $char_bin = '0' . substr($bin, $start_front, $length_front) . substr($bin, $start_end, $length_end);
    if ($debug)  
      echo "$char_bin " . chr(bindec($char_bin)) . "\n";
    $message .= chr(bindec($char_bin));

    if ($length_front == 0)
    {
      $start_front -= 7;
      $length_front = 7;
      $length_end = 0;
    }
    else
    {
      $start_front += 9;
      $length_front--;
      $length_end++;
      $start_end += 8;
    }
  }

  return $message;
}

// prepends 0s to $bin to make it 8 long, fails if >8 long, no change if 8 long
function makeEightBit($bin)
{
  global $debug;

  return str_repeat("0", 8-strlen($bin)) . $bin;
}

function convertDate($data, $start)
{
  global $debug;

  if ($debug)
    echo "Date start: $start\n";
  $date_s = substr($data, $start, 6);
  $date = '';
  for ($i = $start; $i < $start+6; $i++)
  {
    $no = dechex(ord(substr($data, $i, 1)));
    if (strlen($no) == 1)
      $no = '0' . $no;
    $date .= strrev($no);
  }

  // two digits dates
  if (substr($date, 0, 1) == "0")
    return '20' . $date;
  else
    return '19' . $date;
}

function writeFile($name, $number, $date, $message, $from)
{
  if ($date != "")
    $filename = $name . '-' . $date;
  else
    $filename = $name;

  $filename = getUniqueFilename($filename);

  if (!$handle = fopen($filename, 'w')) {
    echo "Cannot create file ($filename)";
    return;
  }

  if ($from === true)
    $data = "From: $name";
  else
    $data = "To: $name";
  $data .= "\nNumber: $number\nDate: $date\n$message\n";

  if (fwrite($handle, $data) === FALSE) {
    echo "Cannot write to file ($filename)";
    return;
  }

  echo $data;
}

function getUniqueFilename($filename)
{
  if (!file_exists($filename))
    return $filename;
  return tempnam(getcwd(), $filename . '-');
}
?>