//
//  ID3Tagger.m
//  id3libtest
//
//  Created by Richard Low on Mon Jul 26 2004.
//

/* this class provides a wrapper for id3lib
 * to read/write id3 tags
 */

#import "ID3Tagger.h"
#include <id3/tag.h>
#include <id3/misc_support.h>
#import "MP3Header.h"

@interface ID3Tagger (PrivateAPI)
// NB these can't be delcared in ID3Tagger.h as would then include id3/tag.h in other files which causes errors
// unless a -I./id3lib/include compiler flag is set
// but should be private anyway
- (NSString *)frameContentsFromTag:(ID3_Tag *)tag frameID:(ID3_FrameID)frameID;
- (int)intFrameContentsFromTag:(ID3_Tag *)tag frameID:(ID3_FrameID)frameID;
- (void)setFrameContentsForTag:(ID3_Tag *)tag frameID:(ID3_FrameID)frameID contents:(NSString *)contents;

- (unsigned)wavFileLength:(NSString *)path;
@end

@implementation ID3Tagger

// init/dealloc methods

- (id)init
{
	return [self initWithFilename:nil];
}

- (id)initWithFilename:(NSString *)newFilename
{
	if (self = [super init])
	{
		[self setFilename:newFilename];
	}
	return self;
}

- (void)dealloc
{
	[filename release];
	[fileExtensionsToCodecsDictionary release];
	[super dealloc];
}

// accessor methods

- (void)setFilename:(NSString *)newFilename
{
	[newFilename retain];
	[filename release];
	filename = newFilename;
}

- (void)setFileExtensionsToCodecsDictionary:(NSDictionary *)newFileExtensionsToCodecsDictionary
{
	[newFileExtensionsToCodecsDictionary retain];
	[fileExtensionsToCodecsDictionary release];
	fileExtensionsToCodecsDictionary = newFileExtensionsToCodecsDictionary;
}

/**********************/

/* returns the codec for the current file
 * based on extension
 */
- (codecTypes)codec
{
	NSEnumerator *enumerator = [fileExtensionsToCodecsDictionary keyEnumerator];
	NSString *key;
	
	while (key = [enumerator nextObject]) {
		if ([[filename pathExtension] compare:key options:NSCaseInsensitiveSearch] == NSOrderedSame)
		{
			return [[fileExtensionsToCodecsDictionary objectForKey:key] intValue];
		}
	}
	return CODEC_UNDEFINED;
}

/* read the id3 tag and return it as
 * a track object
 * will get the length of wav files but not wma
 */
- (Track *)readTrack
{	
	// check file exists
	NSFileManager *fileManager = [[NSFileManager alloc] init];
	// this is used below to get filesize
	NSDictionary *fileAttributes = [fileManager fileAttributesAtPath:filename traverseLink:YES];
	if (fileAttributes == nil)
		return nil;
	
	Track *track = [[Track alloc] init];
		
	[track setFilesize:[[fileAttributes objectForKey:NSFileSize] intValue]];
	[track setCodec:[self codec]];
	
	if ([self codec] == CODEC_MP3)
	{
		ID3_Tag myTag;
		myTag.Link([filename UTF8String]);
		
		[track setTitle:[self frameContentsFromTag:&myTag frameID:ID3FID_TITLE]];
		[track setAlbum:[self frameContentsFromTag:&myTag frameID:ID3FID_ALBUM]];
		[track setArtist:[self frameContentsFromTag:&myTag frameID:ID3FID_LEADARTIST]];
		
		NSString *genre = [self frameContentsFromTag:&myTag frameID:ID3FID_CONTENTTYPE];
		if (genre != nil && [genre length] > 0)
		{
			// genre may be of the form (ddd) and be an index so look...
			// test for open bracket
			if ([genre characterAtIndex:0] == 40)
			{
				NSRange bracketPos = [genre rangeOfCharacterFromSet:[NSCharacterSet characterSetWithCharactersInString:@")"]];
				if (bracketPos.length == 1 && bracketPos.location == [genre length] - 1)
				{
					NSRange numberRange;
					numberRange.length = [genre length] - 2;
					numberRange.location = 1;
					NSString *numberString = [genre substringWithRange:numberRange];
					int number = [numberString intValue];
					if (number >= 0 && number < ID3_NR_OF_V1_GENRES)
					{
						genre = [NSString stringWithCString:ID3_v1_genre_description[number]];
					}
				}
			}
		}
		[track setGenre:genre];

		// this may be of the form a/b for track a out of b but parsing to integer takes care of this
		[track setTrackNumber:[self intFrameContentsFromTag:&myTag frameID:ID3FID_TRACKNUM]];
		[track setYear:[self intFrameContentsFromTag:&myTag frameID:ID3FID_YEAR]];
		
		// read the length from the header
		MP3Header *header = [[MP3Header alloc] init];
		[track setLength:[header length:filename]];
		[header release];
		
		// why can't I use id3lib like this? always returns NULL
		/*const Mp3_Headerinfo *header = myTag.GetMp3HeaderInfo();
		[track setLength:header->time];*/
		
		myTag.Clear();
		
		[track setFullPath:filename];
	}
	else
	{
		// use filename as title, but strip extension
		NSString *title = [[filename lastPathComponent] stringByDeletingPathExtension];
		[track setTitle:title];
		// todo: can we get the length of WMA files?
		if ([self codec] == CODEC_WAV)
		{
			[track setLength:[self wavFileLength:filename]];
		}
	}
	
	[fileManager release];
	return [track autorelease];
}

/* gets the contents of the frame frameID as an NSString
 */
- (NSString *)frameContentsFromTag:(ID3_Tag *)tag frameID:(ID3_FrameID)frameID
{
	NSString *ret = nil;
	ID3_Frame *frame = tag->Find(frameID);
	if (frame != NULL)
	{
		char *str = ID3_GetString(frame, ID3FN_TEXT);
		if (str != NULL)
		{
			ret = [NSString stringWithCString:str];
			delete [] str;
		}
	}
	return ret;
}

/* returns an integer value from the frame
 */
- (int)intFrameContentsFromTag:(ID3_Tag *)tag frameID:(ID3_FrameID)frameID
{
	return [[self frameContentsFromTag:tag frameID:frameID] intValue];
}

/* write the string contents to the frame frameiD
 */
- (void)setFrameContentsForTag:(ID3_Tag *)tag frameID:(ID3_FrameID)frameID contents:(NSString *)contents
{
	ID3_Frame *frame = new ID3_Frame(frameID);
	if (frame)
	{
		ID3_Field *field = frame->GetField(ID3FN_TEXT);
		if (field != NULL)
		{
			field->Set([contents cString]);
		}
		tag->AttachFrame(frame);
	}
}

/* write the tag in track to the file
 * in [track fullPath]
 */
- (void)writeTrack:(Track *)track
{
	ID3_Tag myTag;
	myTag.Link([[track fullPath] UTF8String]);
	
	ID3_RemoveTitles(&myTag);
	[self setFrameContentsForTag:&myTag frameID:ID3FID_TITLE contents:[track title]];
	ID3_RemoveAlbums(&myTag);
	[self setFrameContentsForTag:&myTag frameID:ID3FID_ALBUM contents:[track album]];
	ID3_RemoveArtists(&myTag);
	[self setFrameContentsForTag:&myTag frameID:ID3FID_LEADARTIST contents:[track artist]];
	ID3_RemoveGenres(&myTag);
	[self setFrameContentsForTag:&myTag frameID:ID3FID_CONTENTTYPE contents:[track genre]];
		
	int nTotal = 0;
	ID3_Frame *frame = myTag.Find(ID3FID_TRACKNUM);
	if (frame != NULL)
	{
		ID3_Field* field = frame->GetField(ID3FN_TEXT);
		if (field != NULL)
		{
			NSString *trackNum = [NSString stringWithCString:field->GetRawText()];
			NSRange slashPos = [trackNum rangeOfCharacterFromSet:[NSCharacterSet characterSetWithCharactersInString:@"/"]];
			if (slashPos.length == 1 && slashPos.location < [trackNum length])
			{
				NSRange numberRange;
				numberRange.length = [trackNum length] - slashPos.location - 1;
				numberRange.location = slashPos.location + 1;
				NSString *numberString = [trackNum substringWithRange:numberRange];
				nTotal = [numberString intValue];
			}
		}
	}

	ID3_AddTrack(&myTag, [track trackNumber], nTotal, true);
	
	ID3_RemoveYears(&myTag);
	[self setFrameContentsForTag:&myTag frameID:ID3FID_YEAR contents:[NSString stringWithFormat:@"%d", [track year]]];

	myTag.Update();
	myTag.Clear();
}

/* get the length of the wav file at path
 * see http://ccrma.stanford.edu/courses/422/projects/WaveFormat/ for wave format spec
 */
- (unsigned)wavFileLength:(NSString *)path
{
	// open the file handle for the specified path
  NSFileHandle *file = [NSFileHandle fileHandleForReadingAtPath:path];
  if (file == nil)
  {
		NSLog(@"Can not open file :%@", path);
    return 0;
	}
	
	[file seekToFileOffset:28];
	
	NSData *buffer = [file readDataOfLength:4];
	const unsigned char *bytesPerSecondChar = (const unsigned char*)[buffer bytes];
	unsigned int bytesPerSecond = bytesPerSecondChar[0] + (bytesPerSecondChar[1] << 8) + (bytesPerSecondChar[2] << 16) + (bytesPerSecondChar[3] << 24);
	
	[file seekToFileOffset:40];
	
	buffer = [file readDataOfLength:4];
	const unsigned char *bytesLongChar = (const unsigned char*)[buffer bytes];
	unsigned int bytesLong = bytesLongChar[0] + (bytesLongChar[1] << 8) + (bytesLongChar[2] << 16) + (bytesLongChar[3] << 24);
	
	[file closeFile];
	
	if (bytesPerSecond == 0)
	{
		NSLog(@"bytesPerSecond == 0 in file %@", path);
		return 0;
	}
	// round up to be on the safe side
	if (bytesLong % bytesPerSecond == 0)
		return (bytesLong / bytesPerSecond);
	else
		return (bytesLong / bytesPerSecond) + 1;
}

@end
