//
//  NJB.m
//  XNJB
//
//  Created by Richard Low on Wed Jul 21 2004.
//

/* An obj-c wrapper for libnjb
 * most methods return an NJBTransactionResult
 * to indicate success/failure
 */

#import "NJB.h"
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#import "Track.h"
#include "string.h"
#include "njbusb.h"
#import "defs.h"

// declared so can be accessed from C function progress
// set to be the outlet statusDisplayer in awakeFromNib
StatusDisplayer *statusDisplayerGlobal;

// declare the private methods
@interface NJB (PrivateAPI)
- (void)updateDiskSpace;
@end

@implementation NJB

// init/dealloc methods

- (id)init
{
	if (self = [super init])
	{
		[self setDebug:0];
	}
	return self;
}

- (void)dealloc
{
	[statusDisplayerGlobal release];
	[super dealloc];
}

// accessor methods

- (void)setDebug:(unsigned)newDebug
{
	// see libnjb.h for the Debug flags
	debug = newDebug;
}

- (statusTypes)status
{
	return status;
}

- (void)setStatus:(statusTypes)newStatus
{
	status = newStatus;
	[statusDisplayer setStatus:status];
}

- (BOOL)isConnected
{
	return connected;
}

/**********************/

- (void)awakeFromNib
{
	statusDisplayerGlobal = [statusDisplayer retain];
}

/* connect to the NJB
 */
- (NJBTransactionResult *)connect
{
	njb_t njbs[NJB_MAX_DEVICES];
	int n;
	
	if ( debug ) NJB_Set_Debug(debug);
	
	if ( NJB_Discover(njbs, 0, &n) == -1 ) {
		njb_error_dump(stderr);
	}
	
	if ( n == 0 ) {
		[self setStatus:STATUS_NO_NJB];
		connected = NO;
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	}
	
	njb= njbs;
	if ( NJB_Open(njb) == -1 ) {
		njb_error_dump(stderr);
		[self setStatus:STATUS_COULD_NOT_OPEN];
		connected = NO;
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	}
	
	if ( NJB_Capture(njb) == -1 ) {
		njb_error_dump(stderr);
		[self setStatus:STATUS_COULD_NOT_CAPTURE];
		[self disconnect];
		connected = NO;
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	}
	
	[self setStatus:STATUS_CONNECTED];
	connected = YES;
	
	[self updateDiskSpace];
	
	// enable UTF8
	NJB_Set_Unicode(NJB_UC_UTF8);
	
	// send notification
	[statusDisplayer postNotificationName:NOTIFICATION_CONNECTED object:self];
	
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

/* disconnect from the NJB
 */
- (void)disconnect
{
	if ([self isConnected])
	{
		NSLog(@"Releasing...");
		NJB_Release(njb);
		NSLog(@"Disconnecting...");
		NJB_Close(njb);
		connected = NO;
		[self setStatus:STATUS_DISCONNECTED];
		
		// send notification
		[statusDisplayer postNotificationName:NOTIFICATION_DISCONNECTED object:self];
	}
}

- (NSString *)ownerString
{
	char *ownerString = NJB_Get_Owner_String (njb);
	NSString *nsOwnerString = [[NSString alloc] initWithUTF8String:ownerString];
	free(ownerString);
	return [nsOwnerString autorelease];
}

- (MyMutableArray *)tracks
{
	if (![self isConnected])
		return nil;
	MyMutableArray *tracks = [[MyMutableArray alloc] init];
	NJB_Reset_Get_Track_Tag(njb);
	songid_t *songtag;
	while ((songtag = NJB_Get_Track_Tag(njb))) {
		Track *track = [[Track alloc] init];
		
		const char *str;
		str = songid_label(songtag, FR_TITLE);
		if (str != NULL)
			[track setTitle:[NSString stringWithUTF8String:str]];
		str = songid_label(songtag, FR_ALBUM);
		if (str != NULL)
			[track setAlbum:[NSString stringWithUTF8String:str]];
		str = songid_label(songtag, FR_ARTIST);
		if (str != NULL)
			[track setArtist:[NSString stringWithUTF8String:str]];
		str = songid_label(songtag, FR_GENRE);
		if (str != NULL)
			[track setGenre:[NSString stringWithUTF8String:str]];
		str = songid_label(songtag, FR_FNAME);
		// this appears to be always blank on mine
		if (str != NULL)
			[track setFilename:[NSString stringWithUTF8String:str]];
		[track setFilesize:(unsigned)songid_size(songtag)];
		[track setLength:(unsigned)songid_length(songtag)];
		if ([self isProtocol3Device])
		{
			// track number a string for NJB3
			str = songid_label(songtag, FR_TRACK);
			if (str != NULL)
			{	
				NSString *trackNumber = [NSString stringWithCString:str];
				[track setTrackNumber:(unsigned)[trackNumber intValue]];
			}
		}
		else
			[track setTrackNumber:(unsigned)songid_track(songtag)];
		
		str = songid_label(songtag, FR_CODEC);
		if (strcmp(str, NJB_CODEC_MP3) == 0)
			[track setCodec:CODEC_MP3];
		else if (strcmp(str, NJB_CODEC_WMA) == 0)
			[track setCodec:CODEC_WMA];
		else if (strcmp(str, NJB_CODEC_WAV) == 0)
			[track setCodec:CODEC_WAV];
		else
			[track setCodec:CODEC_UNDEFINED];
		
		[track setTrackID:songtag->trid];
		
		str = songid_label(songtag, FR_YEAR);
		if (str != NULL)
		{
			NSString *yearNumber = [NSString stringWithCString:str];
			[track setYear:(unsigned)[yearNumber intValue]];
		}
		
		[tracks addObject:track];
		[track release];
		
		songid_destroy(songtag);
	}
	return [tracks autorelease];
}

/* uploads the track and sets the track id
 */
- (NJBTransactionResult *)uploadTrack:(Track *)track
{
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	NSLog(@"Uploading track: %@", [track fullPath]);
	[self setStatus:STATUS_UPLOADING_TRACK];
	int trackid;
	if (NJB_Send_Track(njb, [[track fullPath] UTF8String], [[track njbCodec] UTF8String], [[track title] UTF8String], [[track album] UTF8String], [[track genre] UTF8String],
											[[track artist] UTF8String], [track length], [track trackNumber], [[NSString stringWithFormat:@"%d", [track year]] UTF8String], 0, progress, NULL, &trackid) == -1 ) {
		njb_error_dump(stderr);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	} else {
		[self updateDiskSpace];
		[track setTrackID:trackid];
		return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
	}
}

- (NJBTransactionResult *)deleteTrack:(NSNumber *)trackID
{
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	if (NJB_Delete_Track(njb, [trackID unsignedIntValue]) == -1)
	{
		njb_error_dump(stderr);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	}
	else
	{
		[self updateDiskSpace];
		return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
	}
		
}

- (NJBTransactionResult *)deleteFile:(NSNumber *)fileID
{
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	if (NJB_Delete_Datafile(njb, [fileID unsignedIntValue]) == -1)
	{
		njb_error_dump(stderr);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	}
	else
	{
		[self updateDiskSpace];
		return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
	}
}

- (NJBTransactionResult *)downloadTrack:(Track *)track
{
	if (NJB_Get_Track(njb, [track trackID], [track filesize], [[track fullPath] UTF8String], progress, NULL) == -1)
	{
		njb_error_dump(stderr);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

// todo: does NJB_Replace_Track_Tag ever fail?
- (NJBTransactionResult *)changeTrackTag:(Track *)track
{
	if (NJB_Replace_Track_Tag(njb, [track trackID], [[track njbCodec] UTF8String], [[track title] UTF8String], [[track album] UTF8String],
														 [[track genre] UTF8String], [[track artist] UTF8String], [track length], [track trackNumber],
														[track filesize], [[track filename] UTF8String], [[NSString stringWithFormat:@"%d", [track year]] UTF8String], 0 ) == -1)
	{
		njb_error_dump(stderr);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	}
	else
	{
		[self updateDiskSpace];
		return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
	}
}

- (MyMutableArray *)dataFiles
{
	if (![self isConnected])
		return nil;
	MyMutableArray *dataFiles = [[MyMutableArray alloc] init];
	NJB_Reset_Get_Datafile_Tag(njb);
	datafile_t *filetag;
	while ((filetag = NJB_Get_Datafile_Tag (njb)))
	{
		DataFile *dataFile = [[DataFile alloc] init];
		const char *str = filetag->filename;
		if (str != NULL)
			[dataFile setFilename:[NSString stringWithUTF8String:str]];
		[dataFile setSize:datafile_size(filetag)];
//		[dataFile setTimestampSince1970:filetag->timestamp];
		[dataFile setFileID:filetag->dfid];
		
		[dataFiles addObject:dataFile];
		[dataFile release];
		
		datafile_destroy(filetag);
	}
	return [dataFiles autorelease];
}

- (NJBTransactionResult *)downloadFile:(DataFile *)dataFile
{
	if (NJB_Get_File(njb, [dataFile fileID], [dataFile size], [[dataFile fullPath] UTF8String], progress, NULL) == -1)
	{
		njb_error_dump(stderr);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)uploadFile:(DataFile *)dataFile
{
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	NSLog(@"Uploading file: %@", [dataFile fullPath]);
	[self setStatus:STATUS_UPLOADING_FILE];
	int fileid;
	if (NJB_Send_File(njb, [[dataFile fullPath] UTF8String], [[[dataFile fullPath] lastPathComponent] UTF8String], progress, NULL, &fileid) == -1 ) {
		njb_error_dump(stderr);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	} else {
		[dataFile setFileID:fileid];
		[self updateDiskSpace];
		return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
	}
}

- (unsigned long long)totalDiskSpace
{
	return totalDiskSpace;
}

- (unsigned long long)freeDiskSpace
{
	return freeDiskSpace;
}

- (void)updateDiskSpace
{
	NJB_Get_Disk_Usage(njb, &totalDiskSpace, &freeDiskSpace);
	[statusDisplayer updateDiskSpace:totalDiskSpace withFreeSpace:freeDiskSpace];
}

- (NSCalendarDate *)jukeboxTime
{
	njb_time_t *time = NJB_Get_Time(njb);
	
	// assume njb time is our timezone
	NSCalendarDate *date = [NSCalendarDate dateWithYear:time->year month:time->month day:time->day hour:time->hours minute:time->minutes second:time->seconds
																						 timeZone:[NSTimeZone localTimeZone]];
	return date;
}

- (BOOL)isProtocol3Device
{
	return (PROTOCOL3_DEVICE(njb->device_type));
}

- (NJBTransactionResult *)setOwnerString:(NSString *)owner
{
	if (NJB_Set_Owner_String (njb, [owner UTF8String]) == -1)
	{
		njb_error_dump(stderr);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)setBitmap:(NSString *)bitmapPath
{
	if (NJB_Set_Bitmap (njb, [bitmapPath UTF8String]) == -1)
	{
		njb_error_dump(stderr);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)setTime:(NSNumber *)timeIntervalSinceNow
{
	njb_time_t time;
	
	NSTimeInterval jukeboxTimeInterval = [timeIntervalSinceNow doubleValue];
	NSCalendarDate *date = [NSCalendarDate dateWithTimeIntervalSinceNow:jukeboxTimeInterval];
	
	time.year = [date yearOfCommonEra];
	time.month = [date monthOfYear];
	time.day = [date dayOfMonth];
	time.weekday = [date dayOfWeek];
	time.hours = [date hourOfDay];
	time.minutes = [date minuteOfHour];
	time.seconds = [date secondOfMinute];
	
	if (NJB_Set_Time(njb, &time) == -1)
	{
		njb_error_dump(stderr);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (MyMutableArray *)playlists
{
	if (![self isConnected])
		return nil;
	
	MyMutableArray *playlists = [[MyMutableArray alloc] init];
	NJB_Reset_Get_Playlist(njb);
	playlist_t *playlist;
	while ((playlist = NJB_Get_Playlist(njb)))
	{
		Playlist *newPlaylist = [[Playlist alloc] init];
				
		const char *str = playlist->name;
		if (str != NULL)
			[newPlaylist setName:[NSString stringWithUTF8String:str]];
		[newPlaylist setPlaylistID:playlist->plid];
		
		playlist_track_t *track;
		playlist_reset_gettrack(playlist);
		while ((track = playlist_gettrack(playlist)))
		{
			Track *newTrack = [[Track alloc] init];
			[newTrack setTrackID:track->trackid];
			[newPlaylist addTrack:newTrack];
			[newTrack release];
		}
		
		playlist_destroy(playlist);
		
		[newPlaylist setState:NJB_PL_UNCHANGED];
		
		[playlists addObject:newPlaylist];
		[newPlaylist release];
	}
	return [playlists autorelease];
}

/* updates the playlist on the NJB
 * may add a new playlist even if playlist state is not new
 * so the playlist ID may change.
 * This is implemented by modifying libnjb (njb3_add_multiple_tracks_to_playlist)
 * for the NJB3 protocol
 */
- (NJBTransactionResult *)updatePlaylist:(Playlist *)playlist
{
	[playlist lock];
	// don't do anything if it's unchanged
	if ([playlist state] == NJB_PL_UNCHANGED)
	{
		[playlist unlock];
		return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
	}
	
	// create new playlist_t object
	playlist_t *pl = playlist_new();
	playlist_set_name(pl, [[playlist name] UTF8String]);
	pl->plid = [playlist playlistID];
	NSEnumerator *enumerator = [[playlist tracks] objectEnumerator];
	Track *currentTrack;
	while (currentTrack = [enumerator nextObject])
	{
		playlist_track_t *newTrack = playlist_track_new([currentTrack trackID]);
		playlist_addtrack(pl, newTrack, NJB_PL_END);
	}
	pl->_state = [playlist state];
	// set this now as as far as the caller is concerned the NJB is updated
	[playlist setState:NJB_PL_UNCHANGED];
	[playlist unlock];
	if (NJB_Update_Playlist(njb, pl) == -1)
	{
		playlist_destroy(pl);
		njb_error_dump(stderr);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	}
	[playlist lock];
	// update the ID as this is changed when tracks added/removed as a new playlist is created
	[playlist setPlaylistID:pl->plid];
	[playlist unlock];
	playlist_destroy(pl);
	
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)deletePlaylist:(Playlist *)playlist
{
	[playlist lock];
	unsigned plid = [playlist playlistID];
	[playlist unlock];
	if (NJB_Delete_Playlist(njb, plid) == -1)
	{
		njb_error_dump(stderr);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

/* called by the file/track transfer functions of libnjb
 * to tell us the progress
 */
int progress(u_int64_t sent, u_int64_t total, const char* buf, unsigned len, void *data)
{
	double percent = (double)sent*100.0/(double)total;
	[statusDisplayerGlobal updateTaskProgress:percent];
	
	return 0;
}

@end
