#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include "libnjb.h"
#include "njbusb.h"
#include "protocol.h"
#include "byteorder.h"
#include "njb_error.h"
#include "usb_io.h"
#include "ioutil.h"
#include "defs.h"
#include "base.h"
#include "eax.h"

extern int __sub_depth;


/**
 * Initializes the basic state of the njb->protocol_state for the
 * NJB1-device.
 */
int njb_init_state (njb_t *njb) {
  __dsub= "njb_init_state";

  njb_state_t *state;

  __enter;

  state = malloc(sizeof(njb_state_t));
  if (state == NULL) {
    __leave;
    return -1;
  }

  state->session_updated = 0;
  state->libcount = 0;
  state->first_eax = NULL;
  state->next_eax = NULL;
  njb->protocol_state = (unsigned char *) state;

  __leave;
  return 0;
}

/*
 * Beginning with firmware 2.93, very strange things started happening.
 * For certian functions, a non-zero status code does not automatically
 * imply an error.  Instead, the NJB seems to send a code with the 
 * four low bits cleared (meaning, a success can be 0x20, 0x40, 0x80,
 * etc.).  I have only seen this on a few functions, and I am not sure
 * what it means, but I have implemented it where I've run into it.
 */

int njb_set_library_counter (njb_t *njb, u_int64_t count)
{
	__dsub= "njb_set_library_counter";
	unsigned char data[8];

	__enter;

	memset(data, 0, 8);
	from_64bit_to_njb1_bytes(count, &data[0]);

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
		NJB_CMD_SET_LIBRARY_COUNTER, 0, 0, 8, data) == -1 ) {

		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

#if (DEBUG>=2)
	fprintf(stderr, "Count: 0x%08qx\n", count);
#endif
	__leave;
	return 0;
}

int njb_get_library_counter (njb_t *njb, njblibctr_t *lcount)
{
	__dsub= "njb_get_library_counter";
	unsigned char data[25];

	__enter;

	memset(lcount, 0, sizeof(njblibctr_t));
	memset(data, 0, 25);
	
	if ( usb_setup(njb, UT_READ_VENDOR_OTHER,
		NJB_CMD_GET_LIBRARY_COUNTER, 0, 0, 25, data) == -1 ) {
		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	if ( data[0] & 0xf ) {
		NJB_STATUS(data[0]);
		__leave;
		return -1;
	} else if ( data[0] ) {
		njb_get_library_counter(njb, lcount);
	} else {
		memcpy(&lcount->id, &data[1], 16);
		lcount->count = njb1_bytes_to_64bit(&data[17]);
#if (DEBUG>=2)
		fprintf(stderr, "NJB ID: ");
		data_dump(stderr, lcount->id, 16);
		fprintf(stderr, "Count: 0x%08qx\n", lcount->count);
#endif
	}

	__leave;
	return 0;
}

int njb_ping (njb_t *njb, njbid_t *njbid)
{
	__dsub= "njb_ping";
	ssize_t bread;
	unsigned char data[58];

	__enter;
	
	memset(data, 0, 58);

       	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
       		NJB_CMD_PING, 0, 0, 0, NULL) == -1 ) {
       		NJB_ERROR(EO_USBCTL);
       		__leave;
       		return -1;
       	}

       	if ( (bread= usb_pipe_read(njb, data, 58)) == -1 ) {
       		NJB_ERROR(EO_USBBLK);
       		__leave;
       		return -1;
       	} else if ( bread < 58 ) {
       		NJB_ERROR(EO_RDSHORT);
       		__leave;
       	       	return -1;
       	}

       	if ( data[0] ) {
       		NJB_STATUS(data[0]);
       		__leave;
       		return -1;
       	}
       	memcpy(&njbid->id, &data[1], 16);
       	njbid->fwRel= (u_int8_t) 0;
       	njbid->fwMinor= (u_int8_t) data[19];
       	njbid->fwMajor= (u_int8_t) data[20];
	/* Copy this -- eventually move it into njb_t */
	njb->fwRel = njbid->fwRel;
	njb->fwMinor = njbid->fwMinor;
	njb->fwMajor = njbid->fwMajor;
       	memcpy(&njbid->productName, &data[25], 32);
       	njbid->power= (u_int8_t) data[57];

#if (DEBUG>=2)
	printf("NJB ID   : ");
	data_dump(stdout, &njbid->id, 16);
	printf("Firmware : %d.%d\n", njbid->fwMajor, njbid->fwMinor);
	printf("Product  : %s\n", njbid->productName);
	printf("Power    : %d\n", njbid->power);
#endif
	__leave;
	return 0;
}

int njb_verify_last_command (njb_t *njb)
{
	__dsub= "njb_verify_last_command";
	unsigned char data = '\0';

	__enter;

	if ( usb_setup(njb, UT_READ_VENDOR_OTHER,
		NJB_CMD_VERIFY_LAST_CMD, 0, 0, 1, &data) == -1 ) {

		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	if ( data ) {
		NJB_STATUS(data);
		__leave;
		return -1;
	}

	__leave;
	return 0;
}

int njb_capture (njb_t *njb, int which)
{
	__dsub= "njb_capture";

       	unsigned char data = '\0';
       	int mode= (which == NJB_CAPTURE) ? NJB_CMD_CAPTURE_NJB :
       		NJB_CMD_RELEASE_NJB;

	__enter;

       	if ( usb_setup(njb, UT_READ_VENDOR_OTHER,
       		mode, 0, 0, 1, &data) == -1 ) {

       		NJB_ERROR(EO_USBCTL);
       		__leave;
       		return -1;
       	}

       	if ( data && (data & 0xf) ) {
       		NJB_ERROR(EO_BADSTATUS);
       		__leave;
       		return -1;
       	}

	__leave;
	return 0;
}

int njb_get_track_tag_header (njb_t *njb, njbttaghdr_t *tagh, int cmd)
{
	__dsub= "njb_get_track_tag_header";
	unsigned char data[9];

	__enter;

	memset(data, 0, 9);
	
	if ( usb_setup(njb, UT_READ_VENDOR_OTHER, cmd, 0, 0, 9, data)
		 == -1 ) {

		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	if ( data[0] == NJB_ERR_TRACK_NOT_FOUND ) {
		NJB_ERROR(EO_EOM);
		__leave;
		return -1;
	} else if ( data[0] ) {
		NJB_STATUS(data[0]);
		__leave;
		return -1;
	}

	tagh->trackid = njb1_bytes_to_32bit(&data[1]);
	tagh->size = njb1_bytes_to_32bit(&data[5]);

	__leave;
	return 0;
}

songid_t *njb_get_track_tag (njb_t *njb, njbttaghdr_t *tagh)
{
	__dsub= "njb_get_track_tag";
	u_int16_t msw, lsw;
	ssize_t bread;
	void *data;
	songid_t *song;
	unsigned char *dp;

	__enter;

	data= malloc(tagh->size+5);
	if ( data == NULL ) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return NULL;
	}
	memset(data, 0, tagh->size+5);

	lsw= get_lsw(tagh->trackid);
	msw= get_msw(tagh->trackid);

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
		NJB_CMD_GET_TRACK_TAG, msw, lsw, 0, NULL) == -1 ) {
		NJB_ERROR(EO_USBCTL);
		free(data);
		__leave;
		return NULL;
	}

	if ( (bread= usb_pipe_read(njb, data, tagh->size+5)) == -1 ) {
		NJB_ERROR(EO_USBBLK);
		free(data);
		__leave;
		return NULL;
	}

	dp= (unsigned char *) data;

	song= songid_unpack(&dp[5], tagh->size);
	if ( song != NULL ) song->trid= tagh->trackid;

	free(data);
	
	__leave;
	return song;
}

int njb_get_playlist_header (njb_t *njb, njbplhdr_t *plh, int cmd)
{
	__dsub= "njb_get_playlist_header";
	unsigned char data[9];

	__enter;

	memset(data, 0, 9);
	
	if ( usb_setup(njb, UT_READ_VENDOR_OTHER, cmd, 0, 0, 9, data)
		 == -1 ) {

		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	if ( data[0] == NJB_ERR_PLAYLIST_NOT_FOUND ) {
		NJB_ERROR(EO_EOM);
		__leave;
		return -1;
	} else if ( (data[0]) & 0xf ) { 
		NJB_STATUS(data[0]);
		__leave;
		return -1;
	} else if ( data[0] ) {
		NJB_ERROR(EO_AGAIN);
		__leave;
		return -1;
	}

	plh->plid = njb1_bytes_to_32bit(&data[1]);
	plh->size = njb1_bytes_to_32bit(&data[5]);

	__leave;
	return 0;
}

playlist_t *njb_get_playlist (njb_t *njb, njbplhdr_t *plh)
{
	__dsub= "njb_get_playlist";
	u_int16_t msw, lsw;
	ssize_t bread;
	void *data;
	playlist_t *pl;
	unsigned char *dp;

	__enter;

	data= malloc(plh->size+5);
	if ( data == NULL ) {
		NJB_ERROR(EO_EOM);
		__leave;
		return NULL;
	}
	memset(data, 0, plh->size+5);

	lsw= get_lsw(plh->plid);
	msw= get_msw(plh->plid);

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
		NJB_CMD_GET_PLAYLIST, msw, lsw, 0, NULL) == -1 ) {

		NJB_ERROR(EO_USBCTL);
		free(data);
		__leave;
		return NULL;
	}

	if ( (bread= usb_pipe_read(njb, data, plh->size+5)) == -1 ) {
		NJB_ERROR(EO_USBBLK);
		free(data);
		__leave;
		return NULL;
	} else if ( bread < plh->size+5 ) {
		NJB_ERROR(EO_RDSHORT);
		free(data);
		__leave;
		return NULL;
	}

	dp= (unsigned char *) data;

	pl= playlist_unpack(&dp[5], plh->size);

	free(data);
	
	__leave;
	return pl;
}

int njb_get_disk_usage (njb_t *njb, u_int64_t *total, u_int64_t *free)
{
	__dsub= "njb_get_disk_usage";
	unsigned char data[17];

	__enter;

	memset(data, 0, 17);
	
	if ( usb_setup(njb, UT_READ_VENDOR_OTHER,
		NJB_CMD_GET_DISK_USAGE, 0, 0, 17, data) == -1 ) {

		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	if ( data[0] & 0xf ) {
		NJB_STATUS(data[0]);
		__leave;
		return -1;
	} else if ( data[0] ) {
		NJB_ERROR(EO_AGAIN);
		__leave;
		return -1;
	}

	*total = njb1_bytes_to_64bit(&data[1]);
	*free = njb1_bytes_to_64bit(&data[9]);

	__leave;
	return 0;
}

int njb_get_owner_string (njb_t *njb, owner_string name)
{
	__dsub= "njb_get_owner_string";
	ssize_t bread;
	char data[OWNER_STRING_LENGTH+1];

	__enter;

	memset(data,0,OWNER_STRING_LENGTH+1);
       	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
       		NJB_CMD_GET_OWNER_STRING, 0, 0, 0, NULL) == -1 ) {

       		NJB_ERROR(EO_USBCTL);
       		__leave;
       		return -1;
       	}

       	if ( (bread= usb_pipe_read(njb, data, OWNER_STRING_LENGTH+1))
       		== -1 ) {

       		NJB_ERROR(EO_USBBLK);
       		__leave;
       		return -1;
       	} else if ( bread < OWNER_STRING_LENGTH+1 ) {
       		NJB_ERROR(EO_RDSHORT);
       		__leave;
       		return -1;
       	}
       	if ( data[0] ) {
       		NJB_STATUS((unsigned char) data[0]);
       		__leave;
       		return -1;
       	}

       	strncpy(name, &data[1], OWNER_STRING_LENGTH);
	name[OWNER_STRING_LENGTH]= '\0'; /* Just in case */

	__leave;
	return 0;
}

int njb_set_owner_string (njb_t *njb, owner_string name)
{
	__dsub= "njb_set_owner_string";
	ssize_t bwritten;

	__enter;

       	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
       		NJB_CMD_SET_OWNER_STRING, 0, 0, 0, NULL) == -1 ) {

       		NJB_ERROR(EO_USBCTL);
       		__leave;
       		return -1;
       	}

       	if ( (bwritten= usb_pipe_write(njb, name, OWNER_STRING_LENGTH))
       		== -1 ) {

       		NJB_ERROR(EO_USBBLK);
       		__leave;
       		return -1;
       	} else if ( bwritten < OWNER_STRING_LENGTH ) {
       		NJB_ERROR(EO_WRSHORT);
       		__leave;
       		return -1;
       	}

	__leave;

	return 0;
}

int njb_get_datafile_header (njb_t *njb, njbdfhdr_t *dfh, int cmd)
{
	__dsub= "njb_get_datafile_header";
	unsigned char data[9];

	__enter;

	memset(data, 0, 9);
	
	if ( usb_setup(njb, UT_READ_VENDOR_OTHER, cmd, 0, 0, 9, data)
		 == -1 ) {

		NJB_ERROR(EO_USBCTL);
		__leave;
		return 0;
	}

	if ( data[0] == 0x60 ) {
		NJB_ERROR(EO_EOM);
		__leave;
		return -1;
	} else if ( data[0] ) {
		NJB_STATUS(data[0]);
		__leave;
		return -1;
	}

	dfh->dfid = njb1_bytes_to_32bit(&data[1]);
	dfh->size = njb1_bytes_to_32bit(&data[5]);

	__leave;
	return 0;
}

datafile_t *njb_get_datafile_tag (njb_t *njb, njbdfhdr_t *dfh)
{
	__dsub= "njb_get_datafile_tag";
	u_int16_t msw, lsw;
	ssize_t bread;
	void *data;
	datafile_t *df;
	unsigned char *dp;

	__enter;

	data= malloc(dfh->size+5);
	if ( data == NULL ) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return NULL;
	}
	memset(data, 0, dfh->size+5);
	
	lsw= get_lsw(dfh->dfid);
	msw= get_msw(dfh->dfid);

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
		NJB_CMD_GET_DATAFILE_TAG, msw, lsw, 0, NULL) == -1 ) {

		NJB_ERROR(EO_USBCTL);
		free(data);
		__leave;
		return NULL;
	}

	if ( (bread= usb_pipe_read(njb, data, dfh->size+5)) == -1 ) {
		NJB_ERROR(EO_USBBLK);
		free(data);
		__leave;
		return NULL;
	} else if ( bread < dfh->size+5 ) {
		NJB_ERROR(EO_RDSHORT);
		free(data);
		__leave;
		return NULL;
	}

	dp= (unsigned char *) data;

	df= datafile_unpack(&dp[5], dfh->size);
	if ( df != NULL ) df->dfid= dfh->dfid;

	free(data);
	__leave;
	return df;
}

/**
 * offset    is the offset into the file, starting at 0.
 * bsize     indicates the recieved buffer max size.
 * bp        points to the recieve buffer
 *           (atleast NJB_XFER_BLOCK_SIZE + NJB_XFER_BLOCK_HEADER_SIZE)
 * lastsort  indicates if last transfer was short (ended before
 *           requested number of bytes were recieved).
 *
 * If lastshort == 1, the last call to this function returned a
 * a short read. In that case, a new setup command shall not be sent,
 * the bus shall just keep retrieveing buffer contents from the
 * bulk pipe.
 */
u_int32_t njb_receive_file_block (njb_t *njb, u_int32_t offset, u_int32_t bsize, 
				  void *bp)
{
	__dsub= "njb_receive_file_block";
	ssize_t bread;
	unsigned char filedata[8];
	unsigned char status = '\0';

	__enter;

	/* Too large transfer requested */
	if ( bsize > NJB_XFER_BLOCK_SIZE ) {
		NJB_ERROR(EO_TOOBIG);
		__leave;
		return -1;
	}

	memset(filedata, 0, 8);
	
	/* Logs indicate that you should send a new retrieve 
	 * command only if last read was not a short read, but
	 * in practice you should, as far as we can see. */
	from_32bit_to_njb1_bytes(offset, &filedata[0]);
	from_32bit_to_njb1_bytes(bsize, &filedata[4]);

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
		       NJB_CMD_RECEIVE_FILE_BLOCK, 0, 0, 8, filedata) == -1 ) {
	  
	  NJB_ERROR(EO_USBCTL);
	  __leave;
	  return -1;
	}

	/* Read the status and block header first */
	if ( (bread = usb_pipe_read(njb, bp, bsize + NJB_XFER_BLOCK_HEADER_SIZE)) == -1 ) {
	  NJB_ERROR(EO_USBBLK);
	  __leave;
	  return -1;
	} else if ( bread < bsize + NJB_XFER_BLOCK_HEADER_SIZE ) {
	  /* This was a short transfer, OK... */
	  /* printf("Short transfer on pipe.\n"); */
	}

	/* Check status in header, should it be if (!lastshort)??? */
	status = ((unsigned char *)bp)[0];
	if (status) {
	  NJB_STATUS(status);
	  __leave;
	  return -1;
	}

	__leave;
	return bread;
}

int njb_request_file (njb_t *njb, u_int32_t fileid)
{
	__dsub= "njb_request_track";
	unsigned char data[4];

	__enter;

	memset(data, 0, 4);
	from_32bit_to_njb1_bytes(fileid, &data[0]);

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
		NJB_CMD_REQUEST_TRACK, 1, 0, 4, data) == -1 ) {
		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	__leave;
	return 0;
}

int njb_transfer_complete (njb_t *njb)
{
	__dsub= "njb_transfer_complete";
	unsigned char data = '\0';

	__enter;

	if ( usb_setup(njb, UT_READ_VENDOR_OTHER,
		NJB_CMD_TRANSFER_COMPLETE, 0, 0, 1, &data) == -1 ) {

		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	if ( data && (data & 0xf) ) {
		NJB_STATUS(data);
		__leave;
		return -1;
	}

	__leave;
	return 0;
}

int njb_send_track_tag (njb_t *njb, njbttaghdr_t *tagh, void *tag)
{
	__dsub= "njb_send_track_tag";
	unsigned char utagsz[4];
	unsigned char data[5];
	ssize_t bread, bwritten;

	__enter;

	memset(utagsz, 0, 4);
	memset(data, 0, 5);
	
	from_32bit_to_njb1_bytes(tagh->size, &utagsz[0]);

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
		NJB_CMD_SEND_TRACK_TAG, 0, 0, 4, utagsz) == -1 ) {

		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	bwritten= usb_pipe_write(njb, tag, tagh->size);
	if ( bwritten == -1 ) {
		NJB_ERROR(EO_USBBLK);
		__leave;
		return -1;
	} else if ( bwritten < tagh->size ) {
		NJB_ERROR(EO_WRSHORT);
		__leave;
		return -1;
	}

	bread= usb_pipe_read(njb, data, 5);
	if ( bread == -1 ) {
		NJB_ERROR(EO_USBBLK);
		__leave;
		return -1;
	} else if ( bread < 5 ) {
		NJB_ERROR(EO_RDSHORT);
		__leave;
		return -1;
	}

	if ( data[0] ) {
		NJB_STATUS(data[0]);
		__leave;
		return -1;
	}

	tagh->trackid = njb1_bytes_to_32bit(&data[1]);

	__leave;
	return 0;
}

int njb_send_datafile_tag (njb_t *njb, njbdfhdr_t *dfh, void *tag)
{
	__dsub= "njb_send_datafile_tag";
	unsigned char utagsz[4];
	unsigned char data[5];
	unsigned char *padded_tag;
	ssize_t bread, bwritten;

	__enter;

	memset(utagsz, 0, 4);
	memset(data, 0, 5);

	/* Add five here for the header overhead */
	from_32bit_to_njb1_bytes(dfh->size+5, &utagsz[0]);

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
		NJB_CMD_SEND_DATAFILE_TAG, 0, 0, 4, utagsz) == -1 ) {

		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}
	
	/* Pad the tag with 0x0000 0x0000 <tag> 0x00
	 * The purpose of these zeroes is unknown. */

	padded_tag = (unsigned char *) malloc(dfh->size+5);
	if ( padded_tag == NULL ) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return -1;
	}

	memset(padded_tag, 0, dfh->size+5);
	memcpy(padded_tag+4, tag, dfh->size);

	/* Write the tag */

	bwritten= usb_pipe_write(njb, padded_tag, dfh->size+5);
	if ( bwritten == -1 ) {
		free(padded_tag);
		NJB_ERROR(EO_USBBLK);
		__leave;
		return -1;
	} else if ( bwritten < dfh->size+5 ) {
		free(padded_tag);
		NJB_ERROR(EO_WRSHORT);
		__leave;
		return -1;
	}

	free(padded_tag);

	/* Read back datafile ID */

	bread = usb_pipe_read(njb, data, 5);
	if ( bread == -1 ) {
		NJB_ERROR(EO_USBBLK);
		__leave;
		return -1;
	} else if ( bread < 5 ) {
		NJB_ERROR(EO_RDSHORT);
		__leave;
		return -1;
	}

	if ( data[0] ) {
		NJB_STATUS(data[0]);
		__leave;
		return -1;
	}

	dfh->dfid = njb1_bytes_to_32bit(&data[1]);

	__leave;
	return 0;
}

/*
 * This function transfers a block of <= NJB_XFER_BLOCK_SIZE to the
 * jukebox and returns the number of bytes actually sent. Short transfers
 * are OK but must be accounted for.
 */
u_int32_t njb_send_file_block (njb_t *njb, void *data, u_int32_t blocksize)
{
	__dsub= "njb_send_file_block";
	u_int32_t bwritten;
	u_int32_t msw, lsw;
	unsigned char status = 0;
	/* We may need to retry this command if the device is not ready
	 * to receive the file block. */
	int retry = 1;
	int retries = 20;

	__enter;

	if ( blocksize > NJB_XFER_BLOCK_SIZE ) {
		NJB_ERROR(EO_TOOBIG);
		__leave;
		return -1;
	}

	/* THIS IS WEIRD. NOT LIKE OTHER COMMANDS THAT SIMPLY USE get_msw/get_lsw */
	msw = get_msw(blocksize);
	lsw = get_lsw(blocksize);

	while (retry) {

	  /* Only if something fails, retry */
	  retry = 0;

	  if ( usb_setup(njb, UT_READ_VENDOR_OTHER,
			 NJB_CMD_SEND_FILE_BLOCK, lsw, msw, 1, &status)
	       == -1 ) {
	    
	    NJB_ERROR(EO_USBCTL);
	    __leave;
	    return -1;
	  }

	  /* Check return status, retry if failed */
	  if ( status ) {
	    /* printf("Bad status byte in njb_send_file_block(): 0x%02x\n", status); */
	    retry = 1;
	    /* The NJB device is not following us. Sleep for a while
	     * and retry. If usleep is not available to sleep for 200ms,
	     * we'll just sleep for a second instead, just so it works
	     * either way. Tony Smolar noticed that sometimes, if the file
	     * exceeds 3 MB this need to wait for ready usually appears. 
	     * Generally, the nomad needs to wait 200-800 ms. */
#ifdef HAVE_USLEEP
	    usleep(200000);
#else
	    sleep(1);
#endif
	    retries --;
	    if ( !retries ) {
	      NJB_ERROR(EO_BADSTATUS);
	      __leave;
	      return -1;
	    }
	  }
	}

	bwritten = usb_pipe_write(njb, data, blocksize);

	if ( bwritten == -1 ) {
		NJB_ERROR(EO_USBBLK);
		__leave;
		return -1;
	}

	__leave;
	return bwritten;
}

int njb_stop_play (njb_t *njb)
{
	__dsub= "njb_stop_play";
	unsigned char data = '\0';

	__enter;

	if ( usb_setup(njb, UT_READ_VENDOR_OTHER,
		NJB_CMD_STOP_PLAY, 0, 0, 1, &data) == -1 ) {

		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	if ( data ) {
		NJB_STATUS(data);
		__leave;
		return -1;
	}

	__leave;
	return 0;
}

int njb_get_eax_size (njb_t *njb, u_int32_t *size)
{
	__dsub= "njb_get_eax_size";
	unsigned char data[5];

	__enter;

	memset(data, 0, 5);
	
	if ( usb_setup(njb, UT_READ_VENDOR_OTHER,
		NJB_CMD_GET_EAX_SIZE, 0, 0, 5, data) == -1 ) {

		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	if ( data[0] ) {
		NJB_ERROR(EO_BADSTATUS);
		__leave;
		return -1;
	}

	*size = njb1_bytes_to_32bit(&data[1]);

	__leave;
	return 0;
}

/* This is an old, deprecated routine */
eax_t *njb_get_eax (njb_t *njb, u_int32_t size)
{
	__dsub= "njb_get_eax";
	eax_t *eax= NULL;
	char *data;
	ssize_t bread;
	u_int32_t actsize;
	unsigned char usize[4];

	__enter;

	data= (char *) malloc(size);
	if ( data == NULL ) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return NULL;
	}
	memset(data, 0, size);
	memset(usize, 0, 4);
	
	from_32bit_to_njb1_bytes(size, &usize[0]);

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER, NJB_CMD_GET_EAX,
		0, 0, 4, usize) == -1 ) {

		NJB_ERROR(EO_USBCTL);
		free(data);
		__leave;
		return NULL;
	}

	bread= usb_pipe_read(njb, data, size+5);
	if ( bread == -1 ) {
		NJB_ERROR(EO_USBBLK);
		free(data);
		__leave;
		return NULL;
	} else if ( bread < size ) {
		NJB_ERROR(EO_RDSHORT);
		free(data);
		__leave;
		return NULL;
	}

	if ( data[0] ) {
		NJB_STATUS((int) data[0]);
		free(data);
		__leave;
		return NULL;
	}

	actsize = njb1_bytes_to_32bit(&data[1]);

	/* Unpack the EAX structure */

	eax= eax_unpack(&data[5], actsize);

	free(data);

	__leave;
	return eax;
}

/* This is the new EAX API */
void njb_read_eaxtypes (njb_t *njb, u_int32_t size)
{
  __dsub= "njb_read_eaxtypes";

  njb_state_t *state = (njb_state_t *) njb->protocol_state;
  unsigned char *data;
  ssize_t bread;
  u_int32_t actsize;
  unsigned char usize[4];

  __enter;

  data = (unsigned char *) malloc(size);
  if ( data == NULL ) {
    NJB_ERROR(EO_NOMEM);
    __leave;
    return;
  }
  memset(data, 0, size);
  memset(usize, 0, 4);
	
  from_32bit_to_njb1_bytes(size, &usize[0]);

  if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER, NJB_CMD_GET_EAX,
		 0, 0, 4, usize) == -1 ) {
    
    NJB_ERROR(EO_USBCTL);
    free(data);
    __leave;
    return;
  }

  bread = usb_pipe_read(njb, data, size+5);
  if ( bread == -1 ) {
    NJB_ERROR(EO_USBBLK);
    free(data);
    __leave;
    return;
  } else if ( bread < size ) {
    NJB_ERROR(EO_RDSHORT);
    free(data);
    __leave;
    return;
  }

  if ( data[0] ) {
    NJB_STATUS((int) data[0]);
    free(data);
    __leave;
    return;
  }

  actsize = njb1_bytes_to_32bit(&data[1]);

  /* Unpack the EAX structure */
  eax_unpack_new_api(&data[5], actsize, state);

  free(data);

  __leave;
  return;
}

njb_eax_t *njb_get_nexteax(njb_t *njb)
{
  njb_state_t *state = (njb_state_t *) njb->protocol_state;
  njb_eax_t *ret;

  ret = state->next_eax;
  if (ret != NULL) {
    state->next_eax = ret->next;
  }
  return ret;
}

/* This routine is dangerous. Do not call it... */
int njb_refresh_eax(njb_t *njb, eax_t *eax)
{
	__dsub= "njb_refresh_eax";
	unsigned char *data;
	u_int32_t actsize = 10;
	u_int32_t bread;

	__enter;

	data= malloc(4);
	if (data == NULL) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return -1;
	}
	memset(data, 0, 4);
	
	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
			NJB_CMD_GET_EAX, 1, 0, 4, data) == -1 ) {
		NJB_ERROR(EO_USBCTL);
		free(data);
		__leave;
		return -1;
	}

	/* In newer firmware, the four bytes returned in data
	 * contain rubbish, and can not be used for determining
	 * the size of the EAX parameters. You have to discard it. */
	free(data);

	/* 15 bytes + Status byte + 4 bytes size (again) */
	data= malloc(20);
	if (data == NULL) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return -1;
	}

	/* Read from bulk */
	if ( (bread= usb_pipe_read(njb, data, actsize+5))
		== -1 ) {

		NJB_ERROR(EO_USBBLK);
		free(data);
		__leave;
		return -1;
	} else if ( bread < actsize+5 ) {
		NJB_ERROR(EO_RDSHORT);
		free(data);
		__leave;
		return -1;
	}
	
	if ( data[0] ) {
		NJB_ERROR(EO_BADSTATUS);
		free(data);
		__leave;
		return -1;
	}

	actsize = njb1_bytes_to_32bit(&data[1]);

	eax_refresh(eax, data, actsize);
	free(data);

	__leave;
	return 0;
}

njb_time_t *njb_get_time(njb_t *njb)
{
	__dsub= "njb_get_time";
	njb_time_t *time = NULL;
	unsigned char *data;

	__enter;

	data= malloc(17);
	if (data == NULL) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return NULL;
	}
	memset(data, 0, 17);
	
       	if ( usb_setup(njb, UT_READ_VENDOR_OTHER,
       			NJB_CMD_GET_TIME, 0, 0, 17, data) == -1 ) {
       		NJB_ERROR(EO_USBCTL);
		free(data);
       		__leave;
       		return NULL;
       	}
       	if ( data[0] ) {
       		NJB_ERROR(EO_BADSTATUS);
		free(data);
       		__leave;
       		return NULL;
       	}
       	time= time_unpack(&data[1]);
	
	free(data);

	__leave;
	return time;
}

int njb_set_time(njb_t *njb, njb_time_t *time)
{
	__dsub= "njb_set_time";
	unsigned char *data;

       	data= time_pack(time);

       	if ( data == NULL ) {
       		__leave;
       		return -1;
       	}

       	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
       		NJB_CMD_SET_TIME, 0, 0, 16, data) == -1 ) {

       		NJB_ERROR(EO_USBCTL);
       	free(data);
       		__leave;
       		return -1;
       	}
       	free(data);

	__leave;
	return 0;
}

int njb_create_playlist (njb_t *njb, char *name, u_int32_t *plid)
{
	__dsub= "njb_create_playlist";
	ssize_t size, bread, bwritten;
	u_int16_t msw, lsw;
	unsigned char retdata[5];

	__enter;

	size= strlen(name) + 1;
	if ( size > 0xffffffff ) {
		NJB_ERROR(EO_TOOBIG);
		__leave;
		return 0;
	}

	memset(retdata, 0, 5);
	
	msw= get_msw( (u_int32_t) size );
	lsw= get_lsw( (u_int32_t) size );

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
		NJB_CMD_CREATE_PLAYLIST, lsw, msw, 0, NULL) == -1 ) {
		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	bwritten= usb_pipe_write(njb, name, size);
	if ( bwritten == -1 ) {
		NJB_ERROR(EO_USBBLK);
		__leave;
		return -1;
	} else if ( bwritten < size ) {
		NJB_ERROR(EO_WRSHORT);
		__leave;
		return -1;
	}

	bread= usb_pipe_read(njb, retdata, 5);
	if ( bread == -1 ) {
		NJB_ERROR(EO_USBBLK);
		__leave;
		return -1;
	} else if ( bread < 5 ) {
		NJB_ERROR(EO_RDSHORT);
		__leave;
		return -1;
	}

	if ( retdata[0] ) {
		NJB_STATUS(retdata[0]);
		__leave;
		return -1;
	}

	*plid = njb1_bytes_to_32bit(&retdata[1]);

	__leave;
	return 0;
}

int njb_delete_playlist (njb_t *njb, u_int32_t plid)
{
	__dsub= "njb_delete_playlist";
	u_int16_t msw, lsw;
	unsigned char data = '\0';

	__enter;

	msw= get_msw(plid);
	lsw= get_lsw(plid);

	if ( usb_setup(njb, UT_READ_VENDOR_OTHER,
		NJB_CMD_DELETE_PLAYLIST, msw, lsw, 1, &data) == -1 ) {

		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	if ( data ) {
		NJB_STATUS(data);
		__leave;
		return -1;
	}

	__leave;
	return 0;
}

int njb_rename_playlist (njb_t *njb, u_int32_t plid, char *name)
{
	__dsub= "njb_rename_playlist";
	size_t size;
	ssize_t bwritten;
	unsigned char data[8];

	__enter;

	memset(data, 0, 8);
	
	size= strlen(name) + 1;
	if ( size > 0xffffffff ) {
		NJB_ERROR(EO_TOOBIG);
		__leave;
		return 0;
	}

	from_32bit_to_njb1_bytes(plid, &data[0]);
	from_32bit_to_njb1_bytes(size, &data[4]);

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
		NJB_CMD_RENAME_PLAYLIST, 0, 0, 8, data) == -1 ) {

		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	bwritten= usb_pipe_write(njb, name, size);
	if ( bwritten == -1 ) {
		NJB_ERROR(EO_USBBLK);
		__leave;
		return -1;
	} else if ( bwritten < size ) {
		NJB_ERROR(EO_WRSHORT);
		__leave;
		return -1;
	}

	__leave;
	return 0;
}

int njb_add_track_to_playlist (njb_t *njb, u_int32_t plid, u_int32_t trid)
{
	__dsub= "njb_add_track_to_playlist";
	unsigned char data[10];

	__enter;

	memset(data, 0, 10);
	from_32bit_to_njb1_bytes(trid, &data[2]);
	from_32bit_to_njb1_bytes(plid, &data[6]);

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
		NJB_CMD_ADD_TRACK_TO_PLAYLIST, 0, 0, 10, data) == -1 ) {

		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	__leave;
	return 0;
}

int njb_add_multiple_tracks_to_playlist (njb_t *njb, u_int32_t plid, 
	u_int32_t *trids, u_int16_t ntracks)
{
	__dsub= "njb_add_multiple_tracks_to_playlist";
	unsigned char data[6];
	unsigned char *block, *bp;
	u_int16_t i;
	u_int32_t bsize;
	ssize_t bwritten;

	__enter;

	bsize= ntracks*6;
	block= (unsigned char *) malloc(bsize);
	if ( block == NULL ) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return -1;
	}
	memset(block, 0, bsize);
	memset(data, 0, 6);
	
	bp= block;
	for (i= 0; i< ntracks; i++) {
		memset(bp, 0, 2);
		bp+= 2;
		from_32bit_to_njb1_bytes(trids[i], &bp[0]);
		bp+= 4;
	}

	from_32bit_to_njb1_bytes(plid, &data[0]);
	from_16bit_to_njb1_bytes(ntracks, &data[4]);

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
		NJB_CMD_ADD_MULTIPLE_TRACKS_TO_PLAYLIST, 0, 0, 6, data) 
		== -1 ) {

		NJB_ERROR(EO_USBCTL);
		free(block);
		__leave;
		return -1;
	}

	bwritten= usb_pipe_write(njb, block, bsize);
	if ( bwritten == -1 ) {
		NJB_ERROR(EO_USBBLK);
		free(block);
		__leave;
		return -1;
	} else if ( bwritten < bsize ) {
		NJB_ERROR(EO_WRSHORT);
		free(block);
		__leave;
		return -1;
	}

	free(block);

	__leave;
	return 0;
}

int njb_delete_track (njb_t *njb, u_int32_t trackid)
{
	__dsub= "njb_delete_track";
	u_int16_t msw, lsw;
	unsigned char data = '\0';

	__enter;

	msw= get_msw(trackid);
	lsw= get_lsw(trackid);

	if ( usb_setup(njb, UT_READ_VENDOR_OTHER,
		NJB_CMD_DELETE_TRACK, msw, lsw, 1, &data) == -1 ) {
		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	if ( data ) {
		NJB_STATUS(data);
		__leave;
		return -1;
	}

	__leave;
	return 0;
}

int njb_delete_datafile (njb_t *njb, u_int32_t fileid)
{
	__dsub= "njb_delete_track";
	u_int16_t msw, lsw;
	unsigned char data = '\0';

	__enter;

	msw= get_msw(fileid);
	lsw= get_lsw(fileid);

	if ( usb_setup(njb, UT_READ_VENDOR_OTHER,
		NJB_CMD_DELETE_DATAFILE, msw, lsw, 1, &data) == -1 ) {
		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	if ( data ) {
		NJB_STATUS(data);
		__leave;
		return -1;
	}

	__leave;
	return 0;
}

int njb_play_or_queue (njb_t *njb, u_int32_t trackid, int cmd)
{
	__dsub= "njb_play_or_queue";
	unsigned char data[4];

	__enter;

	memset(data, 0, 4);
	
	from_32bit_to_njb1_bytes(trackid, &data[0]);

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER, cmd, 0, 0, 4, data)
		 == -1 ) {

		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	__leave;
	return 0;
}

int njb_elapsed_time (njb_t *njb, u_int16_t *elapsed, int *change)
{
	__dsub= "njb_get_elapsed_time";
	unsigned char data[3];

	__enter;

	memset(data, 0, 3);
	
	if ( usb_setup(njb, UT_READ_VENDOR_OTHER,
		NJB_CMD_ELAPSED_TIME, 0, 0, 3, data) == -1 ) {
		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	if ( data[0] == NJB_MSG_QUEUED_AUDIO_STARTED ) {
		*change= 1;
	} else if ( data[0] == 0 ) {
		*change= 0;
	} else {
		NJB_STATUS(data[0]);
		__leave;
		return -1;
	}

	*elapsed = njb1_bytes_to_16bit(&data[1]);

	__leave;
	return 0;
}

int njb_replace_track_tag (njb_t *njb, njbttaghdr_t *tagh, void *tag)
{
	__dsub= "njb_replace_track_tag";
	u_int16_t msw, lsw;
	unsigned char *data;
	ssize_t bwritten;

	__enter;

	msw= get_msw(tagh->size);
	lsw= get_lsw(tagh->size);

	data= (unsigned char *) malloc(tagh->size+4);
	if ( data == NULL ) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return -1;
	}
	memset(data, 0, tagh->size+4);

	from_32bit_to_njb1_bytes(tagh->trackid, &data[0]);
	memcpy(&data[4], tag, tagh->size);

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
		NJB_CMD_REPLACE_TRACK_TAG, lsw, msw, 0, NULL) == -1 ) {

		NJB_ERROR(EO_USBCTL);
		free(data);
		__leave;
		return -1;
	}

	bwritten= usb_pipe_write(njb, data, tagh->size+4);
	if ( bwritten == -1 ) {
		NJB_ERROR(EO_USBBLK);
		__leave;
		free(data);
		return -1;
	} else if ( bwritten < (tagh->size+4) ) {
		NJB_ERROR(EO_WRSHORT);
		free(data);
		__leave;
		return -1;
	}

	free(data);
	__leave;
	return 0;
}

int njb_adjust_sound(njb_t *njb, u_int8_t effect, int16_t value)
{
	__dsub= "njb_adjust_sound";
	unsigned char data[3];
	u_int16_t sendvalue = ((u_int16_t) value) & 0xff;

	__enter;

	memset(data, 0, 3);
	data[0] = effect;

	printf("Effect %d, sending value %d (%04X)\n", effect, sendvalue, sendvalue);

	from_16bit_to_njb1_bytes(sendvalue, &data[1]);

	if ( usb_setup(njb, UT_WRITE_VENDOR_OTHER,
		NJB_CMD_ADJUST_SOUND, 0, 0, 3, data) == -1 ) {
		NJB_ERROR(EO_USBCTL);
		__leave;
		return -1;
	}

	__leave;
	return 0;
}
