#ifndef __LIBNJB__H
#define __LIBNJB__H

#define LIBNJB_VERSION 1.2
#define LIBNJB_COMPILED_FOR_LIBUSB 1

#include <sys/types.h>
#ifdef __WIN32__
/* Windows specific code, types that do not exist in Windows
 * sys/types.h
 */
typedef char int8_t;
typedef unsigned char u_int8_t;
typedef short int16_t;
typedef unsigned short u_int16_t;
typedef int int32_t;
typedef unsigned int u_int32_t;
typedef unsigned __int64 u_int64_t;
#endif

#include <stdio.h>
#include <usb.h>

#define NJB_MAX_DEVICES 0xFF

/* Enumerators to identify different jukeboxes */
#define NJB_DEVICE_NJB1		0x00
#define NJB_DEVICE_NJB2		0x01
#define NJB_DEVICE_NJB3		0x02
#define NJB_DEVICE_NJBZEN	0x03
#define NJB_DEVICE_NJBZEN2	0x04
#define NJB_DEVICE_NJBZENNX	0x05
#define NJB_DEVICE_NJBZENXTRA	0x06
#define NJB_DEVICE_DELLDJ	0x07
#define NJB_DEVICE_NJBZENTOUCH	0x08

#define NJB_POWER_BATTERY       0x00
#define NJB_POWER_AC_CHARGED    0x01
#define NJB_POWER_AC_CHARGING   0x03

#define ID_DATA_ASCII	0
#define ID_DATA_BIN	1
#define ID_DATA_UNI     2

#define NJB_CODEC_MP3	"MP3"
#define NJB_CODEC_WMA	"WMA"
#define NJB_CODEC_WAV	"WAV"

#define FR_SIZE		"FILE SIZE"
#define FR_LENGTH	"LENGTH"
#define FR_CODEC	"CODEC"
#define FR_TITLE	"TITLE"
#define FR_ALBUM	"ALBUM"
#define FR_GENRE	"GENRE"
#define FR_ARTIST	"ARTIST"
#define FR_TRACK	"TRACK NUM"
#define FR_FNAME	"FNAME"
#define FR_YEAR		"YEAR"
#define FR_PROTECTED	"PlayOnly"
/* These two are used by Notmad on NJB1, not Creative */
#define FR_BITRATE	"BITRATE"
#define FR_COMMENT	"COMMENT"

/* Sound adjustment parameters - deprecated: use new EAX API */
/* These are to be moved into protocol.h and only used internally */
#define NJB_SOUND_SET_VOLUME	0x01
#define NJB_SOUND_SET_BASS	0x02
#define NJB_SOUND_SET_TREBLE	0x03
#define NJB_SOUND_SET_MUTING	0x04
#define NJB_SOUND_SET_MIDRANGE	0x05
#define NJB_SOUND_SET_MIDFREQ	0x06
#define NJB_SOUND_SET_EAX	0x07
#define NJB_SOUND_SET_EAXAMT	0x08
#define NJB_SOUND_SET_HEADPHONE	0x09
#define NJB_SOUND_SET_REAR	0x0A
#define NJB_SOUND_SET_EQSTATUS	0x0D

/* Error codes */

#define EO_USBCTL       1	/* I/O failure on USB control pipe	*/
#define EO_USBBLK       2	/* I/O failure on USB data pipe		*/
#define EO_RDSHORT      3	/* short read on USB data pipe		*/
#define EO_NOMEM        4	/* out of memory			*/
#define EO_BADDATA      5	/* invalid data				*/
#define EO_EOM          6	/* end of data				*/
#define EO_BADSTATUS    7	/* bad status from Jukebox		*/
#define EO_BADNJBID     8	/* Jukebox ID has changed		*/
#define EO_BADCOUNT     9	/* library count mismatch		*/
#define EO_WRSHORT      10	/* short write on USB data pipe		*/
#define EO_NULLTMP      11	/* temporary transfer dir not defined	*/
#define EO_TOOBIG       12	/* block size too big			*/
#define EO_CANTMOVE     13	/* can't move file to destination	*/
#define EO_TIMEOUT      14	/* operation timed out			*/
#define EO_ABORTED      15	/* operation aborted			*/
#define EO_EOF          16	/* received EOF				*/
#define EO_DEVICE       17	/* can't open device for read/write	*/
#define EO_INIT         18	/* can't initialize device		*/
#define EO_TMPFILE      19	/* can't create temporary file		*/
#define EO_XFERDENIED   20	/* transfer request denied		*/
#define EO_WRFILE       21	/* error writing output file		*/
#define EO_XFERERROR    22	/* bad transfer completion status	*/
#define EO_SRCFILE      23	/* can't read source file		*/
#define EO_INVALID	24	/* invalid arguments			*/
#define EO_AGAIN	25	/* resource temporarily unavailable	*/

/* Debug flags */

#define DD_USBCTL	1
#define DD_USBBLKLIM	2
#define DD_USBBLK	4
#define DD_SUBTRACE	8

/* Unicode flags */

#define NJB_UC_8859     0
#define NJB_UC_UTF8     1

/* NJB structure definitions */

typedef struct {
	unsigned char id[16]; /* Move all of this into njb_t eventually */
	u_int8_t fwMajor;
	u_int8_t fwMinor;
	u_int8_t fwRel;
	u_int8_t hwMajor;
	u_int8_t hwMinor;
	u_int8_t hwRel;
	char productName[33];
	u_int8_t power;
} njbid_t;

typedef struct {
	unsigned char id[16];
	u_int64_t count;
} njblibctr_t;

#define OWNER_STRING_LENGTH	128
typedef unsigned char owner_string[OWNER_STRING_LENGTH + 1];

typedef struct njb_t njb_t;
typedef struct songid_frame_t songid_frame_t;
typedef struct songid_t songid_t;
typedef struct playlist_track_struct playlist_track_t;
typedef struct playlist_struct playlist_t;
typedef struct datafile_struct datafile_t;
typedef struct eax_struct njb_eax_t;

struct njb_t {
	struct usb_device *device;
	usb_dev_handle *dev;
	int bulkep;
	int intrep;
	int device_type; /* what kind of jukebox it is */
	unsigned char id[16];
	char idstring[33];
	int updated;
	u_int32_t xfersize;
	void *protocol_state; /* dereferenced and maintained individually by protocol implementations */
	u_int8_t fwMajor; /* should eventually be moved here from njbid_t */
	u_int8_t fwMinor;
	u_int8_t fwRel;
};

/* Song/track tag definitions */

struct songid_frame_t {
	u_int16_t type;
	u_int16_t labelsz;
	u_int16_t datasz;
	void *label;
	void *data;
	songid_frame_t *next;
};

struct songid_t {
	u_int32_t trid;
	u_int16_t nframes;
	songid_frame_t *first;
	songid_frame_t *last;
	songid_frame_t *cur;
	songid_t *next; /* Used on NJB3 only */
};

/* Playlist definitions */

struct playlist_track_struct {
	u_int32_t trackid;
	struct playlist_track_struct *prev;
	struct playlist_track_struct *next;
};

struct playlist_struct {
	char *name;
	int _state;
#define NJB_PL_NEW		0
#define NJB_PL_UNCHANGED	1
#define NJB_PL_CHNAME		2
#define NJB_PL_CHTRACKS		3
	u_int32_t ntracks;
	u_int32_t plid;
	playlist_track_t *first;
	playlist_track_t *last;
	playlist_track_t *cur;
	playlist_t *nextpl; /* Used on NJB3 only */
};

/* Data file tag definitions */

struct datafile_struct {
	char *filename;
	u_int32_t timestamp;
	u_int32_t dfid;
	u_int32_t msdw;
	u_int32_t lsdw;
	datafile_t *nextdf;
};

/* EAX block definition - deprecated: use new EAX API */

typedef struct {
	u_int8_t volume;
	u_int8_t volumemin;
	u_int8_t volumemax;
	int16_t muted;
	int16_t eq_status;
	int8_t bass;
	int8_t bassmin;
	int8_t bassmax;
	int8_t midrange;
	int8_t midrangemin;
	int8_t midrangemax;
	int8_t treble;
	int8_t treblemin;
	int8_t treblemax;
	u_int8_t nfreq;
	u_int8_t freqset;
	int16_t *frequencies;
	u_int8_t neffects;
	u_int8_t acteffect;
	char **effects;
	u_int8_t *effamts;
	u_int8_t nphone;
	u_int8_t actphone;
	char **phonemodes;
	u_int8_t nrear;
	u_int8_t actrear;
	char **rears;
} eax_t;

/* EAX API type */
struct eax_struct {
	u_int16_t number;
	char *name;
	/* 
         * The "exclusive" attribute signifies if this effect can be used 
         * in parallell with other EAX effects,
	 * GUI:s shall make sure all other effects are reset to default
         * values when one effect is chosen.
         */
	u_int8_t exclusive; /* 0x00 = not exclusive, 0x01 = exclusive */
	/*
         * The visual group for this effect - 
	 * effects appearing after each other are grouped if their 
	 * group numbers are identical
	 */
	u_int8_t group;
	/* If this effect has several different settings */
	u_int8_t selectable; /* 0x00 = not selectable, 0x01 = selectable */
	u_int16_t current_selectionvalue;
	u_int16_t min_selectionvalue;
	u_int16_t max_selectionvalue;
	char **selection_names; /* Array with names for these selections */
	/* If this effect can be slided on a scale */
	u_int8_t scaleable; /* 0x00 = not scalable, 0x01 = scalable */
	int16_t current_scalevalue;
	int16_t min_scalevalue;
	int16_t max_scalevalue;
	njb_eax_t *next; /* Only to be used inside of libnjb */
};

/* Time set/read struct */
typedef struct {
	int16_t year;
	int16_t month;
	int16_t day;
	int16_t weekday;
	int16_t hours;
	int16_t minutes;
	int16_t seconds;
} njb_time_t;

/* Struct to hold key/value pairs */
typedef struct njb_keyval_t njb_keyval_t;
struct njb_keyval_t {
  char key[5];
  u_int32_t value1;
  u_int32_t value2;
  unsigned char deviceid[16];
  njb_keyval_t *next;
};


#ifdef __cplusplus
extern "C" {
#endif

/* NJB commands */

typedef int XferCallback(u_int64_t sent, u_int64_t total,
		const char* buf, unsigned len, void *data);

int NJB_Discover(njb_t *njbs, int limit, int *n);
int NJB_Open(njb_t *njb);
void NJB_Close(njb_t *njb);
int NJB_Capture (njb_t *njb);
int NJB_Release (njb_t *njb);
njbid_t *NJB_Ping(njb_t *njb);
void NJB_Get_Extended_Tags (njb_t *njb, int extended);
void NJB_Reset_Get_Track_Tag (njb_t *njb);
songid_t *NJB_Get_Track_Tag (njb_t *njb);
void NJB_Reset_Get_Playlist (njb_t *njb);
playlist_t *NJB_Get_Playlist (njb_t *njb);
int NJB_Get_Disk_Usage (njb_t *njb, u_int64_t *btotal, u_int64_t *bfree);
char *NJB_Get_Owner_String (njb_t *njb);
int NJB_Set_Owner_String (njb_t *njb, const char *name);
void NJB_Reset_Get_Datafile_Tag (njb_t *njb);
datafile_t *NJB_Get_Datafile_Tag (njb_t *njb);
eax_t *NJB_Get_EAX (njb_t *njb); /* Deprecated, use new EAX API */
int NJB_Refresh_EAX (njb_t *njb, eax_t *eax); /* Deprecated, use new EAX API */
void NJB_Reset_Get_EAX_Type (njb_t *njb);
njb_eax_t *NJB_Get_EAX_Type (njb_t *njb);
void NJB_Destroy_EAX_Type (njb_eax_t *eax);
void NJB_Adjust_EAX (njb_t *njb,
	u_int16_t eaxid,
	u_int16_t patchindex,
	int16_t scalevalue);
njb_time_t *NJB_Get_Time(njb_t *njb);
int NJB_Set_Time(njb_t *njb, njb_time_t *time);
void NJB_Destroy_Time(njb_time_t *time);
int NJB_Set_Bitmap(njb_t *njb, const char * pbm_file);
int NJB_Delete_Playlist (njb_t *njb, u_int32_t plid);
int NJB_Update_Playlist (njb_t *njb, playlist_t *pl);
int NJB_Delete_Track (njb_t *njb, u_int32_t trackid);
int NJB_Delete_Datafile (njb_t *njb, u_int32_t fileid);
#define NJB_Get_File NJB_Get_Track
int NJB_Get_Track (njb_t *njb, u_int32_t trackid, u_int32_t size,
	const char *path, XferCallback *callback, void *data);
int NJB_Send_Track (njb_t *njb, const char *path, const char *codec,
        const char *title, const char *album, const char *genre,
	const char *artist, u_int32_t length, u_int32_t tracknum,
	const char *year, int protect,
	XferCallback *callback, void *data, u_int32_t *trackid);
int NJB_Send_File (njb_t *njb, const char *path, const char *name,
	XferCallback *callback, void *data, u_int32_t *fileid);
int NJB_Replace_Track_Tag(njb_t *njb, u_int32_t trackid, const char *codec,
	const char *title, const char *album, const char *genre,
	const char *artist, u_int32_t length, u_int32_t tracknum,
	u_int32_t filesize, const char *fname, const char *year, int protect);
int NJB_Play_Track (njb_t *njb, u_int32_t trackid);
int NJB_Queue_Track (njb_t *njb, u_int32_t trackid);
int NJB_Stop_Play (njb_t *njb);
int NJB_Pause_Play (njb_t *njb);
int NJB_Resume_Play (njb_t *njb);
int NJB_Seek_Track (njb_t *njb, u_int32_t position);
int NJB_Elapsed_Time (njb_t *njb, u_int16_t *elapsed, int *change);
int NJB_Adjust_Sound(njb_t *njb, u_int8_t effect, u_int16_t value);
void NJB_Set_Debug (int debug_flags);
void NJB_Set_Unicode (int unicode_flag);
njb_keyval_t *NJB_Get_Keys(njb_t *njb);

/* Error functions */

void njb_error_reset_geterror (void);
const char *njb_error_geterror (void);
const char *njb_error_string (int err);
void njb_error_dump (FILE *fp);

/* Song/track tag functions */

songid_t *songid_new (void);
songid_t *songid_unpack (void *data, size_t nbytes);
unsigned char *songid_pack (songid_t *song, u_int32_t *size);
unsigned char *songid_pack3 (songid_t *song, u_int32_t *size);
void songid_destroy (songid_t *song);
void songid_addframe (songid_t *song, songid_frame_t *frame);
void songid_dump (songid_t *song, FILE *fp);
void songid_reset_getframe (songid_t *song);
songid_frame_t *songid_getframe (songid_t *song);
songid_frame_t *songid_findframe (songid_t *song, const char *label);

songid_frame_t *songid_frame_new (u_int16_t labelsz, const void *label,
	u_int16_t type, u_int16_t datasz, const void *data);
void songid_frame_destroy (songid_frame_t *frame);
void songid_frame_dump (songid_frame_t *frame, FILE *fp);
u_int32_t songid_frame_data32 (songid_frame_t *frame);

#define songid_frame_new_codec(a) songid_frame_new(6,FR_CODEC,ID_DATA_ASCII,strlen(a)+1,a)
#define songid_frame_new_title(a) songid_frame_new(6,FR_TITLE,ID_DATA_ASCII,strlen(a)+1,a)
#define songid_frame_new_album(a) songid_frame_new(6,FR_ALBUM,ID_DATA_ASCII,strlen(a)+1,a)
#define songid_frame_new_genre(a) songid_frame_new(6,FR_GENRE,ID_DATA_ASCII,strlen(a)+1,a)
#define songid_frame_new_artist(a) songid_frame_new(7,FR_ARTIST,ID_DATA_ASCII,strlen(a)+1,a)
#define songid_frame_new_length(a) songid_frame_new(7,FR_LENGTH,ID_DATA_BIN,sizeof(u_int32_t),a)
#define songid_frame_new_filesize(a) songid_frame_new(10,FR_SIZE,ID_DATA_BIN,sizeof(u_int32_t),a)
#define songid_frame_new_tracknum(a) songid_frame_new(10,FR_TRACK,ID_DATA_BIN,sizeof(u_int32_t),a)
#define songid_frame_new_fname(a) songid_frame_new(10,FR_FNAME,ID_DATA_ASCII,strlen(a)+1,a)
#define songid_frame_new_year(a) songid_frame_new(10,FR_YEAR,ID_DATA_ASCII,strlen(a)+1,a)
#define songid_frame_new_protected(a) songid_frame_new(10,FR_PROTECTED,ID_DATA_ASCII,strlen(a)+1,a)

u_int32_t songid_size (songid_t *song);
u_int32_t songid_length (songid_t *song);
u_int32_t songid_track (songid_t *song);
const char *songid_label (songid_t *song, const char *label);
#define songid_title(a)  songid_label(a,FR_TITLE);
#define songid_album(a)  songid_label(a,FR_ALBUM);
#define songid_genre(a)  songid_label(a,FR_GENRE);
#define songid_codec(a)  songid_label(a,FR_CODEC);
#define songid_fname(a)  songid_label(a,FR_FNAME);
#define songid_artist(a) songid_label(a,FR_ARTIST);
#define songid_year(a) songid_label(a,FR_YEAR);
#define songid_protected(a) songid_label(a,FR_PROTECTED);

/* Playlist fuctions */

playlist_t *playlist_new (void);
void playlist_destroy (playlist_t *pl);
void playlist_addtrack (playlist_t *pl, playlist_track_t *track, 
	unsigned int pos);
#define NJB_PL_END	0
#define NJB_PL_START	1
void playlist_reset_gettrack (playlist_t *pl);
playlist_track_t *playlist_gettrack (playlist_t *pl);
int playlist_set_name (playlist_t *pl, const char *name);
void playlist_deltrack (playlist_t *pl, unsigned int pos);
void playlist_dump(playlist_t *pl, FILE *fp);
playlist_t *playlist_unpack (void *data, size_t nbytes);
u_int32_t playlist_pack (playlist_t *pl, char *data);

playlist_track_t *playlist_track_new (u_int32_t trackid);
void playlist_track_destroy (playlist_track_t *track);

/* Data file tag functions */

datafile_t *datafile_new (void);
void datafile_destroy (datafile_t *df);
void datafile_dump(datafile_t *df, FILE *fp);
datafile_t *datafile_unpack (unsigned char *data, size_t nbytes);
unsigned char *datafile_pack (datafile_t *df, u_int32_t *size);
u_int64_t datafile_size (datafile_t *df);
void datafile_set_size (datafile_t *df, u_int64_t size);
void datafile_set_time (datafile_t *df, time_t ts);
int datafile_set_name (datafile_t *df, const char *filename);
unsigned char *datafile_pack3 (njb_t *njb, datafile_t *df, u_int32_t *size);

/* EAX functions */
void eax_destroy(eax_t *eax);

/* Time functions */

njb_time_t *time_unpack(void *data);
njb_time_t *time_unpack3(void *data);
void *time_pack(njb_time_t *time);
void *time_pack3(njb_time_t *time);

#ifdef __cplusplus
}
#endif

#endif
