#include <sys/stat.h> /* stat() */
#include <ctype.h> /* isspace() */
#include <fcntl.h>
#ifdef HAVE_LIBGEN_H
#include <libgen.h> /* basename() */
#endif
#include <string.h> /* Various memory and string functions */
#include "libnjb.h"
#include "procedure.h"
#include "protocol.h"
#include "protocol3.h"
#include "base.h"
#include "njbusb.h"
#include "njb_error.h"
#include "defs.h"
#include "ioutil.h"
#include "unicode.h"
#include "byteorder.h"
#include "eax.h"

static int _lib_ctr_update (njb_t *njb);
void _skip_whitespaces (FILE *);
int _verify_pbm (FILE *);
int _file_size (const char *path, u_int64_t *size);
int _file_time (const char *path, time_t *ts);
int NJB_Handshake (njb_t *njb);

extern int njb_error;
extern int njb_unicode_flag;
extern int __sub_depth;

/* Function that compensate for missing libgen.h on Windows */
#ifndef HAVE_LIBGEN_H
static char *basename(char *in) {
  char *p;
  if (in == NULL)
    return NULL;
  p = in + strlen(in) - 1;
  while (*p != '\\' && *p != '/' && *p != ':')
    { p--; }
  return ++p;
}
#endif

static int _lib_ctr_update (njb_t *njb)
{
	__dsub= "_lib_ctr_update";
	njblibctr_t lcount;
	njb_state_t *state = (njb_state_t *) njb->protocol_state;

	__enter;

	if ( state->session_updated ) return 0;

	if ( njb_get_library_counter(njb, &lcount) == -1 ) {
		__leave;
		return -1;
	}

	if ( memcmp(njb->id, lcount.id, 16) ) {
		NJB_ERROR(EO_BADNJBID);
		__leave;
		return -1;
	}

	lcount.count++;

	if ( njb_set_library_counter(njb, lcount.count) == -1 ) {
		__leave;
		return -1;
	}

	if ( njb_verify_last_command(njb) == -1 ) {
		__leave;
		return -1;
	}

	state->session_updated = 1;
	state->libcount ++;
	__leave;
	return 0;
}

int NJB_Discover (njb_t *njbs, int limit, int *n)
{
	__dsub= "NJB_Discover";
	int ret;

	__enter;

	njb_error_clear();
	ret= njb_discover(njbs, limit, n);

	__leave;
	return ret;
}

int NJB_Open (njb_t *njb)
{
	__dsub= "NJB_Open";
	int ret = 0;

	__enter;

	njb_error_clear();
	
	if ( (ret = njb_open(njb)) != -1 )
	{
	  /* This was invented for the Zen but is now activated
	   * for NJB3. It is apparently not applicable for USB 2.0
	   * devices (Zen USB 2.0, NJB 2, Zen NX).
	   * Also see NJB_Close() (similar code). LW.
	   */

	  if (njb->device_type == NJB_DEVICE_NJB1) {
	    if ( njb_init_state(njb) == -1) {
	      __leave;
	      return -1;
	    }
	  }

	  if (PROTOCOL3_DEVICE(njb->device_type)) {
	    if ( njb3_init_state(njb) == -1) {
	      __leave;
	      return -1;
	    }
	  }

	  if (njb->device_type == NJB_DEVICE_NJBZEN
	      || njb->device_type == NJB_DEVICE_NJB3) {
	    njb3_capture(njb);
	  }

	  ret = NJB_Handshake(njb);

	}

	__leave;
	return ret;
}

void NJB_Close (njb_t *njb)
{
	__dsub= "NJB_Close";

	__enter;

	/* Release 3-series devices */
	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  /*
	   * This special was written by Dwight Engen due to problems with
	   * the ZEN refusing to reconnect after running sample programs.
	   * the same thing used for NJB3. USB 2.0 devices should not use this.
	   * Also see NJB_Open() (similar code).
	   */
	  if (njb->device_type == NJB_DEVICE_NJBZEN
	      || njb->device_type == NJB_DEVICE_NJB3
	      ) {
	    njb3_ping(njb, NULL, 1);
	  }
	  njb3_release(njb);

	  /* Free dangling pointers in njb_t struct */
	  njb3_destroy_state(njb);

	}
	
	njb_close(njb);

	__leave;
}


int NJB_Capture (njb_t *njb)
{
	__dsub= "NJB_Capture";
	__enter;


	/* This is simply ignored on NJB3, where it isn't applicable.
	 * The ZEN is captured at NJB_Open and released at NJB_Close.
	 */
	if (njb->device_type == NJB_DEVICE_NJB1) {
		njblibctr_t lcount;
		njb_state_t *state = (njb_state_t *) njb->protocol_state;

		njb_error_clear();

		if ( njb_capture(njb, NJB_CAPTURE) == -1 ) {
			__leave;
			return -1;
		}

		if ( njb_get_library_counter(njb, &lcount) == -1 ) {
			__leave;
			return -1;
		}

		if ( state->libcount != lcount.count ) {
			njb_capture(njb, NJB_RELEASE);
			NJB_ERROR(EO_BADCOUNT);
			__leave;
			return -1;
		}
	}

	__leave;
	return 0;
}

int NJB_Release (njb_t *njb)
{
	__dsub= "NJB_Release";
	int ret;

	__enter;

	/* This is simply ignored on NJB3, where it isn't applicable */
	if (njb->device_type == NJB_DEVICE_NJB1) {
		njb_error_clear();
		ret= njb_capture(njb, NJB_RELEASE);
	} else {
		ret= 0;
	}

	__leave;
	return ret;
}

int NJB_Handshake (njb_t *njb)
{
	__dsub= "NJB_Handshake";
	njbid_t njbid;
	char *cp;
	int i;

	__enter;

	njb_error_clear();

	if ( njb->device_type == NJB_DEVICE_NJB1 ) {
		if ( njb_ping(njb, &njbid) == -1 ) {
			__leave;
			return -1;
		}
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {

	  if ( njb3_ping(njb, &njbid, 0) == -1 ) {
	    __leave;
	    return -1;
	  }
	  /* Retrieve jukebox ID */
	  if ( njb3_readid(njb, &njbid) == -1 ) {
	    __leave;
	    return -1;
	  }
	  /* Retrieve the list of supported codecs, just for fun. */
	  if ( njb3_get_codecs(njb) == -1 ) {
	    __leave;
	    return -1;
	  }
	  /*
	   * Retrieve keys - this is not used for anything yet,
	   * part of the "AR00" key should be used in the storage
	   * of an MP3 file. 
	   */
	  if ( njb3_read_keys(njb) == -1 ) {
	    __leave;
	    return -1;
	  }
	}

	memcpy(njb->id, njbid.id, 16);
	cp= njb->idstring;
	for (i= 0; i<16; i++) {
		sprintf(cp, "%02X", njb->id[i]);
		cp+=2;
	}
	njb->idstring[32]= 0;

	/* No library counters on NJB3 */
	if (njb->device_type == NJB_DEVICE_NJB1) {
	        njblibctr_t lcount, lcount_new, lcount_old;
	        njb_state_t *state = (njb_state_t *) njb->protocol_state;

		if ( njb_get_library_counter(njb, &lcount) == -1 ) {
			__leave;
			return -1;
		}

		if ( memcmp(njb->id, lcount.id, 16) ) {
			NJB_ERROR(EO_BADNJBID);
			__leave;
			return -1;
		}

		lcount_new= lcount;
		lcount_old= lcount;

		lcount_new.count++;

		if ( njb_set_library_counter(njb, lcount_new.count) == -1 ) {
			__leave;
			return -1;
		}

		if ( njb_verify_last_command(njb) == -1 ) {
			__leave;
			return -1;
		}

		if ( njb_get_library_counter(njb, &lcount) == -1 ) {
			__leave;
			return -1;
		}

		if ( memcmp(njb->id, lcount.id, 16) ) {
			NJB_ERROR(EO_BADNJBID);
			__leave;
			return -1;
		}

		if ( lcount.count != lcount_new.count ) {
			NJB_ERROR(EO_BADCOUNT);
			__leave;
			return -1;
		}

		if ( njb_set_library_counter(njb, lcount_old.count) == -1 ) {
			__leave;
			return -1;
		}

		if ( njb_verify_last_command(njb) == -1 ) {
			__leave;
			return -1;
		}

		state->libcount = lcount_old.count;
	}

	__leave;
	return 0;
}

void NJB_Get_Extended_Tags (njb_t *njb, int extended)
{
  __dsub= "NJB_Get_Extended_Tags";
  njb3_state_t *state = (njb3_state_t *) njb->protocol_state;
  __enter;

  state->get_extended_tag_info = extended;

  __leave;
}

void NJB_Reset_Get_Track_Tag (njb_t *njb)
{
	__dsub= "NJB_Reset_Get_Track_Tag";

	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1) {
	  njb_state_t *state = (njb_state_t *) njb->protocol_state;

	  state->reset_get_track_tag = 1;
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  if (njb3_reset_get_track_tag(njb) == -1) {
	    /* FIXME: do something with this error some day. */
	  }
	}

	__leave;
}

songid_t *NJB_Get_Track_Tag (njb_t *njb)
{
	__dsub= "NJB_Get_Track_Tag";
	int status;
	songid_t *ret = NULL;

	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1) {
		njbttaghdr_t tagh;
		njb_state_t *state = (njb_state_t *) njb->protocol_state;

		if ( state->reset_get_track_tag ) {
			status= njb_get_first_track_tag_header(njb, &tagh);
			state->reset_get_track_tag = 0;
		} else {
			status= njb_get_next_track_tag_header(njb, &tagh);
		}
		ret= ( status == -1 ) ? NULL : njb_get_track_tag(njb, &tagh);
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  ret = njb3_get_next_track_tag(njb);
	}

	__leave;
	return ret;
}

void NJB_Reset_Get_Playlist (njb_t *njb)
{
	__dsub= "NJB_Reset_Get_Playlist";

	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1) {
	  njb_state_t *state = (njb_state_t *) njb->protocol_state;

	  state->reset_get_playlist= 1;
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  if (njb3_reset_get_playlist_tag(njb) == -1) {
	    /* FIXME: do something about this error eventually */
	  }
	}

	__leave;
}

playlist_t *NJB_Get_Playlist (njb_t *njb)
{
	__dsub= "NJB_Get_Playlist";
	njbplhdr_t plh;
	int retry= 3;
	playlist_t *ret = NULL;

	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1) {
	        njb_state_t *state = (njb_state_t *) njb->protocol_state;

		if ( state->reset_get_playlist ) {
			while (retry) {
				if ( njb_get_first_playlist_header(njb, &plh) == -1 ) {
					if ( njb_error == EO_AGAIN && retry ) retry--;
					else {
						__leave;
						return NULL;
					}
				} else break;
			}
			state->reset_get_playlist= 0;
		} else {
			while (retry) {
				if ( njb_get_next_playlist_header(njb, &plh) == -1 ) {
					if ( njb_error == EO_AGAIN && retry ) retry--;
					else {
						__leave;
						return NULL;
					}
				} else break;
			}
		}

		ret = njb_get_playlist(njb, &plh);
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  ret = njb3_get_next_playlist_tag(njb);
	}

	__leave;
	return ret;
}

int NJB_Get_Disk_Usage (njb_t *njb, u_int64_t *btotal, u_int64_t *bfree)
{
	__dsub= "NJB_Get_Disk_Usage";
	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1) {
		int retry= 3;
		while (retry) {
			if ( njb_get_disk_usage(njb, btotal, bfree) == -1 ) {
				if ( njb_error == EO_AGAIN ) retry--;
				else {
					__leave;
					return -1;
				}
			} else break;
		}
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  if ( njb3_get_disk_usage(njb, btotal, bfree) == -1 ) {
	    __leave;
	    return -1;
	  }
	}

	__leave;
	return 0;
}

char *NJB_Get_Owner_String (njb_t *njb)
{
	__dsub= "NJB_Get_Owner_String";
	owner_string name;
	char *op = NULL;

	njb_error_clear();

	__enter;

	if ( njb->device_type == NJB_DEVICE_NJB1 ) {
		if ( njb_get_owner_string(njb, name) == -1 ) {
			__leave;
			return NULL;
		}
		if (njb_unicode_flag == NJB_UC_UTF8) {
			op = strtoutf8((char *) name);
		} else {
			op = strdup((char *) name);
		}
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  if ( njb3_get_owner_string(njb, (char *)name) == -1 ) {
	    __leave;
	    return NULL;
	  }
	  op = strdup((char *) name);
	}

	if ( op == NULL ) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return NULL;
	}

	__leave;
	return op;
}

int NJB_Set_Owner_String (njb_t *njb, const char *name)
{
	__dsub= "NJB_Set_Owner_String";
	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1) {
		owner_string oname;

		memset(oname, 0, OWNER_STRING_LENGTH);
		if (njb_unicode_flag == NJB_UC_UTF8) {
			char *tmp;

			tmp = utf8tostr(name);
			strncpy((char *) oname, tmp, OWNER_STRING_LENGTH);
			free(tmp);
		} else {
			strncpy((char *) oname, name, OWNER_STRING_LENGTH);
		}

		if ( njb_set_owner_string(njb, oname) == -1 ) {
			__leave;
			return -1;
		}

		if ( njb_verify_last_command(njb) == -1 ) {
			__leave;
			return -1;
		}
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  if ( njb3_set_owner_string(njb, name) == -1 ) {
	    __leave;
	    return -1;
	  }
	}

	__leave;
	return 0;
}

void NJB_Reset_Get_Datafile_Tag (njb_t *njb)
{
	__dsub= "NJB_Get_Datafile_Tag";

	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1 ) {
	  njb_state_t *state = (njb_state_t *) njb->protocol_state;

	  state->reset_get_datafile_tag = 1;
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  if (njb3_reset_get_datafile_tag(njb) == -1) {
	    /* FIXME: handle this error eventually */
	  }
	}

	__leave;
}

datafile_t *NJB_Get_Datafile_Tag (njb_t *njb)
{
	__dsub= "NJB_Get_Datafile_Tag";
	njbdfhdr_t dfh;
	int status;
	datafile_t *ret = NULL;

	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1 ) {
	        njb_state_t *state = (njb_state_t *) njb->protocol_state;

		if ( state->reset_get_datafile_tag ) {
			status = njb_get_first_datafile_header(njb, &dfh);
			state->reset_get_datafile_tag= 0;
		} else {
			status = njb_get_next_datafile_header(njb, &dfh);
		}

		ret= ( status == -1 ) ? NULL : njb_get_datafile_tag(njb, &dfh);
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  ret = njb3_get_next_datafile_tag(njb);
	}

	__leave;
	return ret;
}

int NJB_Get_Track (njb_t *njb, u_int32_t fileid, u_int32_t size,
	const char *path, XferCallback *callback, void *data)
{
	__dsub= "NJB_Get_Track";
	unsigned char *block = NULL;
	int fd = 0;
	u_int32_t bsize, offset;
	u_int32_t remain = size;
	int abortxfer = 0;
	int ret;

	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1) {

		if ( njb_request_file(njb, fileid) == -1 ) {
			ret = -1;
			goto clean_up_and_return;
		}

		if ( njb_verify_last_command(njb) == -1 ) {
			if ( njb_error == EO_BADSTATUS ) NJB_ERROR(EO_XFERDENIED);
			ret = -1;
			goto clean_up_and_return;
		}
	}

#ifdef __WIN32__
	if ( path && (fd= open(path, O_CREAT|O_TRUNC|O_WRONLY|O_BINARY, 0664)) == -1 ) {
#else
	if ( path && (fd= open(path, O_CREAT|O_TRUNC|O_WRONLY, 0664)) == -1 ) {
#endif
		NJB_RERROR("open", -1);
		NJB_ERROR(EO_TMPFILE);
		ret = -1;
		goto clean_up_and_return;
	}

	/* File transfer routine for the NJB1 */
	if (njb->device_type == NJB_DEVICE_NJB1) {
	        u_int32_t bread = 0;
		int wasshort = 0;

		block = (unsigned char *) malloc(NJB_XFER_BLOCK_SIZE + NJB_XFER_BLOCK_HEADER_SIZE);
		if ( block == NULL ) {
			NJB_ERROR(EO_NOMEM);
			ret = -1;
			goto clean_up_and_return;
		}
		memset(block, 0, NJB_XFER_BLOCK_SIZE + NJB_XFER_BLOCK_HEADER_SIZE);
		offset = 0;
		while (remain != 0 && !abortxfer) {
		        /* Request as much as possible unless the last chunk is reached */
			bsize= (remain > NJB_XFER_BLOCK_SIZE) ? NJB_XFER_BLOCK_SIZE : remain;

			bread = 0;

			bread = njb_receive_file_block(njb, offset, bsize, &block[0]);
			if (bread < bsize + NJB_XFER_BLOCK_HEADER_SIZE) {
			  wasshort = 1;
			} else {
			  wasshort = 0;
			}
			
			/* Handle errors */
			if (bread == -1) {
			  ret = -1;
			  goto clean_up_and_return;
			}
			
			/* Good debug messages */
			/*
			fprintf(stdout, "File get: recieved %04x bytes", bread);
			if (wasshort)
			  fprintf(stdout, " - short read");
			fprintf(stdout,"\n");
			*/
		
			/* Write receieved bytes to file */
			if (path && bread > NJB_XFER_BLOCK_HEADER_SIZE) {
			  if ( (write(fd, &block[NJB_XFER_BLOCK_HEADER_SIZE], bread-NJB_XFER_BLOCK_HEADER_SIZE)) == -1 ) {
			    NJB_RERROR("write", -1);
			    NJB_ERROR(EO_WRFILE);
			    __leave;
			    ret = -1;
			    goto clean_up_and_return;
			  }
			}

			remain -= (bread-NJB_XFER_BLOCK_HEADER_SIZE);
			offset += (bread-NJB_XFER_BLOCK_HEADER_SIZE);

			if ( callback != NULL ) {
			  const char *last_block= (const char *) &block[NJB_XFER_BLOCK_HEADER_SIZE];
			  if ( callback(offset, size, last_block, bread-NJB_XFER_BLOCK_HEADER_SIZE, data) == -1 ) {
			    abortxfer = 1;
			  }
			}
		}

		/* This is probably not the right way to abort a file transfer */

		if ( abortxfer ) {
			njb_transfer_complete(njb);
			NJB_ERROR(EO_ABORTED);
			ret = -1;
			goto clean_up_and_return;
		} 

		if ( njb_transfer_complete(njb) == -1 ) {
			NJB_ERROR(EO_XFERERROR);
			ret = -1;
			goto clean_up_and_return;
		}
	}

	/* Protocol 3 series file transfer routine */
	if (PROTOCOL3_DEVICE(njb->device_type)) {
		int confirm_size;

		/* FIXME: add this and test. */
		/* njb3_ctrl_playing(njb, NJB3_STOP_PLAY); */

		block = (unsigned char *) malloc(NJB3_CHUNK_SIZE);
		if (block == NULL) {
		  NJB_ERROR(EO_NOMEM);
		  ret = -1;
		  goto clean_up_and_return;
		}

		/* Start at beginning of file */
		offset = 0;
		remain = size;

		while (remain != 0 && abortxfer == 0) {
		  u_int32_t bwritten;
		  int chunk_size;
		  int chunk_remain;
		  int bread;

		  /*
		   * Request chunks of size NJB3_CHUNK_SIZE at a time, 
		   * indexed at an offset into the file. 
		   */
		  chunk_size = njb3_request_file_chunk(njb, fileid, offset);
		  if ( chunk_size == -1 ) {
		    ret = -1;
		    goto clean_up_and_return;
		  }

		  chunk_remain = chunk_size;

		  /* Do NOT break on abort here - all blocks must be sent before 
		   * cancelling the transfer! */
		  while (chunk_remain != 0) {

		    /*
		     * Try to get maximum amount, if we run into a short 
		     * read, that's OK 
		     */
		    /* printf("Requesting chunk size %08X, remaining %08X...\n", NJB3_SUBCHUNK_SIZE, chunk_remain); */
		    bread = njb3_get_file_block(njb, block, NJB3_GET_FILE_BLOCK_SIZE);
		    /* Negative value signals error */
		    if ( bread == -1 ) {
		      ret = -1;
		      goto clean_up_and_return;
		    }

		    /*
		     * Sometimes the chunk will be one
		     * byte larger than the reported chunk size.
		     * The byte is not part of the actual file.
		     *
		     * Adjust this downwards as of now...
		     */
		    if (bread > chunk_remain) {
		      int extraneous = bread - chunk_remain;
		      if (extraneous == 1) {
			/* printf("LIBNJB: one extraneous byte.\n"); */
		      }
		      if (extraneous > 1) {
			/* This should, however, not happen. */
			printf("LIBNJB panic: recieved %d extraneous bytes!\n", extraneous);
		      }
		      bread = chunk_remain;
		    }
		  
		    /* Write the bytes to the destination file */
		    if ( (bwritten = write(fd, block, bread) ) == -1 ) {
		      NJB_RERROR("write", -1);
		      NJB_ERROR(EO_WRFILE);
		      ret = -1;
		      goto clean_up_and_return;
		    } else if ( bwritten != bread ) {
		      NJB_ERROR(EO_WRFILE);
		      ret = -1;
		      goto clean_up_and_return;
		    }

		    chunk_remain -= bread;
		    offset += bread;

		    if ( callback != NULL) {
		      if ( callback(offset, size, NULL, 0, data) == -1 )
			abortxfer= 1;
		    }
		  }

		  /*
		   * We have recieved a whole chunk or are at the 
		   * end of the file.
		   */
		  if (chunk_size > remain) {
		    int extraneous = chunk_size - remain;
		    if (extraneous == 1) {
		      /* printf("LIBNJB: one extraneous byte in chunk.\n"); */
		    }
		    if (extraneous > 1) {
		      /* This should, however, not happen. */
		      printf("LIBNJB panic: recieved %d extraneous bytes!\n", extraneous);
		    }
		    chunk_size = remain;
		  }
		  remain -= chunk_size;
		  if (remain < 0) {
		    printf("LIBNJB panic: Remain < 0!\n");
		    remain = 0;
		  }
		}

		/*
		 * File confirmation is accomplished by requesting 
		 * file offset set at the file size (e.g. one past 
		 * the end)
		 */
		confirm_size = njb3_request_file_chunk(njb, fileid, size);
		if (confirm_size != 0) {
		  ret = -1; /* FIXME: add some error message here */
		  goto clean_up_and_return;
		}
	}

	if ( abortxfer ) {
	  NJB_ERROR(EO_ABORTED);
	  ret = -1;
	  goto clean_up_and_return;
	} 

	if ( path != NULL ) {
	       	close(fd);
		fd = 0;
	}

	ret = 0;

clean_up_and_return:

	if ( block != NULL ) {
	  free(block);
	}

	if ( fd != 0 ) {
	  close(fd);
	}

	/* On error, remove the partial file. */
	if ( ret == -1 ) {
	  unlink(path);
	}

	__leave;
	return ret;
}

/**
 * This is a helper function for sending tracks and files.
 * The NJB_Send_Track() and NJB_Send_File() functions set up
 * the transfer, then call this to transfer the file
 * chunks, then commit the transfer with special commands.
 */

static int send_file (njb_t *njb, const char *path, u_int64_t size, u_int32_t fileid,
		      XferCallback *callback, void *data)
{
  __dsub= "send_file";
  u_int64_t remain, offset;
  u_int32_t bp;
  size_t bread;
  unsigned char *block;
  int fd;
  int abortxfer= 0;
  FILE *fp;
  int retry = 15;
  struct stat sb;
  
  __enter;

#ifdef __WIN32__
  if ( (fd= open(path, O_RDONLY|O_BINARY)) == -1 ) {
#else
  if ( (fd= open(path, O_RDONLY)) == -1 ) {
#endif
    NJB_RERROR("open", -1);
    NJB_ERROR(EO_SRCFILE);
    __leave;
    return -1;
  }
  
  if ( fstat(fd, &sb) == -1 ) {
    NJB_RERROR("fstat", -1);
    NJB_ERROR(EO_SRCFILE);
    __leave;
    return -1;
  }

  if ( (fp= fdopen(fd, "rb")) == NULL ) {
    close(fd);
    NJB_RERROR("fdopen", -1);
    NJB_ERROR(EO_SRCFILE);
    __leave;
    return -1;
  }
  
  /* This space will be used as a reading ring buffer for the transfers */
  block = (unsigned char *) malloc(NJB_BUFSIZ);
  
  /* Terminate if memory block could not be allocated */
  if ( block == NULL ) {
    fclose(fp);
    close(fd);
    NJB_ERROR(EO_NOMEM);
    __leave;
    return -1;
  }
  
  offset = 0;
  remain = size;
  
  /* Set pointer at end of block, so that a new block will be read. */
  bp = NJB_BUFSIZ;

  /* Keep sending while there is stuff to send and noone
   * interrupts us */
  while (remain && !abortxfer) {
    u_int32_t bwritten = 0;
    u_int32_t xfersize;
    u_int32_t readsize;
    u_int32_t maxblock = 0;
    
    if (PROTOCOL3_DEVICE(njb->device_type)) {
      maxblock = NJB3_SEND_FILE_BLOCK_SIZE;
    } else if (njb->device_type == NJB_DEVICE_NJB1) {
      maxblock = NJB_XFER_BLOCK_SIZE;
    }
    
    /* Next we will try to send this much */
    xfersize = (remain < maxblock) ? remain : maxblock;
    /*
      printf("Remain %08x bytes, Buffer %04x bytes, next we will send %04x bytes\n", 
      (u_int32_t) remain, (u_int32_t) NJB_BUFSIZ - bp, xfersize);
    */
    
    /* If the remaining bytes in the buffer is lower than
     * the largest transfer unit, read in a new chunk
     */
    if ( NJB_BUFSIZ - bp < xfersize ) {
      /* Bytes remaining at the bottom of the ringbuffer */
      u_int32_t bufbottom = NJB_BUFSIZ - bp;
      /* Buffer size to fill */
      u_int32_t fillsize = NJB_BUFSIZ - bufbottom;
      
      /* Copy end of buffer to top if there is something in it */
      if (bufbottom > 0) {
	memcpy(&block[0], &block[bp], bufbottom);
      }
      
      /* Read in as much as fills the buffer */
      readsize = (remain - bufbottom > (size_t) fillsize) ? (size_t) fillsize : (size_t) remain - bufbottom;
      /*
	printf("Remain %08x bytes, filling buffer with %08x bytes\n", (u_int32_t) remain, readsize);
      */
      
      if ( (bread = fread(&block[bufbottom], readsize, 1, fp)) < 1 ) {
	if ( ferror(fp) ) {
	  NJB_RERROR("fread", -1);
	  NJB_ERROR(EO_SRCFILE);
	} else {
	  NJB_ERROR2("reached EOF", EO_SRCFILE);
	}
	
	fclose(fp);
	close(fd);
	free(block);
	
	__leave;
	return -1;
      }
      
      /* Reset buffer pointer */
      bp = 0;
      
    }
    
    /*
      printf("Remain %08x bytes, buffer has %08x bytes sending bytes...\n", 
      (u_int32_t) remain, NJB_BUFSIZ-bp);
    */
    
    if (PROTOCOL3_DEVICE(njb->device_type)) {
      bwritten = njb3_send_file_chunk(njb, &block[bp], xfersize, fileid);
    } else if (njb->device_type == NJB_DEVICE_NJB1) {
      bwritten = njb_send_file_block(njb, &block[bp], xfersize);
    }

    if ( bwritten == -1 ) {
      fclose(fp);
      close(fd);
      free(block);
      __leave;
      return -1;
    }

    /* We use bwritten to cope with short transfers */
    remain -= (u_int64_t) bwritten;
    offset += (u_int64_t) bwritten;
    bp += bwritten;
    
    if (PROTOCOL3_DEVICE(njb->device_type)) {
      /* DO NOTHING */
    } else if (njb->device_type == NJB_DEVICE_NJB1) {
      if ( njb_verify_last_command(njb) == -1 ) {
	close(fd);
	free(block);
	
	__leave;
	return -1;
      }
    }

    if (callback != NULL) {
      if ( callback(offset, size, NULL, 0, data) == -1 )
	abortxfer = 1;
    }
  }
  
  free(block);
  fclose(fp);
  close(fd);
	
  
  if (PROTOCOL3_DEVICE(njb->device_type)) {
    /* Complete transfer for NJB3 series devices */
    if ( njb3_send_file_complete(njb, fileid) == -1 ) {
      __leave;
      return -1;
    }
    /* On abort, delete the partial file */
    if ( abortxfer ) {
      if ( njb3_delete_item(njb, fileid) == -1) {
	__leave;
	return -1;
      }
      NJB_ERROR(EO_ABORTED);
      __leave;
      return -1;
    }
  } else {
    /* Complete transfer for NJB1  */
    while ( retry ) {
      if ( njb_transfer_complete(njb) == 0 ) {
	if ( abortxfer ) {
	  NJB_ERROR(EO_ABORTED);
	  __leave;
	  return -1;
	}
	if ( _lib_ctr_update(njb) == -1 ) {
	  NJB_ERROR(EO_BADCOUNT);
	  __leave;
	  return -1;
	}
	__leave;
	return 0;
      }
      sleep(1);
      
      retry--;
    }
    NJB_ERROR(EO_TIMEOUT);
    __leave;
    return -1;
  }
  return 0;
}

int NJB_Send_Track (njb_t *njb, const char *path, const char *codec,
	const char *title, const char *album, const char *genre,
	const char *artist, u_int32_t length, u_int32_t tracknum,
	const char *year, int protect,
	XferCallback *callback, void *data, u_int32_t *trackid)
{
	__dsub= "NJB_Send_Track";
	u_int64_t btotal, bfree, size;
	songid_t song;
	songid_frame_t *frame;
	unsigned char *ptag;
	njbttaghdr_t tagh;
	u_int32_t tagsize, size32;
	char *fname;
	int retry= 3;

	__enter;

	njb_error_clear();

	memset(&song, 0, sizeof(song));

	if (njb->device_type == NJB_DEVICE_NJB1) {
		while (retry) {
			if ( njb_get_disk_usage(njb, &btotal, &bfree) == -1 ) {
				if ( njb_error == EO_AGAIN ) retry--;
				else {
					NJB_ERROR(EO_XFERDENIED);
					__leave;
					return -1;
				}
			} else break;
		}
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  if ( njb3_get_disk_usage(njb, &btotal, &bfree) == -1 ) {
	    NJB_ERROR(EO_XFERDENIED);
	    __leave;
	    return -1;
	  }
	}

	if ( _file_size(path, &size) == -1 ) {
		NJB_ERROR(EO_SRCFILE);
		__leave;
		return -1;
	}

	if ( size > bfree ) {
		NJB_ERROR(EO_TOOBIG);
		__leave;
		return -1;
	}
	size32= (u_int32_t) size;

	/* These pieces are required */

	if ( codec == NULL || path == NULL ) {
		NJB_ERROR(EO_INVALID);
		__leave;
		return -1;
	}

	/* Build the tag for the track */

	if ( (frame= songid_frame_new_codec(codec)) == NULL ) return -1;
	songid_addframe(&song, frame);

	/* FIXME: warning: basename() may modify the contents of path */
	fname= basename( (char*)path);
	if ( (frame= songid_frame_new_fname(fname)) == NULL ) {
		__leave;
		return -1;
	}
	songid_addframe(&song, frame);

	if ( title == NULL ) {
		if ( (frame= songid_frame_new_title(fname)) == NULL ) {
			__leave;
			return -1;
		}
	} else {
		if ( (frame= songid_frame_new_title(title)) == NULL ) {
			__leave;
			return -1;
		}
	}
	songid_addframe(&song, frame);

	if ( (frame= songid_frame_new_filesize(&size32)) == NULL ) {
		__leave;
		return -1;
	}
	songid_addframe(&song, frame);

	if ( album != NULL ) {
		if ( (frame= songid_frame_new_album(album)) == NULL ) {
			__leave;
			return -1;
		}
		songid_addframe(&song, frame);
	}
	else {
		if ( (frame= songid_frame_new_album("<Unknown>")) == NULL ) {
			__leave;
			return -1;
		}
		songid_addframe(&song, frame);
	}
	
	if ( genre != NULL ) {
		if ( (frame= songid_frame_new_genre(genre)) == NULL ) {
			__leave;
			return -1;
		}
		songid_addframe(&song, frame);
	}
	else {
		if ( (frame= songid_frame_new_genre("<Unknown>")) == NULL ) {
			__leave;
			return -1;
		}
		songid_addframe(&song, frame);
	}

	if ( artist != NULL ) {
		if ( (frame= songid_frame_new_artist(artist)) == NULL ) {
			__leave;
			return -1;
		}
		songid_addframe(&song, frame);
	}
	else {
		if ( (frame= songid_frame_new_artist("<Unknown>")) == NULL ) {
			__leave;
			return -1;
		}
		songid_addframe(&song, frame);
	}	

	if ( length ) {
		if ( (frame= songid_frame_new_length(&length)) == NULL ) {
			__leave;
			return -1;
		}
		songid_addframe(&song, frame);
	}

	if ( year != NULL ) {
	       /* The year is a char* on NJB1 series and thus in the interface
	        * to calling applications, whereas in the PROTOCOL3 series devices
	        * it is an u_int16_t. For PROTOCOL3 devices, the conversion will 
		* take place in songid_pack3() in songid.c.
	        */
		if ( (frame= songid_frame_new_year(year)) == NULL ) {
			__leave;
			return -1;
		}
		songid_addframe(&song, frame);
	}
	
	if ( protect == 1 ) {
		if ( (frame= songid_frame_new_protected("1")) == NULL ) {
			__leave;
			return -1;
		}
		songid_addframe(&song, frame);
	}
						
	if ( tracknum ) {
		if ( (frame= songid_frame_new_tracknum(&tracknum)) == NULL ) {
			__leave;
			return -1;
		}
		songid_addframe(&song, frame);
	}

	/* Pack the tag and send it */

	if (njb->device_type == NJB_DEVICE_NJB1) {
		if ( (ptag= songid_pack(&song, &tagsize)) == NULL ) return -1;
		tagh.size= tagsize;

		if ( njb_send_track_tag(njb, &tagh, ptag) == -1 ) {
			if ( njb_error == EO_BADSTATUS ) NJB_ERROR(EO_XFERDENIED);
			free(ptag);
			__leave;
			return -1;
		}

		free(ptag);

		*trackid= tagh.trackid;

		/* The trackid referenced is not actually used with the NJB1 */
		if ( send_file(njb, path, size, *trackid, callback, data) == -1 ) {
			__leave;
			return -1;
		}

	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  /* Request to stop playing before sending a track */
	  njb3_ctrl_playing(njb, NJB3_STOP_PLAY);

	  if ( (ptag= songid_pack3(&song, &tagsize)) == NULL )
	    return -1;
	  if ( (*trackid = njb3_create_file(njb, ptag, tagsize, NJB3_FILETYPE_TRACK)) == 0 ) {
	    if ( njb_error == EO_BADSTATUS ) NJB_ERROR(EO_XFERDENIED);
	    free(ptag);
	    __leave;
	    return -1;
	  }
	  free(ptag);
	  
	  if ( send_file(njb, path, size, *trackid, callback, data) == -1 ) {
	    __leave;
	    return -1;
	  }
	}

	__leave;
	return 0;
}


int NJB_Send_File (njb_t *njb, const char *path, const char *name,
	XferCallback *callback, void *data, u_int32_t *fileid)
{
	__dsub= "NJB_Send_File";
	u_int64_t btotal, bfree, size;
	time_t ts;
	datafile_t df;
	unsigned char *phdr;
	njbdfhdr_t fh;
	u_int32_t fhsize;
	int status;

	__enter;

	njb_error_clear();

	memset(&df, 0, sizeof(df));

	if ( path == NULL ) {
		NJB_ERROR(EO_INVALID);
		__leave;
		return -1;
	}

	if ( name == NULL ) {
		status= datafile_set_name(&df, path);
	} else {
		status= datafile_set_name(&df, name);
	}
	if ( status == -1 ) {
		__leave;
		return -1;
	}

	if ( _file_size(path, &size) == -1 ) {
		NJB_ERROR(EO_SRCFILE);
		__leave;
		return -1;
	}
	if ( _file_time(path, &ts) == -1 ) {
		NJB_ERROR(EO_SRCFILE);
		__leave;
		return -1;
	}

	if (njb->device_type == NJB_DEVICE_NJB1) {
		if ( njb_get_disk_usage(njb, &btotal, &bfree) == -1 ) {
			__leave;
			return -1;
		}
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  if ( njb3_get_disk_usage(njb, &btotal, &bfree) == -1 ) {
	    NJB_ERROR(EO_XFERDENIED);
	    __leave;
	    return -1;
	  }
	}

	if ( size > bfree ) {
		NJB_ERROR(EO_TOOBIG);
		__leave;
		return -1;
	}

	datafile_set_size(&df, size);
	datafile_set_time(&df, ts);

	/* Pack the file tag and send it */

	if (njb->device_type == NJB_DEVICE_NJB1) {

		if ( (phdr = datafile_pack(&df, &fhsize)) == NULL ) return -1;
		fh.size = fhsize;

		if ( njb_send_datafile_tag(njb, &fh, phdr) == -1 ) {
			if ( njb_error == EO_BADSTATUS ) 
			  NJB_ERROR(EO_XFERDENIED);
			free(phdr);
			__leave;
			return -1;
		}

		free(phdr);

		*fileid = fh.dfid;

		/* The fileid referenced is not actually used with the NJB1 */
		if ( send_file(njb, path, size, *fileid, callback, data) == -1 ) {
			__leave;
			return -1;
		}

	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  if ( (phdr = datafile_pack3(njb, &df, &fhsize)) == NULL )
	    return -1;
	  if ( (*fileid = njb3_create_file(njb, phdr, fhsize, NJB3_FILETYPE_FILE)) == 0 ) {
	    if ( njb_error == EO_BADSTATUS ) NJB_ERROR(EO_XFERDENIED);
	    free(phdr);
	    __leave;
	    return -1;
	  }
	  free(phdr);
	  if ( send_file(njb, path, size, *fileid, callback, data) == -1 ) {
	    __leave;
	    return -1;
	  }
	}

	__leave;
	return 0;
}

/* This function is deprecated in favor of the new EAX API */
eax_t *NJB_Get_EAX (njb_t *njb)
{
	__dsub= "NJB_Get_EAX";
	u_int32_t eaxsz;
	eax_t *eax = NULL;

	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1) {
	  if ( njb_get_eax_size(njb, &eaxsz) == -1 ) {
	    __leave;
	    return NULL;
	  }

	  eax= njb_get_eax(njb, eaxsz);
	  if ( eax == NULL ) {
	    __leave;
	    return NULL;
	  }
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  eax = njb3_get_eax(njb);
	}

	__leave;
	return eax;
}

/* This function is deprecated in favor of the new EAX API */
int NJB_Refresh_EAX (njb_t *njb, eax_t *eax)
{
	__dsub= "NJB_Refresh_EAX";
	int res;

	__enter;

	njb_error_clear();

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  __leave;
	  return 0;
	}

	res= njb_refresh_eax(njb, eax);
	if ( res == -1 ) {
		__leave;
		return -1;
	}

	__leave;
	return 0;
}

/* The new EAX API */
void NJB_Reset_Get_EAX_Type (njb_t *njb)
{
  __dsub= "NJB_Reset_Get_EAX_Type";
  __enter;

  njb_error_clear();

  if (njb->device_type == NJB_DEVICE_NJB1) {
    u_int32_t eaxsz;
    if ( njb_get_eax_size(njb, &eaxsz) == -1 ) {
      __leave;
      return;
    }
    njb_read_eaxtypes(njb, eaxsz);
  }
  else if (PROTOCOL3_DEVICE(njb->device_type)) {
    njb3_read_eaxtypes(njb);
  }
  __leave;
  return;
}

njb_eax_t *NJB_Get_EAX_Type (njb_t *njb)
{
  __dsub= "NJB_Get_EAX_Type";
  __enter;
  if (njb->device_type == NJB_DEVICE_NJB1) {
    njb_eax_t *eax = njb_get_nexteax(njb);
    __leave;
    return eax;
  }
  else if (PROTOCOL3_DEVICE(njb->device_type)) {
    njb_eax_t *eax = njb3_get_nexteax(njb);
    __leave;
    return eax;
  }
  __leave;
  return NULL;
}

void NJB_Destroy_EAX_Type (njb_eax_t *eax)

{
  __dsub= "NJB_Destroy_EAX_Type";
  __enter;
  destroy_eax_type(eax);
  __leave;
  return;
}

void NJB_Adjust_EAX (njb_t *njb, 
		     u_int16_t eaxid, 
		     u_int16_t patchindex,
		     int16_t scalevalue)
{
  __dsub= "NJB_Adjust_EAX";
  __enter;


  if (njb->device_type == NJB_DEVICE_NJB1) {
    int16_t sendvalue;
    
    // We assume that a scalevalue != 0 means that this is
    // a scale to be modified.
    if (scalevalue != 0x0000) {
      sendvalue = scalevalue;
    } else {
      sendvalue = (int16_t) patchindex;
    }
    // Ignore return value
    njb_adjust_sound(njb, (u_int8_t) eaxid, sendvalue);
  }
  else if (PROTOCOL3_DEVICE(njb->device_type)) {
    u_int16_t sendindex;
    u_int16_t active;
    njb3_state_t *state = (njb3_state_t *) njb->protocol_state;

    /*
     * Volume is always active. 
     * The rest need to add additional control of the EAX processor 
     * so that it is activated/deactivated as needed.
     */
    if (eaxid == NJB3_VOLUME_FRAME_ID) {
      active = 0x0001;
    } else {
      if (patchindex == 0x0000 && 
	  scalevalue == 0x0000) {
	if (state->eax_processor_active) {
	  njb3_control_eax_processor(njb, 0x0000);
	  state->eax_processor_active = 0x00;
	}
	active = 0x0000;
      } else {
	if (!state->eax_processor_active) {
	  njb3_control_eax_processor(njb, 0x0001);
	  state->eax_processor_active = 0x01;
	}
	active = 0x0001;
      }
    }

    /* Patch 1 is actually patch 0, number 0 is a dummy for the "off" mode */
    if (patchindex > 0x0000) {
      sendindex = patchindex - 1;
    } else {
      sendindex = 0x0000;
    }
    njb3_adjust_eax(njb, eaxid, sendindex, active, scalevalue);
  }
  __leave;
  return;
}

njb_time_t *NJB_Get_Time(njb_t *njb)
{
	__dsub= "NJB_Get_Time";
	njb_time_t *time = NULL;

	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1)
		time= njb_get_time(njb);

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  time= njb3_get_time(njb);
	}

	if ( time == NULL ) {
		__leave;
		return NULL;
	}

	__leave;
	return time;
}

int NJB_Set_Time(njb_t *njb, njb_time_t *time)
{
	__dsub= "NJB_Set_Time";
	int ret;

	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1) {
	    if (njb_set_time(njb, time) == -1) {
		__leave;
	        return -1;
	}

	    ret = njb_verify_last_command(njb);

	    __leave;
	    return ret;
	}
	
	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  if (njb3_set_time(njb, time) == -1) {
	    __leave;
	    return -1;
	  }
	}


	__leave;
	return 0;
}

void NJB_Destroy_Time(njb_time_t *time)
{
  /* This is currently a very simple struct... */
  free(time);
}

int NJB_Delete_Playlist (njb_t *njb, u_int32_t plid)
{
	__dsub= "NJB_Delete_Playlist";
	int ret = 0;

	__enter;

	njb_error_clear();


	if (njb->device_type == NJB_DEVICE_NJB1) {
		ret= njb_delete_playlist(njb, plid);
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  ret= njb3_delete_item(njb, plid);
	}

	__leave;
	return ret;
}

int NJB_Update_Playlist (njb_t *njb, playlist_t *pl)
{
	__dsub= "NJB_Update_Playlist";
	u_int32_t *trids, *tptr;
	u_int32_t oplid = 0;
	playlist_track_t *track;
	int state= pl->_state;
	int ret = 0;
	char *plname;

	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1) {

		switch (state) {
		case NJB_PL_NEW:
			/* Do nothing and fall through */
			break;
		case NJB_PL_UNCHANGED:
			return 0;
		case NJB_PL_CHNAME:
			/* Convert from UTF-8 if needed, then rename */
			if (njb_unicode_flag == NJB_UC_UTF8) {
				plname = utf8tostr(pl->name);
			} else {
				plname = strdup(pl->name);
			}
			if (plname == NULL) {
				NJB_ERROR(EO_NOMEM);
				__leave;
				return -1;
			}
			if ( njb_rename_playlist(njb, pl->plid, plname) == -1 ) {
				free(plname);
				return -1;
			}
			free(plname);
			return njb_verify_last_command(njb);
		case NJB_PL_CHTRACKS:
			oplid= pl->plid;
			if (oplid != 0) {
				if ( njb_rename_playlist(njb, pl->plid, "dead.playlist") == -1 
						|| njb_verify_last_command(njb) == -1) 
					return -1;
			}
			break;
		}

		trids= (u_int32_t *) malloc(sizeof(u_int32_t)*pl->ntracks);
		if ( trids == NULL ) {
			NJB_ERROR(EO_NOMEM);
			__leave;
			return -1;
		}

		playlist_reset_gettrack(pl);
		tptr= trids;
		while ( (track= playlist_gettrack(pl)) != 0 ) {
			*tptr= track->trackid;
			tptr++;
		}
	
		/* Convert name from UTF-8 if needed and create it */
		if (njb_unicode_flag == NJB_UC_UTF8) {
			plname = utf8tostr(pl->name);
		} else {
			plname = strdup(pl->name);
		}
		if (plname == NULL) {
			NJB_ERROR(EO_NOMEM);
			__leave;
			return -1;
		}
		if ( njb_create_playlist(njb, plname, &pl->plid) == -1 ) {
			free(trids);
			__leave;
			return -1;
		}
		free(plname);

		if ( njb_add_multiple_tracks_to_playlist(njb, pl->plid, trids,
			pl->ntracks) == -1 ) {

			free(trids);
			__leave;
			return -1;
		}

		free(trids);

		if ( state == NJB_PL_CHTRACKS && oplid != 0) {
			if ( njb_verify_last_command(njb) == -1 ) {
				__leave;
				return -1;
			}
			if ( njb_delete_playlist(njb, oplid) == -1 ) {
				__leave;
				return -1;
			}
		}

		ret= njb_verify_last_command(njb);
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
		unsigned char *tmpname;

		/* which need to be deallocated, that causes all the fuzz */
		tmpname = strtoucs2(pl->name);

		if (tmpname == NULL) {
			NJB_ERROR(EO_NOMEM);
			__leave;
			return -1;
		}

		switch (state) {
		case NJB_PL_NEW:
			/* Do nothing before creation */
			break;
		case NJB_PL_UNCHANGED:
 		        /* Nothing changed. Just return. */
			ret = 0;
			goto exit_njb3_playlist_update;
		case NJB_PL_CHNAME:
		        /* Changed name, nothing else. */
			if ( njb3_update_string_frame(njb, pl->plid, NJB3_PLNAME_FRAME_ID, tmpname) == -1) {
				ret = -1;
				goto exit_njb3_playlist_update;
			}
			ret = 0;
			goto exit_njb3_playlist_update;
		case NJB_PL_CHTRACKS:
		        /* If track contents are changed, then this deletes the old playlist. */
			oplid= pl->plid;
			if (oplid != 0) {
			  if ( njb3_delete_item(njb, oplid) == -1 ) {
			    ret = -1;
			    goto exit_njb3_playlist_update;
			  }
			}
		}

		trids= (u_int32_t *) malloc(sizeof(u_int32_t)*pl->ntracks);
		if ( trids == NULL ) {
			NJB_ERROR(EO_NOMEM);
			ret = -1;
			goto exit_njb3_playlist_update;
		}

		playlist_reset_gettrack(pl);
		tptr= trids;
		while ( (track= playlist_gettrack(pl)) != 0 ) {
			*tptr= track->trackid;
			tptr++;
		}

		if ( njb3_create_playlist(njb, tmpname, &pl->plid) == -1 ) {
			free(trids);
			ret = -1;
			goto exit_njb3_playlist_update;
		}

		if ( njb3_add_multiple_tracks_to_playlist(njb, &pl->plid, trids,
			pl->ntracks) == -1 ) {

			free(trids);
			ret = -1;
			goto exit_njb3_playlist_update;
		}

		free(trids);


		ret= 0;
	exit_njb3_playlist_update:
		free(tmpname);

	}

	__leave;
	return ret;
}

int NJB_Delete_Track (njb_t *njb, u_int32_t trackid)
{
	__dsub= "NJB_Delete_Track";

	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1) {
		if ( njb_delete_track(njb, trackid) == -1 ) {
			__leave;
			return -1;
		}
		if ( _lib_ctr_update(njb) == -1 ) {
			NJB_ERROR(EO_BADCOUNT);
			__leave;
			return -1;
		}
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  if ( njb3_delete_item(njb, trackid) == -1 ) {
	    __leave;
	    return -1;
	  }
	}

	__leave;
	return 0;
}

int NJB_Delete_Datafile (njb_t *njb, u_int32_t fileid)
{
	__dsub= "NJB_Delete_Datafile";

	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1) {
		if ( njb_delete_datafile(njb, fileid) == -1 ) {
			__leave;
			return -1;
		}

		if ( _lib_ctr_update(njb) == -1 ) {
			NJB_ERROR(EO_BADCOUNT);
			__leave;
			return -1;
		}
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  if (njb3_delete_item (njb, fileid) == -1) {
	    __leave;
	    return -1;
	  }
	}


	__leave;
	return 0;
}

int NJB_Play_Track (njb_t *njb, u_int32_t trackid)
{
	__dsub= "NJB_Play_Track";
	int ret;

	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1) {
	  if ( njb_play_track(njb, trackid) == -1 ) {
	    __leave;
	    return -1;
	  }

	  ret= njb_verify_last_command(njb);

	  __leave;
	  return ret;
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  ret= njb3_play_track (njb, trackid);
	  __leave;
	  return ret;
	}

	__leave;
	return 0;
}

int NJB_Queue_Track (njb_t *njb, u_int32_t trackid)
{
	__dsub= "NJB_Queue_Track";
	int ret;

	__enter;

	njb_error_clear();

	if (njb->device_type == NJB_DEVICE_NJB1) {
		if ( njb_queue_track(njb, trackid) == -1 ) {
			__leave;
			return -1;
		}

		ret= njb_verify_last_command(njb);

		__leave;
		return ret;
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  ret = njb3_queue_track (njb, trackid);
	  __leave;
	  return ret;
	}

	__leave;
	return 0;
}

int NJB_Pause_Play (njb_t *njb)
{
  __dsub= "NJB_Pause_Play";
  int ret;

  __enter;

  njb_error_clear ();

  if (PROTOCOL3_DEVICE(njb->device_type)) {
    ret = njb3_pause_play (njb);
    __leave;
    return ret;
  }

  __leave;
  return 0;
}

int NJB_Resume_Play (njb_t *njb)
{
  __dsub= "NJB_Pause_Play";
  int ret;

  __enter;

  njb_error_clear ();

  if (PROTOCOL3_DEVICE(njb->device_type)) {
    ret = njb3_resume_play (njb);
    __leave;
    return ret;
  }

  __leave;
  return 0;
}

int NJB_Stop_Play (njb_t *njb)
{
	__dsub= "NJB_Stop_Play";
	int ret;

	__enter;

	njb_error_clear();

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  ret= njb3_stop_play (njb);
	  __leave;
	  return ret;
	}

	if (njb->device_type == NJB_DEVICE_NJB1) {
	  ret= njb_stop_play(njb);

	  __leave;
	  return ret;
	}

	__leave;
	return 0;
}

int NJB_Seek_Track (njb_t *njb, u_int32_t position)
{
	__dsub= "NJB_Seek_Track";
	int ret;

	__enter;

	njb_error_clear();

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  ret = njb3_seek_track (njb, position);
	  __leave;
	  return ret;
	}

	__leave;
	return 0;
}

int NJB_Elapsed_Time (njb_t *njb, u_int16_t *elapsed, int *change)
{
	__dsub= "NJB_Elapsed_Time";
	int ret;

	__enter;

	njb_error_clear();

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  ret= njb3_elapsed_time (njb, elapsed, change);
	  __leave;
	  return ret;
	}

	if (njb->device_type == NJB_DEVICE_NJB1) {
	  ret= njb_elapsed_time(njb, elapsed, change);

	  __leave;
	  return ret;
	}

	__leave;
	return 0;
}

int _file_time (const char *path, time_t *ts)
{
	__dsub= "_file_size";
	struct stat sb;

	__enter;

	if ( stat(path, &sb) == -1 ) {
		NJB_ERROR3("stat", path, -1);
		__leave;
		return -1;
	}

	*ts= sb.st_mtime;

	__leave;
	return 0;
}

int _file_size (const char *path, u_int64_t *size)
{
	__dsub= "_file_size";
	struct stat sb;

	__enter;

	if ( stat(path, &sb) == -1 ) {
		NJB_ERROR3("stat", path, -1);
		__leave;
		return -1;
	}

	*size= sb.st_size;

	__leave;
	return 0;
}

void NJB_Set_Debug (int debug_flags)
{
	njb_set_debug(debug_flags);
}

void NJB_Set_Unicode (int unicode_flag)
{
	njb_set_unicode(unicode_flag);
}

int NJB_Replace_Track_Tag (njb_t *njb, u_int32_t trackid, const char *codec,
	const char *title, const char *album, const char *genre,
	const char *artist, u_int32_t length, u_int32_t tracknum,
	u_int32_t filesize, const char *fname, const char *year, int protect)
{
	__dsub= "NJB_Replace_Track_Tag";

	__enter;

	/*
	 * Ad Hoc-code for the Zen NX device by Friso Brugmans
	 * enabled for all devices when proven to work for all
	 * PROTOCOL 3 devices.
	 *
	 * This routine toggles all changed string frames to a
	 * different name, as changing the name back and forth
	 * to the same value seems to cause trouble.
	 */
	if (PROTOCOL3_DEVICE(njb->device_type)) {
	        njb3_state_t *state = (njb3_state_t *) njb->protocol_state;

		if(state->extrarun_on_rename) {
		  state->extrarun_on_rename = 0;
		} else {
			char *tmptitle = NULL;
			char *tmpalbum = NULL;
			char *tmpgenre = NULL;
			char *tmpartist = NULL;
			char *tmpfname = NULL;

			state->extrarun_on_rename = 1;

			if(title) {
				tmptitle = malloc(strlen(title)+6);
				if ( tmptitle == NULL ) {
					NJB_ERROR(EO_NOMEM);
					__leave;
					return -1;
				}
				strcpy(tmptitle, title);
				strcat(tmptitle, ".temp");
			}
			
			if(album) {
				tmpalbum = malloc(strlen(album)+6);
				if ( tmpalbum == NULL ) {
					NJB_ERROR(EO_NOMEM);
					__leave;
					return -1;
				}
				strcpy(tmpalbum, album);
				strcat(tmpalbum, ".temp"); 
			}
			
			if(genre) {
				tmpgenre = malloc(strlen(genre)+6);
				if ( tmpgenre == NULL ) {
					NJB_ERROR(EO_NOMEM);
					__leave;
					return -1;
				}
				strcpy(tmpgenre, genre);
				strcat(tmpgenre, ".temp");
			}
			
			if(artist) {
				tmpartist = malloc(strlen(artist)+6);
				if ( tmpartist == NULL ) {
					NJB_ERROR(EO_NOMEM);
					__leave;
					return -1;
				}
				strcpy(tmpartist, artist);
				strcat(tmpartist, ".temp");
			}
			
			if(fname) {
				tmpfname = malloc(strlen(fname)+6);
				if ( tmpfname == NULL ) {
					NJB_ERROR(EO_NOMEM);
					__leave;
					return -1;
				}
				strcpy(tmpfname, fname);
				strcat(tmpfname, ".temp");
			}
			
			NJB_Replace_Track_Tag (njb, trackid, NULL,
			                       tmptitle, tmpalbum, tmpgenre, tmpartist,
					       0, tracknum, 0, tmpfname, year, 0);
			
			if(tmptitle) free(tmptitle);
			if(tmpalbum) free(tmpalbum);
			if(tmpgenre) free(tmpgenre);
			if(tmpartist) free(tmpartist);
			if(tmpfname) free(tmpfname);
		}
	}

	njb_error_clear();

       	if ( trackid == 0) {
       		NJB_ERROR(EO_INVALID);
       		__leave;
       		return -1;
       	}

	if (njb->device_type == NJB_DEVICE_NJB1) {

		songid_t song;
		songid_frame_t *frame;
		unsigned char *ptag;
		u_int32_t tagsize;
		njbttaghdr_t tagh;

		memset(&song, 0, sizeof(song));

		/* This is required */

		if ( codec == NULL) {
			NJB_ERROR(EO_INVALID);
			__leave;
			return -1;
		}

		/* Build the tag */

		if ( (frame= songid_frame_new_codec(codec)) == NULL ) return -1;
		songid_addframe(&song, frame);

		if ( fname != NULL ) {
			if ( (frame= songid_frame_new_fname(fname)) == NULL ) {
				__leave;
				return -1;
			}
			songid_addframe(&song, frame);
		}
	
		if ( title != NULL ) {
			if ( (frame= songid_frame_new_title(title)) == NULL ) {
				__leave;
				return -1;
			}
			songid_addframe(&song, frame);
		}

		if ( album != NULL ) {
			if ( (frame= songid_frame_new_album(album)) == NULL ) {
				__leave;
				return -1;
			}
			songid_addframe(&song, frame);
		}

		if ( genre != NULL ) {
			if ( (frame= songid_frame_new_genre(genre)) == NULL ) {
				__leave;
				return -1;
			}
			songid_addframe(&song, frame);
		}

		if ( artist != NULL ) {
			if ( (frame= songid_frame_new_artist(artist)) == NULL ) {
				__leave;
				return -1;
			}
			songid_addframe(&song, frame);
		}

		if ( length ) {
			if ( (frame= songid_frame_new_length(&length)) == NULL ) {
				__leave;
				return -1;
			}
			songid_addframe(&song, frame);
		}

		if ( tracknum ) {
			if ( (frame= songid_frame_new_tracknum(&tracknum)) == NULL ) {
				__leave;
				return -1;
			}
			songid_addframe(&song, frame);
		}

		if ( filesize ) {
			if ( (frame= songid_frame_new_filesize(&filesize)) == NULL ) {
				__leave;
				return -1;
			}
			songid_addframe(&song, frame);
		}

		if ( year != NULL ) {
			if ( (frame= songid_frame_new_year(year)) == NULL ) {
				__leave;
				return -1;
			}
			songid_addframe(&song, frame);
		}
		
		if ( protect == 1 ) {
			if ( (frame= songid_frame_new_protected("1")) == NULL ) {
				__leave;
				return -1;
			}
			songid_addframe(&song, frame);
		}
		
		/* Now pack the tag and send it */

		if ( (ptag= songid_pack(&song, &tagsize)) == NULL ) return -1;
		tagh.size= tagsize;
		tagh.trackid= trackid;
	
		if ( njb_replace_track_tag(njb, &tagh, ptag) == -1 ) {
			free(ptag);
			__leave;
			return -1;
		}

		free(ptag);

		if ( _lib_ctr_update(njb) == -1 ) {
			NJB_ERROR(EO_BADCOUNT);
			__leave;
			return -1;
		}

		__leave;
		return 0;
	}

	/**
	 * Rewrite of the above routine by Friso Brugmans to pack edited frames
	 * of a metadata into one single command block.
	 */
	if (PROTOCOL3_DEVICE(njb->device_type)) {
		unsigned char *tmp;
		unsigned char *packedframe;
		u_int32_t packedlen;
		/* Whole section needs checking for null strings from strtoucs2,
		 * and -1 return codes from njb3_update_x_tagframe */

		packedframe = NULL;
		packedlen = 0;

		/* Create a metadata block to fill in */
		packedframe = njb3_create_update_command(&packedlen, trackid);

		if (codec != NULL) {
		  if (!strcmp(codec, NJB_CODEC_MP3))
		    packedframe = njb3_add_16bit_frame(packedframe, 
						       &packedlen, 
						       NJB3_CODEC_FRAME_ID, 
						       NJB3_CODEC_MP3_ID);
		  else if (!strcmp(codec, NJB_CODEC_WAV))
		    packedframe = njb3_add_16bit_frame(packedframe, 
						       &packedlen, 
						       NJB3_CODEC_FRAME_ID, 
						       NJB3_CODEC_WAV_ID);
		  else if (!strcmp(codec, NJB_CODEC_WMA))
		    packedframe = njb3_add_16bit_frame(packedframe, 
						       &packedlen, 
						       NJB3_CODEC_FRAME_ID, 
						       NJB3_CODEC_WMA_ID);
		}

		if (title != NULL) {
		  tmp = strtoucs2(title);
		  packedframe = njb3_add_string_frame(packedframe, 
						      &packedlen, 
						      NJB3_TITLE_FRAME_ID, 
						      tmp);
		  free(tmp);
		}
		if (album != NULL) {
		  tmp = strtoucs2(album);
		  packedframe = njb3_add_string_frame(packedframe, 
						      &packedlen, 
						      NJB3_ALBUM_FRAME_ID, 
						      tmp);
		  free(tmp);
		}
		if (genre != NULL) {
		  tmp = strtoucs2(genre);
		  packedframe = njb3_add_string_frame(packedframe, 
						      &packedlen, 
						      NJB3_GENRE_FRAME_ID, 
						      tmp);
		  free(tmp);
		}
		if (artist != NULL) {
		  tmp = strtoucs2(artist);
		  packedframe = njb3_add_string_frame(packedframe, 
						      &packedlen, 
						      NJB3_ARTIST_FRAME_ID, 
						      tmp);
		  free(tmp);
		}
		if (length != 0) {
		  packedframe = njb3_add_16bit_frame(packedframe, 
						     &packedlen, 
						     NJB3_LENGTH_FRAME_ID, 
						     length);
		}

		/* 0xffffffff is -1 in signed arithmetic */
		if (tracknum != 0xffffffff) {
		  packedframe = njb3_add_16bit_frame(packedframe, 
						     &packedlen, 
						     NJB3_TRACKNO_FRAME_ID, 
						     tracknum);
		}
		/*
		  // You shouldn't update this anyway
		  if (filesize != 0) {
		  njb3_update_32bit_tagframe(trackid, NJB3_FILESIZE_FRAME_ID, filesize);
		  }
		*/

		if (fname != NULL) {
		  tmp = strtoucs2(fname);
		  packedframe = njb3_add_string_frame(packedframe, 
						      &packedlen, 
						      NJB3_FNAME_FRAME_ID, 
						      tmp);
		  free(tmp);
		}

		if (year != NULL) {
		  int yearint;
		  char *dummy;
		  
		  /* The interface uses a string, but PROTOCOL3 series
		   * uses an u_int32 value, so convert it */
		  yearint = (int) strtoul(year, &dummy, 10);
		  packedframe = njb3_add_16bit_frame(packedframe, 
						     &packedlen,
						     NJB3_YEAR_FRAME_ID, 
						     yearint);
		}

		if (protect != 0) {
		  /*
		   * Dunno how to handle this right now, once files are protected,
		   * they cannot be unprotected on a NJB3 family product, but perhaps
		   * you can protect unprotected tracks? 
		   */
		}

		/* Finally send the new metadata */
		njb3_send_packed_frames(njb, packedframe, packedlen);
	}
	return 0;
}

int NJB_Adjust_Sound(njb_t *njb, u_int8_t effect, u_int16_t value)
{
	__dsub= "NJB_Adjust_Sound";
	int ret;

	__enter;

	njb_error_clear();

	if (PROTOCOL3_DEVICE(njb->device_type)) {

	  if (effect == NJB_SOUND_SET_VOLUME) {
	    njb3_adjust_eax(njb, NJB3_VOLUME_FRAME_ID, 0x0000, 0x0001, value);
	    __leave;
	    return 0;
	  }

	  __leave;
	  return 0;
	}

	ret = njb_adjust_sound(njb, effect, value);

	__leave;
	return ret;
}

void _skip_whitespaces (FILE *f) {
  __dsub= "_skip_whitespaces";

  int c;

  __enter;
  while (!feof (f)) {
    c = fgetc (f);
    if (!isspace (c)) {
      if (c == '#') { // Skip comment
	while (!feof (f) && (c = fgetc (f)) != '\n')
	  ;
      }
      else {
	ungetc (c, f);
	break;
      }
    }
  }
  __leave;
}

int _verify_pbm (FILE *f) {
  __dsub= "_verify_pbm";

  u_int16_t magic;
  unsigned int width, height;
  int c;

  __enter;

  if (fread (&magic, 1, 2, f) < 2) {
    NJB_RERROR("fread", -1);
    NJB_ERROR(EO_SRCFILE);
    __leave;
    return -1;
  }

  if (magic != 0x3450) {
    __leave;
    return -1;
  }

  _skip_whitespaces (f);
  width = 0;
  fscanf (f, "%u", &width);
  /* XXX:TODO Needs to be device depended */
  if (width != 132) {
    __leave;
    return -1;
  }
  _skip_whitespaces (f);
  fscanf (f, "%u", &height);
  /* XXX:TODO Needs to be device depended */
  if (height != 64) {
    __leave;
    return -1;
  }
  c = fgetc (f);
  if (!isspace (c)) {
    __leave;
    return -1;
  }

  __leave;
  return 0;
}

int NJB_Set_Bitmap(njb_t *njb, const char * path) {
  __dsub= "NJB_Set_Bitmap";

  __enter;
  if (PROTOCOL3_DEVICE(njb->device_type)) {
    FILE *f;
    char data[1088];

    if ( (f= fopen(path, "rb")) == NULL) {
      NJB_RERROR("fopen", -1);
      NJB_ERROR(EO_SRCFILE);
      __leave;
      return -1;
    }

    if ( _verify_pbm (f) == -1 ) {
      NJB_RERROR("_verify_pbm", -1);
      NJB_ERROR(EO_SRCFILE);
      fclose (f);
      __leave;
      return -1;
    }

    if ( fread (data, 1, 1088, f) < 1088 ) {
      NJB_RERROR("fread", -1);
      NJB_ERROR(EO_SRCFILE);
      fclose (f);
      __leave;
      return -1;
    }

    fclose (f);
    if ( njb2_set_bitmap (njb, data) == -1 ) {
      fclose (f);
      __leave;
      return -1;
    }

    __leave;
    return 0;
  }
  
  __leave;
  return 0;
}

njbid_t *NJB_Ping(njb_t *njb)
{
	__dsub= "NJB_Ping";
	njbid_t *njbid;

	__enter;

	njbid = malloc(sizeof(njbid_t));

	if ( njbid == NULL ) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return NULL;
	}

	if (njb->device_type == NJB_DEVICE_NJB1) {
		if ( njb_ping(njb, njbid) == -1 ) {
			__leave;
			free(njbid);
			return NULL;
		}
	}

	if (PROTOCOL3_DEVICE(njb->device_type)) {
	  if ( njb3_ping(njb, njbid, 0) == -1 ) {
	    __leave;
	    free(njbid);
	    return NULL;
	  }
	  if ( njb3_readid(njb, njbid) == -1 ) {
	    __leave;
	    free(njbid);
	    return NULL;
	  }
	}

	__leave;
	return njbid;
}

/* Retrieve the scanned keys */
njb_keyval_t *NJB_Get_Keys(njb_t *njb)
{
  if (PROTOCOL3_DEVICE(njb->device_type)) {
      return njb3_get_keys(njb);
  }
  return NULL;
}
