/**
 * \file base.c
 *
 * This file contains the basic stuff for opening the device
 * on the USB bus and similar things. Here are the jukebox detection
 * algorithms for example.
 */

#include "libnjb.h"
#include "njbusb.h"
#include "njb_error.h"
#include "defs.h"
#include "base.h"
#include "protocol.h"
#include "protocol3.h"
#include "usb_io.h"

/** The current debug flags for all if libnjb (global) */
int njb_debug_flags = 0;
/** The current subroutine depth for all of libnjb (global) */
int __sub_depth = 0;

#ifdef __NetBSD__
#define MAXDEVNAMES USB_MAX_DEVNAMES
#endif

/**
 * Search the USB bus for a Nomad JukeBox.  We can handle up to
 * NJB_MAX_DEVICES JukeBox's per USB simultaneously (a value arbitrarily
 * set in base.h, because it's easier to work with a fixed size array than
 * build dynamically growing code...if anyone _really_ plans on having
 * more than NJB_MAX_DEVICES jukeboxes on their system, they can recompile
 * with a bigger number).  The USB device starts at usb0 and goes to usbNNN
 * where NNN is USB_MAX_DEVICES.  Check each bus for a usb device.  Return
 * -1 if errors occurred...though this doesn't mean we failed to find
 * devices.  *count gives the number of NJB devices that were found.
 * Store the resulting NJB structues structure into the njbs[NJB_MAX_DEVICES]
 * array.
 *
 * @param njbs an array of jukeboxes to fill up. The array must be
 *             NJB_MAX_DEVICES large.
 * @param limit a variable that is supposed to limit the number of
 *              devices retrieved, currently not used.
 * @param count a pointer to a variable that will hold the number of
 *              devices found after the call to this function.
 * @return 0 on OK, -1 on failure.
 */
int njb_discover (njb_t *njbs, int limit, int *count)
{
	__dsub= "njb_discover";
	struct usb_bus *busses;
	struct usb_bus *bus;
	struct usb_device *device;
	int found;

	__enter;

	found = 0;
	*count = 0;

	usb_init();

	/* Try to locate busses and devices 
	 * used to be:
	 * if ( usb_find_busses() ) retval= -1; 
	 * if ( usb_find_devices() ) retval= -1;
	 * which was no good idea.
	 */
	usb_find_busses();
	usb_find_devices();
	busses = usb_get_busses();

	bus = busses;
	while ( bus != NULL ) {
		device = bus->devices;
		while ( device != NULL ) {
			if ( device->descriptor.idVendor == NJB1_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJB1_PRODUCT_ID ) {
				njbs[found].device = device;
				njbs[found].dev = NULL;
				njbs[found].device_type = NJB_DEVICE_NJB1;
				found ++;
			}
			if ( device->descriptor.idVendor == NJB2_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJB2_PRODUCT_ID ) {
				njbs[found].device = device;
				njbs[found].dev = NULL;
				njbs[found].device_type = NJB_DEVICE_NJB2;
				found ++;
			}
			if ( device->descriptor.idVendor == NJB3_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJB3_PRODUCT_ID ) {
				njbs[found].device = device;
				njbs[found].dev = NULL;
				njbs[found].device_type = NJB_DEVICE_NJB3;
				found ++;
			}
			if ( device->descriptor.idVendor == NJBZEN_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJBZEN_PRODUCT_ID ) {
				njbs[found].device= device;
				njbs[found].dev= NULL;
				njbs[found].device_type= NJB_DEVICE_NJBZEN;
				found ++;
			}
			if ( device->descriptor.idVendor == NJBZEN2_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJBZEN2_PRODUCT_ID ) {
				njbs[found].device = device;
				njbs[found].dev = NULL;
				njbs[found].device_type = NJB_DEVICE_NJBZEN2;
				found ++;
			}
			if ( device->descriptor.idVendor == NJBZENNX_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJBZENNX_PRODUCT_ID ) {
				njbs[found].device = device;
				njbs[found].dev = NULL;
				njbs[found].device_type = NJB_DEVICE_NJBZENNX;
				found ++;
			}
			if ( device->descriptor.idVendor == NJBZENXTRA_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJBZENXTRA_PRODUCT_ID ) {
				njbs[found].device = device;
				njbs[found].dev = NULL;
				njbs[found].device_type = NJB_DEVICE_NJBZENXTRA;
				found ++;
			}
			if ( device->descriptor.idVendor == DELLDJ_VENDOR_ID
				&& device->descriptor.idProduct ==
				DELLDJ_PRODUCT_ID ) {
				njbs[found].device = device;
				njbs[found].dev = NULL;
				njbs[found].device_type = NJB_DEVICE_DELLDJ;
				found ++;
			}
			if ( device->descriptor.idVendor == NJBZENTOUCH_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJBZENTOUCH_PRODUCT_ID ) {
				njbs[found].device = device;
				njbs[found].dev = NULL;
				njbs[found].device_type = NJB_DEVICE_NJBZENTOUCH;
				found ++;
			}
			if ( device->descriptor.idVendor == NJBZENMICRO_VENDOR_ID
				&& device->descriptor.idProduct ==
				NJBZENMICRO_PRODUCT_ID ) {
				njbs[found].device = device;
				njbs[found].dev = NULL;
				njbs[found].device_type = NJB_DEVICE_NJBZENMICRO;
				found ++;
			}
			if ( device->descriptor.idVendor == DELLDJ2_VENDOR_ID
				&& device->descriptor.idProduct ==
				DELLDJ2_PRODUCT_ID ) {
				njbs[found].device = device;
				njbs[found].dev = NULL;
				njbs[found].device_type = NJB_DEVICE_DELLDJ2;
				found ++;
			}
			device = device->next;
		}
		bus = bus->next;
	}

	/* The number of devices we have found */
	*count = found;

	__leave;
	return 0;
}

/**
 * Close a specific njb for reading and writing.  
 *
 * @param njb the jukebox object to close
 */
void njb_close (njb_t *njb)
{
	__dsub= "njb_close";
	__enter;

        usb_release_interface(njb->dev, 0);

        /*
	 * Resetting the USB bus is not popular amongst
	 * NJB2/3/ZEN devices, and will just be made for
	 * NJB1.
         */
	if (njb->device_type == NJB_DEVICE_NJB1) {
	  usb_resetep(njb->dev, njb->usb_bulk_out_ep);
	  usb_reset(njb->dev);
	}

        usb_close(njb->dev);

	if (njb->njbid != NULL)
	  free(njb->njbid);

	__leave;
}

/**
 * Go through the USB descriptor block for the device and try to
 * locate endpoints and interfaces to use for communicating with
 * the device.
 *
 * @param njb the jukebox object associated with the USB device 
 *            to parse
 */
static void parse_usb_descriptor(njb_t *njb)
{
  int i, j, k, l;
  int found_interface = 0;
  int found_in_ep = 0;
  int found_out_ep = 0;
  u_int8_t config = 0x00;
  u_int8_t interface = 0x00;
  u_int8_t in_ep = 0x00;
  u_int8_t out_ep = 0x00;

  if (njb->device_type == NJB_DEVICE_NJB1) {
    njb->usb_config = 0x01; /* The others have 0x00 mostly */
    njb->usb_interface = 0x00;
    njb->usb_bulk_out_ep = 0x02;
    njb->usb_bulk_in_ep = 0x82;
  } else {
    /* Print descriptor information */
    /* printf("The device has %d configurations.\n", njb->device->descriptor.bNumConfigurations); */
    i = 0;
    while (!found_interface && i < njb->device->descriptor.bNumConfigurations) {
      struct usb_config_descriptor *conf = &njb->device->config[i];
      /* printf("Configuration %d, value %d, has %d interfaces.\n", i, conf->bConfigurationValue, conf->bNumInterfaces); */
      j = 0;
      while (!found_interface && j < conf->bNumInterfaces) {
	struct usb_interface *iface = &conf->interface[j];
	/* printf("  Interface %d, has %d altsettings.\n", j, iface->num_altsetting); */
	k = 0;
	while (!found_interface && k < iface->num_altsetting) {
	  struct usb_interface_descriptor *ifdesc = &iface->altsetting[k];
	  /* printf("    Altsetting %d, number %d, has %d endpoints.\n", k, ifdesc->bInterfaceNumber, ifdesc->bNumEndpoints); */
	  found_in_ep = 0;
	  found_out_ep = 0;
	  for (l = 0; l < ifdesc->bNumEndpoints; l++) {
	    struct usb_endpoint_descriptor *ep = &ifdesc->endpoint[l];
	    /* printf("    Endpoint %d, no %02xh, attributes %02xh\n", l, ep->bEndpointAddress, ep->bmAttributes); */
	    if (!found_out_ep && (ep->bEndpointAddress & 0x80) == 0x00) {
	      /* printf("    Found WRITE (OUT) endpoint %02xh\n", ep->bEndpointAddress); */
	      found_out_ep = 1;
	      out_ep = ep->bEndpointAddress;
	    }
	    if (!found_in_ep && (ep->bEndpointAddress & 0x80) != 0x00) {
	      /* printf("    Found READ (IN) endpoint %02xh\n", ep->bEndpointAddress); */
	      found_in_ep = 1;
	      in_ep = ep->bEndpointAddress;
	    }
	  }
	  if (found_in_ep == 1 && found_out_ep == 1) {
	    found_interface = 1;
	    interface = ifdesc->bInterfaceNumber;
	    config = conf->bConfigurationValue;
	  }
	  k++;
	}
	j++;
      }
      i++;
    }
    if (found_interface) {
      /* printf("Found config %d, interface %d, IN EP: %02xh, OUT EP: %02xh\n", config, interface, in_ep, out_ep); */
      njb->usb_config = config;
      njb->usb_interface = interface;
      njb->usb_bulk_out_ep = out_ep;
      njb->usb_bulk_in_ep = in_ep;
    } else {
      /*
       * This is some code that should never need to run!
       */
      printf("LIBNJB panic: could not locate a suitable interface.\n");
      printf("LIBNJB panic: resorting to heuristic interface choice.\n");
      njb->usb_config = 0;
      njb->usb_interface = 0;
      if (USB20_DEVICE(njb->device_type)) {
	if (njb->device_type == NJB_DEVICE_NJBZENMICRO) {
	  njb->usb_bulk_out_ep = 0x02; /* NJB Zen Micro use endpoint 2 OUT */
	}
	/* The other USB 2.0 jukeboxes use endpoint 1 OUT */
	njb->usb_bulk_out_ep = 0x01;
      } else {
	/*
	 * The original NJB1, NJB3 and the NJB Zen FW-edition,
	 * i.e. all USB 1.1 devices use endpoint 2 OUT
	 */
	njb->usb_bulk_out_ep = 0x02;
      }
      /* Default for all devices */
      njb->usb_bulk_in_ep = 0x82;
    }
  }
}

/**
 * Open a specific njb for reading and writing.  
 *
 * @param njb the jukebox object to open
 * @return 0 on success, -1 on failure
 */
int njb_open (njb_t *njb)
{
	__dsub= "njb_open";
	__enter;

	njb->njbid = NULL;
	
	/* Check what config, interface and endpoints to use */
	parse_usb_descriptor(njb);

	if ( (njb->dev = usb_open(njb->device)) == NULL ) {
		njb_error_add(njb, "usb_open", -1);
		__leave;
		return -1;
	}

	/*
	 * The "high speed" devices (USB 2.0) have two configurations.
	 * the second one may be for operating the device under "full speed"
	 * instead, so that it becomes slower.
	 */
	if ( usb_set_configuration(njb->dev, njb->usb_config) ) {
		njb_error_add(njb, "usb_set_configuration", -1);
		__leave;
		return -1;
	}

	/* With several jukeboxes connected, a call will often fail
	 * here when you try to connect to the second jukebox after
	 * closing the first. Why? */
	if ( usb_claim_interface(njb->dev, njb->usb_interface) ) {
		njb_error_add(njb, "usb_claim_interface", -1);
		__leave;
		return -1;
	}
	

	/*
	 * This should not be needed. Removing, cause it 
	 * caused problems on MacOS X.
	 */
	/*
	if ( usb_set_altinterface(njb->dev, 0) ) {
		njb_error_add(njb, "usb_set_altinterface", -1);
		__leave;
		return -1;
	}
	*/

	__leave;
	return 0;
}

/**
 * Set the debug flags to use.
 *
 * @param flags the flags to set
 */
void njb_set_debug (int flags)
{
	njb_debug_flags = flags;
}

/**
 * get the current debug flags
 *
 * @param flags a binary mask that remove some of the
 *              flags.
 */
int njb_debug (int flags)
{
	return njb_debug_flags & flags;
}

