//
//  MusicTab.m
//  XNJB
//

/* this tab is for the music tab
 * upload/download tracks, delete
 */

#import "MusicTab.h"
#import "Track.h"
#import "ID3Tagger.h"
#import "defs.h"

// these correspond to the identifiers in the NIB file
#define COLUMN_TITLE @"Title"
#define COLUMN_ARTIST @"Artist"
#define COLUMN_ALBUM @"Album"
#define COLUMN_GENRE @"Genre"
#define COLUMN_LENGTH @"Length"
#define COLUMN_TRACK_NO @"Track No"
#define COLUMN_CODEC @"Codec"
#define COLUMN_FILESIZE @"Filesize"
#define COLUMN_YEAR @"Year"

// declare the private methods
@interface MusicTab (PrivateAPI)
- (NSString *)filenameForTrack:(Track *)track;
@end

@implementation MusicTab

// init/dealloc

- (id)init
{
	if (self = [super init])
	{
		previousTrackIndexSelected = -1;
		
		fileExtensionsToCodecsDictionary = [[NSDictionary alloc] initWithObjectsAndKeys:[NSNumber numberWithInt:CODEC_MP3],
			@"mp3", [NSNumber numberWithInt:CODEC_WMA], @"wma", [NSNumber numberWithInt:CODEC_WAV], @"wav", nil];
		
		trackListIsUpToDate = NO;
	}
	return self;
}

- (void)dealloc
{
	[fileExtensionsToCodecsDictionary release];
	[super dealloc];
}

// accessor methods

- (BOOL)trackListIsUpToDate
{
	return trackListIsUpToDate;
}

/**********************/

- (void)awakeFromNib
{
	[super awakeFromNib];
	NSArray *extensions = [NSArray arrayWithObjects:@"mp3", @"wma", @"wav", nil];
	[browser setAllowedExtensions:extensions];
	[browser setBaseLocation:[preferences musicTabDir]];
	
	[njbTable setAllowedExtensions:extensions];
	[njbTable setAutosaveName:PREF_KEY_MUSIC_TAB_TABLE];
	[njbTable setAutosaveTableColumns:YES];
}

- (void)onConnectAndActive
{
	[super onConnectAndActive];
	if ([playlistsTab trackListIsUpToDate] && [playlistsTab fullItemArray])
	{
		// playlistsTab has already got it
		[fullItemArray release];
		fullItemArray = [[MyMutableArray alloc] initWithMyMutableArray:[playlistsTab fullItemArray]];
		[self clearSortingArrows];
		[self searchNJBTable];
		
		trackListIsUpToDate = YES;
	}
	else
	{
		[self loadTracks];
		trackListIsUpToDate = YES;
	}
}

- (SEL)sortSelectorForColumn:(NSString *)columnID
{
	if ([columnID isEqual:COLUMN_TITLE])
		return [Track compareByTitleSelector];
	else if ([columnID isEqual:COLUMN_ARTIST])
		return [Track compareByArtistSelector];
	else if ([columnID isEqual:COLUMN_ALBUM])
		return [Track compareByAlbumSelector];
	else if ([columnID isEqual:COLUMN_GENRE])
		return [Track compareByGenreSelector];
	else if ([columnID isEqual:COLUMN_LENGTH])
		return [Track compareByLengthSelector];
	else if ([columnID isEqual:COLUMN_TRACK_NO])
		return [Track compareByTrackNumberSelector];
	else if ([columnID isEqual:COLUMN_CODEC])
		return [Track compareByCodecSelector];
	else if ([columnID isEqual:COLUMN_FILESIZE])
		return [Track compareByFilesizeSelector];
	else if ([columnID isEqual:COLUMN_YEAR])
		return [Track compareByYearSelector];
	else
	{
		NSLog(@"Invalid column %@ in sortSelectorForColumn!", columnID);
		return nil;
	}
}

/* replaces track variables (%t, %a, ...)
 * with actual values for the track given.
 * does not add extension
 * called when copying file from NJB
 */
- (NSString *)filenameForTrack:(Track *)track
{
	// ignores [track filename] as this is always blank on my NJB3
	NSMutableString *filename = [[NSMutableString alloc] init];
	
	NSString *filenameFormat = [preferences filenameFormat];
	int i = -1;
	BOOL gotPercent = NO;
	for (i = 0; i < [filenameFormat length]; i++)
	{
		unichar currentChar = [filenameFormat characterAtIndex:i];
		if (gotPercent)
		{
			gotPercent = NO;
			switch (currentChar)
			{
				case '%':
					[filename appendString:@"%"];
					break;
					// title
				case 't':
					[filename appendString:[self replaceBadFilenameChars:[track title]]];
					break;
					// artist
				case 'a':
					[filename appendString:[self replaceBadFilenameChars:[track artist]]];
					break;
					// album
				case 'A':
					[filename appendString:[self replaceBadFilenameChars:[track album]]];
					break;
					// genre
				case 'g':
					[filename appendString:[self replaceBadFilenameChars:[track genre]]];
					break;
					// track no
				case 'n':
					[filename appendString:[NSString stringWithFormat:@"%d",[track trackNumber]]];
					break;
					// codec
				case 'c':
					[filename appendString:[track codecString]];
					break;
				default:
					// incorrect variable name so print nothing
					break;
			}
		}
		else if (currentChar == '%')
			gotPercent = YES;
		else
			[filename appendString:[NSString stringWithCharacters:&currentChar length:1]];
	}
	NSString *f = [NSString stringWithString:filename];
	[filename release];
	return f;
}

/* copy the item at index in itemArrayDisplaying
 * from the NJB
 */
- (void)copyFileFromNJB:(int)index
{
	Track *track = (Track *)[[self itemArrayDisplaying] objectAtIndex:index];
	NSFileManager *manager = [NSFileManager defaultManager];
	
	NSString *filename = [self filenameForTrack:track];
	NSString *directory = [[browser directory] stringByAppendingString:@"/"];
	
	NSArray *components = [filename pathComponents];
	NSMutableString *dirsSoFar = [[NSMutableString alloc] initWithString:directory];
	int i = -1;
	for (i = 0; i < [components count]; i++)
	{
		NSString *component = [components objectAtIndex:i];
		
		if ([component isEqual:@"/"])
		{
			// the format string starts with /, this will get ignored when prefixing it with directory
			// so don't do anything now
		}
		else if (i != [components count] - 1)
		{
			// this is not the filename, need to make a directory
			[dirsSoFar appendString:component];
			[dirsSoFar appendString:@"/"];
			NSLog(@"making directory %@", dirsSoFar);
			// create the directory, will fail if already made, don't worry
			[manager createDirectoryAtPath:dirsSoFar attributes:nil];
		}
	}
	
	NSString *extension = [[NSString stringWithString:@"."] stringByAppendingString:[[track codecString] lowercaseString]];
	NSString *fullPath = [directory stringByAppendingString:filename];
	
	while ([manager fileExistsAtPath:[fullPath stringByAppendingString:extension]])
	{
		NSLog(@"file exists: %@%@", fullPath, extension);
		fullPath = [fullPath stringByAppendingString:@"_"];
	}
	
	fullPath = [fullPath stringByAppendingString:extension];
	
	NSLog(@"downloading track %@ to %@", [track title], fullPath);
	
	[track setFullPath:fullPath];
		
	[self copyTrackFromNJB:track];
}

/* actually add the track to the queue
 * to be copied
 */
- (void)copyTrackFromNJB:(Track *)track
{
	if (![theNJB isConnected])
		return;
		
	NJBQueueItem *copyTrack = [[NJBQueueItem alloc] initWithTarget:theNJB withSelector:@selector(downloadTrack:) withObject:track];
	[copyTrack setStatus:STATUS_DOWNLOADING_TRACK];
	NSString *description = [NSString stringWithFormat:@"Downloading track '%@'", [track title]];
	
	// tell PlaylistsTab
	NJBQueueItem *updatePlaylistsTab = [[NJBQueueItem alloc] initWithTarget:playlistsTab withSelector:@selector(updateTracks)];
	[updatePlaylistsTab setRunInMainThread:NO];
	[updatePlaylistsTab setDisplayStatus:NO];
	
	// could update browser here, but causes browser to lose selection so annoying
	
	// write track tag to file
	if ([preferences writeTagAfterCopy] && [track codec] == CODEC_MP3)
	{
		NJBQueueItem *writeTag = [[NJBQueueItem alloc] initWithTarget:self withSelector:@selector(writeTagToFile:) withObject:track];
		// putting this to NO causes weird and wonderful crashes later on e.g. memory of the njb_t struct in theNJB getting overwritten...
		// should be YES though (wasn't pre 0.23)
		[writeTag setRunInMainThread:YES];
		[writeTag setDisplayStatus:NO];

		[self addToQueue:copyTrack withSubItems:[NSArray arrayWithObjects:updatePlaylistsTab, writeTag, nil] withDescription:description];
		[writeTag release];
	}
	else
		[self addToQueue:copyTrack withSubItems:[NSArray arrayWithObjects:updatePlaylistsTab, nil] withDescription:description];
	
	[copyTrack release];
	[updatePlaylistsTab release];
}

/* copy the file at filename to
 * the NJB
 */
- (void)copyFileToNJB:(NSString *)filename
{
	ID3Tagger *tagger = [[ID3Tagger alloc] init];
	[tagger setFileExtensionsToCodecsDictionary:fileExtensionsToCodecsDictionary];
	[tagger setFilename:filename];
	Track *track = [tagger readTrack];
	[track setFullPath:filename];
	[self copyTrackToNJB:track];
	[tagger release];
}

/* actually add the track to the queue
 * to be copied
 */
- (void)copyTrackToNJB:(Track *)track
{
	if (![theNJB isConnected])
		return;
	NJBQueueItem *copyTrack = [[NJBQueueItem alloc] initWithTarget:theNJB withSelector:@selector(uploadTrack:) withObject:track];
	[copyTrack setStatus:STATUS_UPLOADING_TRACK];
	
	NJBQueueItem *addToList = [[NJBQueueItem alloc] initWithTarget:self withSelector:@selector(addToFullItemArray:) withObject:track withRunInMainThread:YES];
	[addToList setDisplayStatus:NO];
	
	// tell PlaylistsTab
	NJBQueueItem *updatePlaylistsTab = [[NJBQueueItem alloc] initWithTarget:playlistsTab withSelector:@selector(updateTracks)];
	[updatePlaylistsTab setRunInMainThread:NO];
	[updatePlaylistsTab setDisplayStatus:NO];
	
	NSString *description = [NSString stringWithFormat:@"Uploading track '%@'", [track title]];
	[self addToQueue:copyTrack withSubItems:[NSArray arrayWithObjects:addToList, updatePlaylistsTab, nil] withDescription:description];
	
	[copyTrack release];
	[addToList release];
	[updatePlaylistsTab release];
}

/* write the info in track to the tag in the file
 */
- (void)writeTagToFile:(Track *)track
{
	ID3Tagger *tagger = [[ID3Tagger alloc] init];
	[tagger writeTrack:track];
	[tagger release];
}

- (id)tableView:(NSTableView *)aTableView
   objectValueForTableColumn:(NSTableColumn *)aTableColumn
						row:(int)rowIndex
{	
	NSString *ident = [aTableColumn identifier];
	Track *track = (Track *)[[self itemArrayDisplaying] objectAtIndex:rowIndex];
	if ([ident isEqual:COLUMN_TITLE])
		return [track title];
	else if ([ident isEqual:COLUMN_ALBUM])
		return [track album];
	else if ([ident isEqual:COLUMN_ARTIST])
		return [track artist];
	else if ([ident isEqual:COLUMN_GENRE])
		return [track genre];
	else if ([ident isEqual:COLUMN_LENGTH])
		return [track lengthString];
	else if ([ident isEqual:COLUMN_TRACK_NO])
		return [NSString stringWithFormat:@"%d", [track trackNumber]];
	else if ([ident isEqual:COLUMN_CODEC])
		return [track codecString];
	else if ([ident isEqual:COLUMN_FILESIZE])
		return [track filesizeString];
	else if ([ident isEqual:COLUMN_YEAR])
		return [NSString stringWithFormat:@"%d", [track year]];
	else
	{
		NSLog(@"Invalid column tag in MusicTab");
		return nil;
	}
}

- (void)tableView:(NSTableView *)tableView didClickTableColumn:(NSTableColumn *)tableColumn
{	
	// unchanged tag edit will be lost
	
	// disable and clear the drawer
	[drawerController clearAll];
	[drawerController disableAll];
	
	[super tableView:tableView didClickTableColumn:tableColumn];
}

/* update the drawer when someone selects a different
 * track
 */
- (void)tableViewSelectionDidChange:(NSNotification *)aNotification
{
	[super tableViewSelectionDidChange:aNotification];
	// unchanged tag edit will be lost
	
	int newRow = [njbTable selectedRow];
	previousTrackIndexSelected = newRow;
	if (newRow == -1)
		return;
	
	objectToSaveTrack = OBJECT_NJB_TABLE;
	
	// update display to show new track
	[drawerController showTrack:(Track *)[[self itemArrayDisplaying] objectAtIndex:newRow]];
	if (![theNJB isConnected])
		[drawerController disableWrite];
}

/* called by MainController when we must write
 * a track to a file/njb
 */
- (void)drawerWriteTag:(Track *)modifiedTrack
{
	switch (objectToSaveTrack)
	{
		case OBJECT_NJB_TABLE:
			NSLog(@"Saving to njb");
			int rowIndex = [njbTable selectedRow];
			// none selected, shouldn't be here anyway
			if (rowIndex == -1)
				return;	
			
			NJBQueueItem *changeTrackTag = [[NJBQueueItem alloc] initWithTarget:theNJB withSelector:@selector(changeTrackTag:) withObject:modifiedTrack];
			[changeTrackTag setStatus:STATUS_UPDATING_TRACK_TAG];
			
			Track *oldTrack = (Track *)[[self itemArrayDisplaying] objectAtIndex:rowIndex];
			
			NJBQueueItem *updateTrackList = [[NJBQueueItem alloc] initWithTarget:self withSelector:@selector(replaceTrack:withTrack:)
																																withObject:oldTrack withRunInMainThread:YES];
			[updateTrackList setObject2:modifiedTrack];
			[updateTrackList setDisplayStatus:NO];

			// call searchNJBTable to pretend the search list changed to update
			// saves data and does a reload
			NJBQueueItem *updateDisplay = [[NJBQueueItem alloc] initWithTarget:self withSelector:@selector(searchNJBTable)];
			[updateDisplay setRunInMainThread:YES];
			[updateDisplay setDisplayStatus:NO];
			
			// tell PlaylistsTab
			NJBQueueItem *updatePlaylistsTab = [[NJBQueueItem alloc] initWithTarget:playlistsTab withSelector:@selector(updateTracks)];
			[updatePlaylistsTab setRunInMainThread:NO];
			[updatePlaylistsTab setDisplayStatus:NO];
			
			NSString *description = [NSString stringWithFormat:@"Changing tag for track '%@'", [oldTrack title]];
			[self addToQueue:changeTrackTag withSubItems:[NSArray arrayWithObjects:updateTrackList, updateDisplay, updatePlaylistsTab, nil] withDescription:description];

			[changeTrackTag release];
			[updateTrackList release];
			[updateDisplay release];
			[updatePlaylistsTab release];
			
			break;
		case OBJECT_FILE_BROWSER:
			NSLog(@"Saving to browser, filename %@", [modifiedTrack fullPath]);
			[self writeTagToFile:modifiedTrack];
			break;
		default:
			NSLog(@"Drawer write tag without njbTable/fileBrowser selected!");
	}
}

- (void)searchNJBTable
{
	// will mess up indexes, don't want to save anyway
	previousTrackIndexSelected = -1;
	
	NSString *searchString = [njbSearchField stringValue];
	// release old track list
	[truncatedItemArray release];
	
	if ([searchString length] == 0)
	{
		truncatedItemArray = nil;
		[njbTable reloadData];
		return;
	}
	
	truncatedItemArray = [fullItemArray searchFor:searchString];
	[truncatedItemArray retain];
	
	[njbTable reloadData];
	if ([njbTable selectedRow] != -1)
	{
		// update display to show new track
		objectToSaveTrack = OBJECT_NJB_TABLE;
		[drawerController showTrack:(Track *)[[self itemArrayDisplaying] objectAtIndex:[njbTable selectedRow]]];
		if (![theNJB isConnected])
			[drawerController disableWrite];
	}
	else
	{
		[drawerController clearAll];
		[drawerController disableAll];
	}	
}

- (void)fileSystemBrowserFileClick:(NSString *)path
{
	[super fileSystemBrowserFileClick:path];
	objectToSaveTrack = OBJECT_FILE_BROWSER;
	// unchanged tag edit will be lost
	ID3Tagger *tagger = [[ID3Tagger alloc] initWithFilename:path];
	[tagger setFileExtensionsToCodecsDictionary:fileExtensionsToCodecsDictionary];
	[drawerController enableAll];
	Track *track = [tagger readTrack];
	[drawerController showTrack:track];
	[tagger release];
	if ([track codec] != CODEC_MP3)
		[drawerController disableWrite];
}

- (void)fileSystemBrowserDirectoryClick:(NSString *)path
{
	[super fileSystemBrowserDirectoryClick:path];
	// unchanged tag edit will be lost
	[drawerController disableAll];
}

- (void)onFirstResponderChange:(NSResponder *)aResponder
{
	[super onFirstResponderChange:(NSResponder *)aResponder];
	if (aResponder == njbTable)
	{
		objectToSaveTrack = OBJECT_NJB_TABLE;
		// update display to show new track
		if ([njbTable selectedRow] > -1)
		{
			[drawerController showTrack:(Track *)[[self itemArrayDisplaying] objectAtIndex:[njbTable selectedRow]]];
			if (![theNJB isConnected])
				[drawerController disableWrite];
		}
	}
	else if ([aResponder class] == [NSMatrix class])
		objectToSaveTrack = OBJECT_FILE_BROWSER;
	else if ([aResponder class] == [NSSearchField class])
	{
		[drawerController disableAll];
	}
}

/* add downloadTrackList to the njb queue
 */
- (void)loadTracks
{
	NJBQueueItem *clearSortingArrows = [[NJBQueueItem alloc] initWithTarget:self
																														 withSelector:@selector(clearSortingArrows)
																															 withObject:nil
																											withRunInMainThread:YES];
	[clearSortingArrows setDisplayStatus:NO];
	
	NJBQueueItem *getTracks = [[NJBQueueItem alloc] initWithTarget:self withSelector:@selector(downloadTrackList)];
	[getTracks setStatus:STATUS_DOWNLOADING_TRACK_LIST];

	NJBQueueItem *updateTable = [[NJBQueueItem alloc] initWithTarget:self
																											withSelector:@selector(searchNJBTable)
																												withObject:nil
																							 withRunInMainThread:YES];
	[updateTable setDisplayStatus:NO];
	
	NSString *description = @"Getting track list";
	[self addToQueue:getTracks withSubItems:[NSArray arrayWithObjects:clearSortingArrows, updateTable, nil] withDescription:description];
	[getTracks release];
	[clearSortingArrows release];
	[updateTable release];
}

/* this is called from the queue consumer 
 * to get the tracks off the NJB and put them in fullItemArray
 * will run in worker thread
 */
- (NJBTransactionResult *)downloadTrackList
{
	NSLog(@"Getting tracks");
	[fullItemArray release];
	fullItemArray = [theNJB tracks];
	
	NJBTransactionResult *result = [[NJBTransactionResult alloc] initWithSuccess:(fullItemArray != nil)];
	
	[fullItemArray retain];
	
	return [result autorelease];
}

- (void)deleteFromNJB:(MyItem *)item
{
	Track *track = (Track *)item;

	NJBQueueItem *deleteTrack = [[NJBQueueItem alloc] initWithTarget:theNJB withSelector:@selector(deleteTrack:)
																												withObject:([NSNumber numberWithInt:[track trackID]])];
	[deleteTrack setStatus:STATUS_DELETING_TRACK];
		
	NJBQueueItem *removeFromAllTracks = [[NJBQueueItem alloc] initWithTarget:fullItemArray withSelector:@selector(removeObject:)
																																	withObject:track];
	[removeFromAllTracks setRunInMainThread:YES];
	[removeFromAllTracks setDisplayStatus:NO];
		
	// call searchNJBTable to pretend the search list changed to update
	// saves data and does a reload
	NJBQueueItem *updateTrackList = [[NJBQueueItem alloc] initWithTarget:self withSelector:@selector(searchNJBTable)];
	[updateTrackList setRunInMainThread:YES];
	[updateTrackList setDisplayStatus:NO];
	
	// tell PlaylistsTab
	NJBQueueItem *updatePlaylistsTab = [[NJBQueueItem alloc] initWithTarget:playlistsTab withSelector:@selector(updateTracks)];
	[updatePlaylistsTab setRunInMainThread:NO];
	[updatePlaylistsTab setDisplayStatus:NO];
	
	NSString *description = [NSString stringWithFormat:@"Deleting track '%@'", [track title]];
	[self addToQueue:deleteTrack withSubItems:[NSArray arrayWithObjects:removeFromAllTracks, updateTrackList, updatePlaylistsTab, nil] withDescription:description];

	[deleteTrack release];
	[removeFromAllTracks release];
	[updateTrackList release];
	[updatePlaylistsTab release];
		
	NSLog(@"Deleting track id %d with title %@", [track trackID], [track title]);		
}

- (void)replaceTrack:(Track *)replace withTrack:(Track *)new
{
	[fullItemArray replaceObject:replace withObject:new];
}

- (void)NJBDisconnected:(NSNotification *)note
{
	[super NJBDisconnected:note];
	
	if (isActive && activeObject == OBJECT_NJB_TABLE)
		[drawerController disableWrite];
	
	trackListIsUpToDate = NO;
}

- (void)activate
{
	[super activate];
	if (activeObject == OBJECT_NJB_TABLE)
	{
		if ([njbTable selectedRow] != -1)
		{
			[drawerController showTrack:(Track *)[[self itemArrayDisplaying] objectAtIndex:[njbTable selectedRow]]];
			if (![theNJB isConnected])
				[drawerController disableWrite];
		}
	}
	else if (activeObject == OBJECT_FILE_BROWSER)
	{
		if (![browser isDirectory])
		{
			ID3Tagger *tagger = [[ID3Tagger alloc] initWithFilename:[browser path]];
			[tagger setFileExtensionsToCodecsDictionary:fileExtensionsToCodecsDictionary];
			[drawerController enableAll];
			[drawerController showTrack:[tagger readTrack]];
			[tagger release];
		}
	}
}

- (void)applicationTerminating:(NSNotification *)note
{
	[preferences setLastUsedMusicTabDir:[self browserDirectory]];
	[super applicationTerminating:note];
}

@end