//
//  MyFileTab.m
//  XNJB
//

/* This is for a tab that has both an NSTableView for jukebox items and a FileSystemBrowser 
 * for local items. Implements methods for copying between the two and deleting.
 */

#import "MyFileTab.h"
#import "defs.h"
#import "MyNSWindow.h"

@implementation MyFileTab

// init/dealloc methods

- (id)init
{
	if (self = [super init])
	{
		currentColumnID = nil;
		
		NSNotificationCenter *nc;
		nc = [NSNotificationCenter defaultCenter];
		[nc addObserver:self selector:@selector(applicationTerminating:) name:NOTIFICATION_APPLICATION_TERMINATING object:nil];
	}
	return self;
}

- (void)dealloc
{
	[browser release];
	[super dealloc];
}

/**********************/

- (void)awakeFromNib
{
	[super awakeFromNib];
	browser = [[FileSystemBrowser alloc] initWithBrowser:fileBrowser atLocation:@"/"];
	[browser setTarget:self];
	[browser setFileAction:@selector(fileSystemBrowserFileClick:)];
	[browser setDirectoryAction:@selector(fileSystemBrowserDirectoryClick:)];
	
	// allow drag & drop
	[njbTable registerDragAndDrop];
	[njbTable disallowCopies];
}

/* makes a string into a legal filename
 * this is the filename excluding directory
 */
- (NSString *)replaceBadFilenameChars:(NSString *)filename
{
	if (filename == nil || [filename length] == 0)
		return @"";
	NSMutableString *file = [[NSMutableString alloc] initWithString:filename];
	[file replaceOccurrencesOfString:@"/" withString:@"_" options:NSCaseInsensitiveSearch range:NSMakeRange(0, [filename length])];
	NSString *f = [NSString stringWithString:file];
	[file release];
	
	return f;
}

/* To be implemented by super class:
 * called by MainController when the user clicks on 
 * the write button in the tag drawer
 */
- (void)drawerWriteTag:(Track *)replaceTrackAtIndex{}

/* called when the user clicks on the delete menu
 */
- (void)menuDelete
{
	switch ([self activeObject])
	{
		case OBJECT_FILE_BROWSER:
			[self deleteFromFileSystem];
			break;
		case OBJECT_NJB_TABLE:
			[self deleteFromNJB];
			break;
		default:
			break;
	}
}

/* can we delete something?
 */
- (BOOL)canDelete
{
	switch ([self activeObject])
	{
		case OBJECT_NJB_TABLE:
			if (![theNJB isConnected])
				return NO;
			if ([njbTable selectedRow] != -1)
				return YES;
			else
				return NO;
			break;
		case OBJECT_FILE_BROWSER:
			if ([browser directorySelected])
				return NO;
			else
				return YES;
			break;
		default:
			return NO;
	}		
}

- (BOOL)menuShouldDelete
{
	return [self canDelete];
}

/* called by onFirstResponderChange: to tell us which object is
 * currently active
 */
- (void)setActiveObject:(activeObjects)newActiveObject
{
	activeObject = newActiveObject;
}

- (activeObjects)activeObject
{
	return activeObject;
}

- (void)onFirstResponderChange:(NSResponder *)aResponder
{
	// don't do anything if it's the window - we will have been called with another object
	if ([aResponder class] == [MyNSWindow class])
		return;
	if (aResponder == njbTable)
		[self setActiveObject:OBJECT_NJB_TABLE];
	else if ([aResponder class] == [NSMatrix class])
		[self setActiveObject:OBJECT_FILE_BROWSER];
	else
		[self setActiveObject:OBJECT_OTHER];
	[deleteFromActiveObjectButton setEnabled:[self canDelete]];
}

/* can we copy a file to the NJB?
 */
- (BOOL)canCopyToNJB
{
	return ([browser path] != nil && [[browser path] length] > 0);
}

/* can we copy a file from the NJB?
 */
- (BOOL)canCopyFromNJB
{
	return ([njbTable selectedRow] != -1);
}

/* called when we disconnect
 */
- (void)NJBDisconnected:(NSNotification *)note
{
	[super NJBDisconnected:note];
	[self disableTransfers];
}

/* called when we connect
 */
- (void)NJBConnected:(NSNotification *)note
{
	[super NJBConnected:note];
	[self enableTransfers];
}

/* things to do when we terminate (implemented by superclass)
 */
- (void)applicationTerminating:(NSNotification *)note{}

/* enable the buttons to allow transfers if connected
 */
- (void)enableTransfers
{
	if (![theNJB isConnected])
	{
		[self disableTransfers];
		return;
	}
	[deleteFromActiveObjectButton setEnabled:[self canDelete]];
	[copyToNJBButton setEnabled:[self canCopyToNJB]];
	[copyFromNJBButton setEnabled:[self canCopyFromNJB]];
	[njbTable allowCopies];
}

/* disable transfer buttons
 */
- (void)disableTransfers
{
	[deleteFromActiveObjectButton setEnabled:[self canDelete]];
	[copyToNJBButton setEnabled:NO];
	[copyFromNJBButton setEnabled:NO];
	[njbTable disallowCopies];
}

/* get a list of all the files to copy to the NJB
 */
- (IBAction)copyToNJB:(id)sender
{
	NSArray *filesToCopy = [browser selectedFilesWithRecursion:YES];
	NSString *currentFile;
	NSEnumerator *enumerator = [filesToCopy objectEnumerator];
	while (currentFile = [enumerator nextObject])
	{
		[self copyFileToNJB:currentFile];
	}
}

/* actually copy the file, implemented by superclass
 */
- (void)copyFileToNJB:(NSString *)filename{}

/* get a list of all the files to copy from the NJB
 */
- (IBAction)copyFromNJB:(id)sender
{
	NSEnumerator *trackEnumerator = [njbTable selectedRowEnumerator];
	NSNumber *index;
	while (index = [trackEnumerator nextObject])
	{
		[self copyFileFromNJB:[index intValue]];
	}
}

/* actually copy the file, implemented by superclass
 */ 
- (void)copyFileFromNJB:(int)index{}

/* delete a file locally
 * will delete all files in a directory but not directories or recursively
 */
- (void)deleteFromFileSystem
{
	if ([browser directorySelected])
	{
		// should not be here
		NSLog(@"Error: deleting local files when directory selected!");
		return;
	}
	int result = NSRunAlertPanel(@"File Deletion", @"Are you sure you want to delete the selected local files?", @"No", @"Yes", nil);
	if (result == NSAlertDefaultReturn)
		return;
	
	NSFileManager *fileManager = [[NSFileManager alloc] init];
	NSArray *selectedFiles = [browser selectedFilesWithRecursion:NO];
	NSEnumerator *enumerator = [selectedFiles objectEnumerator];
	NSString *currentFile;
	while (currentFile = [enumerator nextObject])
	{
		NSLog(@"Removing file: %@", currentFile);
		if (![fileManager removeFileAtPath:currentFile handler:nil])
		{
			NSRunAlertPanel(@"File Deletion", [NSString stringWithFormat:@"There was an error deleting file %@.", currentFile], @"OK", nil, nil);
			return;
		}
	}
	[fileManager release];
	[browser reloadData];
}

/* delete all the selected items off the NJB, after warning
 */
- (void)deleteFromNJB
{
	int result = NSRunAlertPanel(@"File Deletion", @"Are you sure you want to delete the selected items off the NJB?", @"No", @"Yes", nil);
	if (result == NSAlertDefaultReturn)
		return;
	
	/* build up an array of items to be deleted
	 * we can't pass indexes as they aren't valid
	 * after items have been deleted
	 */
	NSEnumerator *enumerator = [njbTable selectedRowEnumerator];
	NSNumber *index;
	NSMutableArray *selectedItems = [[NSMutableArray alloc] initWithCapacity:[njbTable numberOfSelectedRows]];
	while (index = [enumerator nextObject])
	{
	  [selectedItems addObject:[[self itemArrayDisplaying] objectAtIndex:[index intValue]]];
	}
	enumerator = [selectedItems objectEnumerator];
	MyItem *currentItem;
	while (currentItem = [enumerator nextObject])
	{
		[self deleteFromNJB:currentItem];
	}
	[selectedItems release];
}

/* delete an off the NJB, to be implemented by superclass
 */
- (void)deleteFromNJB:(MyItem *)item{}

/* clicked on a track/file so we can transfer
 */
- (void)tableViewSelectionDidChange:(NSNotification *)aNotification
{
	[self enableTransfers];
}

/* clicked on a local track/file so we can transfer
 */
- (void)fileSystemBrowserFileClick:(NSString *)path
{
	[self enableTransfers];
}

/* clicked on a local directory so we can transfer
 */
- (void)fileSystemBrowserDirectoryClick:(NSString *)path
{
	[self enableTransfers];
}

- (void)deactivate
{
	[super deactivate];
	[drawerController disableAll];
}

- (NSString *)browserDirectory
{
	return [browser directory];
}

/* user clicked on delete button
 */
- (IBAction)deleteFromActiveObject:(id)sender
{
	[self menuDelete];
}

/* adds the item to fullItemArray and calls
* searchNJBTable. To be run when new item added to
* list.
*/
- (void)addToFullItemArray:(MyItem *)item
{
	[fullItemArray addObject:item];
	[self searchNJBTable];
}

@end
