/**
 * file njb_error.c
 *
 * Functions dealing with generic error handling.
 */

#include <string.h>
#include <stdio.h>
#include <errno.h>
#include "libnjb.h"
#include "defs.h"
#include "protocol.h"
#include "njb_error.h"

/* int _error_overflow (void); */
const char *njb_status_string (unsigned char code);

/**
 * This function returns an error code as a string.
 * 
 * @param code the error code as a number
 * @return a string representing the error code
 */
static const char *njb_error_string (int code)
{
  switch (code)
    {
    case -1:
      return strerror(errno);
    case 0:
      return "";
    case EO_USBCTL:
      return "I/O failure on USB control pipe";
    case EO_USBBLK:
      return "I/O failure on USB data pipe";
    case EO_RDSHORT:
      return "short read on USB data pipe";
    case EO_NOMEM:
      return "out of memory";
    case EO_BADDATA:
      return "invalid data";
    case EO_EOM:
      return "end of data";
    case EO_BADSTATUS:
      return "bad status from Jukebox";
    case EO_BADNJBID:
      return "Jukebox ID has changed";
    case EO_BADCOUNT:
      return "library count mismatch";
    case EO_WRSHORT:
      return "short write on USB data pipe";
    case EO_NULLTMP:
      return "temporary transfer dir not defined";
    case EO_TOOBIG:
      return "block size too big";
    case EO_CANTMOVE:
      return "can't move file to destination";
    case EO_TIMEOUT:
      return "operation timed out";
    case EO_ABORTED:
      return "operation aborted";
    case EO_EOF:
      return "received EOF";
    case EO_DEVICE:
      return "can't open device for read/write";
    case EO_INIT:
      return "can't initialize device";
    case EO_TMPFILE:
      return "can't create temporary file";
    case EO_XFERDENIED:
      return "transfer request denied";
    case EO_WRFILE:	
      return "error writing output file";
    case EO_XFERERROR:
      return "bad transfer completion status";
    case EO_SRCFILE:
      return "can't read source file";
    case EO_INVALID:
      return "invalid arguments";
    case EO_AGAIN:
      return "resource temporarily unavailable";
    case EO_BAD_NJB1_REPLACE:
      return "the NJB1 needs complete tag info when replacing tags";
    default:
      return "(undefined error)";
    }
}

/**
 * This indicates if the error stack has been overflowed.
 *
 * @param njb a pointer to the device object to check
 */
static int error_overflow (njb_t *njb)
{
  njb_error_stack_t *estack = (njb_error_stack_t *) njb->error_stack;

  if ( estack->count >= MAX_ERRORS ) {
    strcpy(estack->msg[MAX_ERRORS], "Error stack overflow");
    estack->count = MAX_ERRORS+1;
    return 1;
  }
  
  return 0;
}

void njb_error_add3 (njb_t *njb, const char *sub, const char *prefix, const char
		     *suffix, int code)
{
  njb_error_stack_t *estack = (njb_error_stack_t *) njb->error_stack;
  char *ep = estack->msg[estack->count];
  
  if ( error_overflow(njb) ) return;
  
  snprintf(ep, MAX_ERRLEN, "%s: %s: %s %s", sub, prefix,
	   njb_error_string(code), suffix);
  
  estack->count++;
}

void njb_error_add2 (njb_t *njb, const char *sub, const char *prefix, int code)
{
  njb_error_stack_t *estack = (njb_error_stack_t *) njb->error_stack;
  char *ep = estack->msg[estack->count];

  if ( error_overflow(njb) ) return;
  
  snprintf(ep, MAX_ERRLEN, "%s: %s: %s", sub, prefix,
	   njb_error_string(code));
  
  estack->count++;
}

void njb_error_add (njb_t *njb, const char *sub, int code)
{
  njb_error_stack_t *estack = (njb_error_stack_t *) njb->error_stack;
  char *ep = estack->msg[estack->count];

  if ( error_overflow(njb) ) return;
  
  snprintf(ep, MAX_ERRLEN, "%s: %s", sub, njb_error_string(code));
  
  estack->count++;
}

void njb_error_add_string (njb_t *njb, const char *sub, const char *error)
{
  njb_error_stack_t *estack = (njb_error_stack_t *) njb->error_stack;
  char *ep = estack->msg[estack->count];

  if ( error_overflow(njb) ) return;
  
  snprintf(ep, MAX_ERRLEN, "%s: %s", sub, error);
  
  estack->count++;
}


/**
 * This clears the internal memory stack for a device.
 *
 * @param njb a pointer to the device object to clear off errors
 */
void njb_error_clear (njb_t *njb)
{
  njb_error_stack_t *estack = (njb_error_stack_t *) njb->error_stack;

  estack->count = 0;
  estack->idx = 0;
}

/**
 * This function tells wheter an error message is queued and pending
 * for the current device. If so, the error should be retrieved
 * using <code>NJB_Error_Geterror()</code> or dumped using
 * <code>NJB_Error_Dumperror()</code>.
 *
 * @param njb a pointer to the NJB object to use
 * @return 0 if there are no errors pending, 1 if there are errors pending
 * @see NJB_Error_Reset_Geterror()
 * @see NJB_Error_Geterror()
 * @see NJB_Error_Dump()
 */
int NJB_Error_Pending(njb_t *njb)
{
  njb_error_stack_t *estack = (njb_error_stack_t *) njb->error_stack;

  if (estack->count > 0) {
    return 1;
  }
  return 0;
}

/**
 * This function resets the internal error stack, so that 
 * old errors do not remain on following calls to retrieve
 * the error.
 * 
 * Retrieve the errors if the function 
 * <code>NJB_Error_Pending()</code> indicate that there are
 * errors pending. Typical usage:
 *
 * <pre>
 * njb_t *njb;
 * char *errstr;
 *
 * if (NJB_Error_Pending(njb) {
 *    NJB_Error_Reset_Geterror(njb);
 *    while ( (errstr = NJB_Error_Geterror(njb)) != NULL )  {
 *          printf("%s\n", errstr);
 *    }
 * }
 * </pre>
 *
 * @param njb a pointer to the NJB object to use
 * @see NJB_Error_Pending()
 * @see NJB_Error_Geterror()
 */
void NJB_Error_Reset_Geterror(njb_t *njb)
{
  njb_error_stack_t *estack = (njb_error_stack_t *) njb->error_stack;

  estack->idx = 0;
}

/**
 * This function returns the topmost error on the error 
 * stack as a string. The result is statically allocated and MUST
 * NOT be freed by the calling application. This function should
 * be repeatedly called until no errors remain and the function
 * returns NULL. After retrieveing all error strings like this,
 * the error stack is cleared and none of the errors will be
 * returned again.
 *
 * @param njb a pointer to the NJB object to use
 * @return a string representing one error on the error stack
 *         or NULL.
 */
const char *NJB_Error_Geterror(njb_t *njb)
{
  njb_error_stack_t *estack = (njb_error_stack_t *) njb->error_stack;

  const char *cp;

  if ( estack->idx == estack->count ) {
      njb_error_clear(njb);
      return NULL;
  }
  cp = estack->msg[estack->idx];
  estack->idx++;
  return cp;
}


/**
 * This function dumps the current libnjb error stack to a file,
 * such as stderr. All errors currently on the stack are dumped.
 * After dumping, the error stack is cleared and errors are not
 * returned again. The function may be called on an empty error
 * stack without effect, but you may just as well check the stack
 * yourself using <code>NJB_Error_Pending()</code> as in the 
 * example below.
 *
 * Typical usage:
 *
 * <pre>
 * njb_t *njb;
 *
 * if (NJB_Error_Pending(njb)) {
 *     NJB_Error_Dump(njb, stderr);
 * }
 * </pre>
 *
 * @param njb a pointer to the NJB object to use
 * @param fp a file pointer to dump the textual representation of
 *        the error stack to.
 * @see NJB_Error_Pending()
 */
void NJB_Error_Dump(njb_t *njb, FILE *fp)
{
  const char *sp;
  
  NJB_Error_Reset_Geterror(njb);
  while ( (sp = NJB_Error_Geterror(njb)) != NULL ) {
    fprintf(fp, "%s\n", sp);
  }
  njb_error_clear(njb);
}
