//
//  ID3Tagger.m
//  id3libtest
//
//  Created by Richard Low on Mon Jul 26 2004.
//

/* this class provides a wrapper for id3lib
 * to read/write id3 tags
 */

#include "wmaread.h"
#import "ID3Tagger.h"
#include <id3/tag.h>
#include <id3/misc_support.h>
#import "WMATagger.h"
#import "UnicodeWrapper.h"
#import "MP3Length.h"
// for stat
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

@interface ID3Tagger (PrivateAPI)
// NB these can't be delcared in ID3Tagger.h as would then include id3/tag.h in other files which causes errors
// unless a -I./id3lib/include compiler flag is set
// but should be private anyway
- (NSString *)frameContentsFromTag:(ID3_Tag *)tag frameID:(ID3_FrameID)frameID;
- (int)intFrameContentsFromTag:(ID3_Tag *)tag frameID:(ID3_FrameID)frameID;
- (void)setFrameContentsForTag:(ID3_Tag *)tag frameID:(ID3_FrameID)frameID contents:(NSString *)contents;

- (unsigned)wavFileLength:(NSString *)path;
@end

@implementation ID3Tagger

// init/dealloc methods

- (id)init
{
	return [self initWithFilename:nil];
}

- (id)initWithFilename:(NSString *)newFilename
{
	if (self = [super init])
	{
		[self setFilename:newFilename];
	}
	return self;
}

- (void)dealloc
{
	[filename release];
	[fileExtensionsToCodecsDictionary release];
	[super dealloc];
}

// accessor methods

- (void)setFilename:(NSString *)newFilename
{
	[newFilename retain];
	[filename release];
	filename = newFilename;
}

- (void)setFileExtensionsToCodecsDictionary:(NSDictionary *)newFileExtensionsToCodecsDictionary
{
	[newFileExtensionsToCodecsDictionary retain];
	[fileExtensionsToCodecsDictionary release];
	fileExtensionsToCodecsDictionary = newFileExtensionsToCodecsDictionary;
}

/**********************/

/* returns the codec for the current file
 * based on extension
 */
- (codecTypes)codec
{
	NSEnumerator *enumerator = [fileExtensionsToCodecsDictionary keyEnumerator];
	NSString *key;
	
	while (key = [enumerator nextObject]) {
		if ([[filename pathExtension] compare:key options:NSCaseInsensitiveSearch] == NSOrderedSame)
		{
			return [[fileExtensionsToCodecsDictionary objectForKey:key] intValue];
		}
	}
	return CODEC_UNDEFINED;
}

/* read the id3 tag and return it as
 * a track object
 * will pass on to WMATagger if a wma file
 */
- (Track *)readTrack
{	
	// if is WMA file, we can't do anything here, pass on to WMATagger
	if ([self codec] == CODEC_WMA)
	{
		return [WMATagger readTrack:filename];
	}
	
	struct stat sb;
	// check file exists
	if (stat([filename fileSystemRepresentation], &sb) == -1)
		return nil;
	
	// N.B. cannot use NSFileManager as this is not thread safe
		
	Track *track = [[Track alloc] init];
	
	[track setFilesize:sb.st_size];
	[track setCodec:[self codec]];
	[track setFullPath:filename];
	
  if ([self codec] == CODEC_MP3)
	{
		ID3_Tag myTag;
		// ID3V2 is the default - I can't get it to load unicode if not
		size_t id3Size = myTag.Link([filename fileSystemRepresentation], ID3TT_ID3V2);
		if (id3Size == 0)
		{
			// try anything else
			NSLog(@"no ID3V2 tag found, trying something else");
			myTag.Clear();
			myTag.Link([filename fileSystemRepresentation]);
		}
		
		[track setTitle:[self frameContentsFromTag:&myTag frameID:ID3FID_TITLE]];
		[track setAlbum:[self frameContentsFromTag:&myTag frameID:ID3FID_ALBUM]];
		[track setArtist:[self frameContentsFromTag:&myTag frameID:ID3FID_LEADARTIST]];
		
		NSString *genre = [self frameContentsFromTag:&myTag frameID:ID3FID_CONTENTTYPE];
		if (genre != nil && [genre length] > 0)
		{
			// genre may be of the form (ddd) and be an index so look...
			// test for open bracket
			if ([genre characterAtIndex:0] == 40)
			{
				NSRange bracketPos = [genre rangeOfCharacterFromSet:[NSCharacterSet characterSetWithCharactersInString:@")"]];
				
				if (bracketPos.length == 1 && bracketPos.location == [genre length] - 1)
				{
					NSRange numberRange;
					numberRange.length = [genre length] - 2;
					numberRange.location = 1;
					NSString *numberString = [genre substringWithRange:numberRange];
					int number = [numberString intValue];
					if (number >= 0 && number < ID3_NR_OF_V1_GENRES)
					{
						genre = [NSString stringWithCString:ID3_v1_genre_description[number]];
					}
				}
			}
		}
		[track setGenre:genre];
		
		// this may be of the form a/b for track a out of b but parsing to integer takes care of this
		[track setTrackNumber:[self intFrameContentsFromTag:&myTag frameID:ID3FID_TRACKNUM]];
		[track setYear:[self intFrameContentsFromTag:&myTag frameID:ID3FID_YEAR]];
		
		// read the length from the header
		[track setLength:[MP3Length length:filename]];
		
		myTag.Clear();
	
	}
	else if ([self codec] == CODEC_WAV)
	{
		// use filename as title, but strip extension
		NSString *title = [[filename lastPathComponent] stringByDeletingPathExtension];
		[track setTitle:title];
		[track setLength:[self wavFileLength:filename]];
	}
	
	return [track autorelease];
}

/* gets the contents of the frame frameID as an NSString
 */
- (NSString *)frameContentsFromTag:(ID3_Tag *)tag frameID:(ID3_FrameID)frameID
{
	ID3_Frame *frame = tag->Find(frameID, ID3FN_TEXTENC, (luint) ID3TE_UNICODE);
	if (frame != NULL)
	{
		ID3_Field *field = frame->GetField(ID3FN_TEXT);
		if (field != NULL)
		{
			field->SetEncoding(ID3TE_UTF16);
			// docs say len is no. of characters - I think it's number of bytes
			size_t len = field->Size()/2;
			unicode_t *uni = new unicode_t[len + 1];
			field->Get(uni, len + 1);
			// check is terminated
			uni[len] = 0;
			NSString *uniString = [UnicodeWrapper stringFromUTF16:uni];
			delete [] uni;
			return uniString;
		}
	}
	else
	{	
		frame = tag->Find(frameID);
		if (frame != NULL)
		{
			ID3_Field *field = frame->GetField(ID3FN_TEXT);
			if (field != NULL)
			{
				const char *rawText = field->GetRawText();
				NSData *data = nil;
				if (field->Size() == 0)
					return [NSString stringWithString:@""];
				if (rawText[field->Size() - 1] == 0)
					// ignore the NULL terminator
					data = [NSData dataWithBytes:field->GetRawText() length:(field->Size() - 1)];
				else
					data = [NSData dataWithBytes:field->GetRawText() length:field->Size()];
				NSString *asciiString = [[NSString alloc] initWithData:data encoding:NSISOLatin1StringEncoding];
				return [asciiString autorelease];
			}
		}
	}
	return nil;
}

/* returns an integer value from the frame
 */
- (int)intFrameContentsFromTag:(ID3_Tag *)tag frameID:(ID3_FrameID)frameID
{
	return [[self frameContentsFromTag:tag frameID:frameID] intValue];
}

/* write the string contents to the frame frameiD
 */
- (void)setFrameContentsForTag:(ID3_Tag *)tag frameID:(ID3_FrameID)frameID contents:(NSString *)contents
{
	ID3_Frame *frame = new ID3_Frame(frameID);
	if (frame)
	{
		frame->GetField(ID3FN_TEXT)->SetEncoding(ID3TE_UNICODE);
		unicode_t *uni = [UnicodeWrapper UTF16FromString:contents];
		frame->GetField(ID3FN_TEXT)->Set(uni);
		free(uni);
		// is this necessary?
		frame->GetField(ID3FN_TEXTENC)->Set(ID3TE_UTF16);
		tag->AttachFrame(frame);
	}
}

/* write the tag in track to the file
 * in [track fullPath]
 */
- (void)writeTrack:(Track *)track
{
	ID3_Tag myTag;
	myTag.Link([[track fullPath] fileSystemRepresentation], ID3TT_ALL);
	
	ID3_RemoveTitles(&myTag);
	[self setFrameContentsForTag:&myTag frameID:ID3FID_TITLE contents:[track title]];
	ID3_RemoveAlbums(&myTag);
	[self setFrameContentsForTag:&myTag frameID:ID3FID_ALBUM contents:[track album]];
	ID3_RemoveArtists(&myTag);
	[self setFrameContentsForTag:&myTag frameID:ID3FID_LEADARTIST contents:[track artist]];
	ID3_RemoveGenres(&myTag);
	[self setFrameContentsForTag:&myTag frameID:ID3FID_CONTENTTYPE contents:[track genre]];
	
	int nTotal = 0;
	ID3_Frame *frame = myTag.Find(ID3FID_TRACKNUM);
	if (frame != NULL)
	{
		ID3_Field* field = frame->GetField(ID3FN_TEXT);
		if (field != NULL)
		{
			NSString *trackNum = [NSString stringWithCString:field->GetRawText()];
			NSRange slashPos = [trackNum rangeOfCharacterFromSet:[NSCharacterSet characterSetWithCharactersInString:@"/"]];
			if (slashPos.length == 1 && slashPos.location < [trackNum length])
			{
				NSRange numberRange;
				numberRange.length = [trackNum length] - slashPos.location - 1;
				numberRange.location = slashPos.location + 1;
				NSString *numberString = [trackNum substringWithRange:numberRange];
				nTotal = [numberString intValue];
			}
		}
	}

	ID3_RemoveTracks(&myTag);
	ID3_AddTrack(&myTag, [track trackNumber], nTotal, true);
	
	ID3_RemoveYears(&myTag);
	[self setFrameContentsForTag:&myTag frameID:ID3FID_YEAR contents:[NSString stringWithFormat:@"%d", [track year]]];

	myTag.Update();
	myTag.Clear();
}

/* get the length of the wav file at path
 * see http://ccrma.stanford.edu/courses/422/projects/WaveFormat/ for wave format spec
 */
- (unsigned)wavFileLength:(NSString *)path
{
	// open the file handle for the specified path
  NSFileHandle *file = [NSFileHandle fileHandleForReadingAtPath:path];
  if (file == nil)
  {
		NSLog(@"Can not open file :%@", path);
    return 0;
	}
	
	[file seekToFileOffset:28];
	
	NSData *buffer = [file readDataOfLength:4];
	const unsigned char *bytesPerSecondChar = (const unsigned char*)[buffer bytes];
	unsigned int bytesPerSecond = bytesPerSecondChar[0] + (bytesPerSecondChar[1] << 8) + (bytesPerSecondChar[2] << 16) + (bytesPerSecondChar[3] << 24);
	
	[file seekToFileOffset:40];
	
	buffer = [file readDataOfLength:4];
	const unsigned char *bytesLongChar = (const unsigned char*)[buffer bytes];
	unsigned int bytesLong = bytesLongChar[0] + (bytesLongChar[1] << 8) + (bytesLongChar[2] << 16) + (bytesLongChar[3] << 24);
	
	[file closeFile];
	
	if (bytesPerSecond == 0)
	{
		NSLog(@"bytesPerSecond == 0 in file %@", path);
		return 0;
	}
	// round up to be on the safe side
	if (bytesLong % bytesPerSecond == 0)
		return (bytesLong / bytesPerSecond);
	else
		return (bytesLong / bytesPerSecond) + 1;
}

@end
