//
//  NJB.m
//  XNJB
//
//  Created by Richard Low on Wed Jul 21 2004.
//

// todo: can we find out how many tracks there are before we download for progress bar & set capacity in NSMutableArray?

/* An obj-c wrapper for libnjb
 * most methods return an NJBTransactionResult
 * to indicate success/failure
 */

#import "NJB.h"
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#import "Track.h"
#include "string.h"
#include "njbusb.h"
#import "defs.h"

// declared so can be accessed from C function progress
// set to be the outlet statusDisplayer in awakeFromNib
StatusDisplayer *statusDisplayerGlobal;

// declare the private methods
@interface NJB (PrivateAPI)
- (void)updateDiskSpace;
- (njb_songid_t *)songidStructFromTrack:(Track *)track;
- (NSString *)njbErrorString;
- (NSString *)rateString:(double) rate;
@end

@implementation NJB

// init/dealloc methods

- (id)init
{
	if (self = [super init])
	{
		[self setDebug:0];
	}
	return self;
}

- (void)dealloc
{
	[statusDisplayerGlobal release];
	[cachedTrackList release];
	[super dealloc];
}

// accessor methods

- (void)setDebug:(unsigned)newDebug
{
	// see libnjb.h for the Debug flags
	debug = newDebug;
}

- (statusTypes)status
{
	return status;
}

- (void)setStatus:(statusTypes)newStatus
{
	status = newStatus;
	[statusDisplayer setStatus:status];
}

- (BOOL)isConnected
{
	return connected;
}

/**********************/

- (void)awakeFromNib
{
	statusDisplayerGlobal = [statusDisplayer retain];
}

- (NSString *)njbErrorString
{
	const char *sp;
	NSMutableString *errorString = [[NSMutableString alloc] init];
	NJB_Error_Reset_Geterror(njb);
  while ((sp = NJB_Error_Geterror(njb)) != NULL)
	{
		[errorString appendString:[NSString stringWithFormat:@"%s, ", sp]];
  }
  njb_error_clear(njb);
	
	return [errorString autorelease];
}

/* connect to the NJB
 */
- (NJBTransactionResult *)connect
{	
	njb_t njbs[NJB_MAX_DEVICES];
	int n;
	
	if (debug)
		NJB_Set_Debug(debug);
	
	if (NJB_Discover(njbs, 0, &n) == -1)
	{
		fprintf(stderr, "could not locate any jukeboxes\n");
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:@"Could not discover any jukeboxes"] autorelease];
	}
	
	if (n == 0)
	{
		[self setStatus:STATUS_NO_NJB];
		connected = NO;
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:@"Could not locate any jukeboxes"] autorelease];
	}
	
	njb = &njbs[0];
	if (NJB_Open(njb) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		[self setStatus:STATUS_COULD_NOT_OPEN];
		connected = NO;
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	
	if (NJB_Capture(njb) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		[self setStatus:STATUS_COULD_NOT_CAPTURE];
		[self disconnect];
		connected = NO;
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	
	[self setStatus:STATUS_CONNECTED];
	connected = YES;
	
	[self updateDiskSpace];
	
	// enable UTF8
	NJB_Set_Unicode(NJB_UC_UTF8);
	
	// send notification
	[statusDisplayer postNotificationName:NOTIFICATION_CONNECTED object:self];
	
	// check we have reset the cache, should have been done on disconnect
	[cachedTrackList release];
	cachedTrackList = nil;
	
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

/* disconnect from the NJB
 */
- (void)disconnect
{
	if ([self isConnected])
	{
		NSLog(@"Releasing...");
		NJB_Release(njb);
		NSLog(@"Disconnecting...");
		NJB_Close(njb);
		connected = NO;
		[self setStatus:STATUS_DISCONNECTED];
		
		// lose the cache
		[cachedTrackList release];
		cachedTrackList = nil;
		
		// send notification
		[statusDisplayer postNotificationName:NOTIFICATION_DISCONNECTED object:self];
	}
}

- (NSString *)productName
{
	return [NSString stringWithCString:njb->njbid->productName];
}

- (NSString *)ownerString
{
	char *ownerString = NJB_Get_Owner_String(njb);
	if (ownerString == NULL)
		return nil;
	NSString *nsOwnerString = [[NSString alloc] initWithUTF8String:ownerString];
	free(ownerString);
	return [nsOwnerString autorelease];
}

- (MyMutableArray *)tracks
{
	if (![self isConnected])
		return nil;
	if (cachedTrackList)
	{
		return [self cachedTrackList];
	}
	cachedTrackList = [[MyMutableArray alloc] init];
	NJB_Reset_Get_Track_Tag(njb);
	njb_songid_t *songtag;
	njb_songid_frame_t *frame;
	while ((songtag = NJB_Get_Track_Tag(njb))) {
		Track *track = [[Track alloc] init];
		
		frame = NJB_Songid_Findframe(songtag, FR_TITLE);
		if (frame != NULL && frame->data.strval != NULL)
			[track setTitle:[NSString stringWithUTF8String:frame->data.strval]];
		frame = NJB_Songid_Findframe(songtag, FR_ALBUM);
		if (frame != NULL && frame->data.strval != NULL)
			[track setAlbum:[NSString stringWithUTF8String:frame->data.strval]];
		frame = NJB_Songid_Findframe(songtag, FR_ARTIST);
		if (frame != NULL && frame->data.strval != NULL)
			[track setArtist:[NSString stringWithUTF8String:frame->data.strval]];
		frame = NJB_Songid_Findframe(songtag, FR_GENRE);
		if (frame != NULL && frame->data.strval != NULL)
			[track setGenre:[NSString stringWithUTF8String:frame->data.strval]];
		// this is not used: we don't get extended track info from njb3
		// njb1 gets it but ignored
		frame = NJB_Songid_Findframe(songtag, FR_FNAME);
		if (frame != NULL && frame->data.strval != NULL)
			[track setFilename:[NSString stringWithUTF8String:frame->data.strval]];
		frame = NJB_Songid_Findframe(songtag, FR_SIZE);
		if (frame != NULL)
		{
			if (frame->type == NJB_TYPE_UINT16)
				[track setFilesize:frame->data.u_int16_val];
			else
				[track setFilesize:frame->data.u_int32_val];
		}
		frame = NJB_Songid_Findframe(songtag, FR_LENGTH);
		if (frame != NULL)
		{
			if (frame->type == NJB_TYPE_UINT16)
				[track setLength:frame->data.u_int16_val];
			else
				[track setLength:frame->data.u_int32_val];
		}
		frame = NJB_Songid_Findframe(songtag, FR_TRACK);
		if (frame != NULL)
		{
			if (frame->type == NJB_TYPE_UINT16)
				[track setTrackNumber:frame->data.u_int16_val];
			else if (frame->type == NJB_TYPE_UINT32)
				[track setTrackNumber:frame->data.u_int32_val];
			// in case it's a string
			else if (frame->type == NJB_TYPE_STRING)
			{
				NSString *trackNumber = [NSString stringWithCString:frame->data.strval];
				[track setTrackNumber:(unsigned)[trackNumber intValue]];
			}
			else
				NSLog(@"type not expected for FR_TRACK field %d", frame->type);
		}
		frame = NJB_Songid_Findframe(songtag, FR_CODEC);
		if (frame != NULL && frame->data.strval == NULL)
			[track setCodec:CODEC_UNDEFINED];
		else
		{
			if (strcmp(frame->data.strval, NJB_CODEC_MP3) == 0)
				[track setCodec:CODEC_MP3];
			else if (strcmp(frame->data.strval, NJB_CODEC_WMA) == 0)
				[track setCodec:CODEC_WMA];
			else if (strcmp(frame->data.strval, NJB_CODEC_WAV) == 0)
				[track setCodec:CODEC_WAV];
			else
				[track setCodec:CODEC_UNDEFINED];
		}
		
		[track setTrackID:songtag->trid];
		
		frame = NJB_Songid_Findframe(songtag, FR_YEAR);
		if (frame != NULL)
		{
			if (frame->type == NJB_TYPE_UINT16)
				[track setYear:frame->data.u_int16_val];
			else if (frame->type == NJB_TYPE_UINT32)
				[track setYear:frame->data.u_int32_val];
			// strings on NJB1
			else if (frame->type == NJB_TYPE_STRING)
			{
				NSString *year = [NSString stringWithCString:frame->data.strval];
				[track setYear:(unsigned)[year intValue]];
			}
			else
				NSLog(@"type not expected for FR_YEAR field %d", frame->type);
		}
		else
			[track setYear:0];
		
		[cachedTrackList addObject:track];
		[track release];
		
		NJB_Songid_Destroy(songtag);
	}
	return [self cachedTrackList];
}

/* uploads the track and sets the track id
 */
- (NJBTransactionResult *)uploadTrack:(Track *)track
{
	NSDate *date = [NSDate date];
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:@"Not connected"] autorelease];
	NSLog(@"Uploading track: %@", [track fullPath]);
	[self setStatus:STATUS_UPLOADING_TRACK];
	int trackid;
	njb_songid_t *songid = [self songidStructFromTrack:track];
	if (NJB_Send_Track(njb, [[track fullPath] fileSystemRepresentation], songid, progress, NULL, &trackid) == -1 ) {
		NSString *error = [self njbErrorString];
		NSLog(error);
		NJB_Songid_Destroy(songid);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	} else {
		[self updateDiskSpace];
		[track setTrackID:trackid];
		
		double time = -[date timeIntervalSinceNow];
		// rate in Mbps
		double rate = ((double)[track filesize] / time) * 8.0 / (1024.0 * 1024.0);
		NSLog(@"Took: %f seconds, speed %f Mbps", time, rate);
		
		NJB_Songid_Destroy(songid);
		// update my cache
		[cachedTrackList addObject:track];
		// tell others the track list has changed
		[statusDisplayer postNotificationName:NOTIFICATION_TRACK_LIST_MODIFIED object:self];
		
		return [[[NJBTransactionResult alloc] initWithSuccess:YES
																						 resultString:[NSString stringWithFormat:@"Speed %@ Mbps", [self rateString:rate]]] autorelease];
	}
}

- (NJBTransactionResult *)deleteTrack:(Track *)track
{
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	if (NJB_Delete_Track(njb, [track trackID]) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	else
	{
		[self updateDiskSpace];
		// update my cache
		[cachedTrackList removeObject:track];
		// tell others the track list has changed
		[statusDisplayer postNotificationName:NOTIFICATION_TRACK_LIST_MODIFIED object:self];
		
		return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
	}
}

- (NJBTransactionResult *)deleteFile:(NSNumber *)fileID
{
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	if (NJB_Delete_Datafile(njb, [fileID unsignedIntValue]) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	else
	{
		[self updateDiskSpace];
		return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
	}
}

- (NJBTransactionResult *)downloadTrack:(Track *)track
{
	NSDate *date = [NSDate date];
	if (NJB_Get_Track(njb, [track trackID], [track filesize], [[track fullPath] fileSystemRepresentation], progress, NULL) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	double time = -[date timeIntervalSinceNow];
	// rate in Mbps
	double rate = ((double)[track filesize] / time) * 8.0 / (1024.0 * 1024.0);
	NSLog(@"Took: %f seconds, speed %f Mbps", time, rate);
	
	return [[[NJBTransactionResult alloc] initWithSuccess:YES
																					 resultString:[NSString stringWithFormat:@"Speed %@ Mbps", [self rateString:rate]]] autorelease];
}

// todo: does NJB_Replace_Track_Tag ever fail?
- (NJBTransactionResult *)changeTrackTagTo:(Track *)newTrack from:(Track *)oldTrack
{
	njb_songid_t *songid = [self songidStructFromTrack:newTrack];
	
	if (NJB_Replace_Track_Tag(njb, [newTrack trackID], songid) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		NJB_Songid_Destroy(songid);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	else
	{
		[self updateDiskSpace];
		NJB_Songid_Destroy(songid);
		
		// update my cache
		[cachedTrackList replaceObject:oldTrack withObject:newTrack];
		
		// tell others the track list has changed
		[statusDisplayer postNotificationName:NOTIFICATION_TRACK_LIST_MODIFIED object:self];
		
		return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
	}
}

- (MyMutableArray *)dataFiles
{
	if (![self isConnected])
		return nil;
	MyMutableArray *dataFiles = [[MyMutableArray alloc] init];
	NJB_Reset_Get_Datafile_Tag(njb);
	njb_datafile_t *filetag;
	while ((filetag = NJB_Get_Datafile_Tag (njb)))
	{
		DataFile *dataFile = [[DataFile alloc] init];
		const char *str = filetag->filename;
		if (str != NULL)
			[dataFile setFilename:[NSString stringWithUTF8String:str]];
		[dataFile setSize:filetag->filesize];
//		[dataFile setTimestampSince1970:filetag->timestamp];
		[dataFile setFileID:filetag->dfid];
		
		[dataFiles addObject:dataFile];
		[dataFile release];
		
  	 NJB_Datafile_Destroy(filetag);
	}
	return [dataFiles autorelease];
}

- (NJBTransactionResult *)downloadFile:(DataFile *)dataFile
{
	NSDate *date = [NSDate date];
	if (NJB_Get_File(njb, [dataFile fileID], [dataFile size], [[dataFile fullPath] fileSystemRepresentation], progress, NULL) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	double time = -[date timeIntervalSinceNow];
	// rate in Mbps
	double rate = ((double)[dataFile size] / time) * 8.0 / (1024.0 * 1024.0);
	NSLog(@"Took: %f seconds, speed %f Mbps", time, rate);
		
	return [[[NJBTransactionResult alloc] initWithSuccess:YES
																					 resultString:[NSString stringWithFormat:@"Speed %@ Mbps", [self rateString:rate]]] autorelease];
}

- (NJBTransactionResult *)uploadFile:(DataFile *)dataFile
{
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	NSLog(@"Uploading file: %@", [dataFile fullPath]);
	NSDate *date = [NSDate date];
	[self setStatus:STATUS_UPLOADING_FILE];
	int fileid;
	int NJB_Send_File (njb_t *njb, const char *path, const char *name, const char *folder,
										 NJB_Xfer_Callback *callback, void *data, u_int32_t *fileid);
	if (NJB_Send_File(njb, [[dataFile fullPath] fileSystemRepresentation], [[[dataFile fullPath] lastPathComponent] UTF8String], NULL, progress, NULL, &fileid) == -1 ) {
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	} else {
		[dataFile setFileID:fileid];
		[self updateDiskSpace];
		
		double time = -[date timeIntervalSinceNow];
		// rate in Mbps
		double rate = ((double)[dataFile size] / time) * 8.0 / (1024.0 * 1024.0);
		NSLog(@"Took: %f seconds, speed %f Mbps", time, rate);
		return [[[NJBTransactionResult alloc] initWithSuccess:YES
																						 resultString:[NSString stringWithFormat:@"Speed %@ Mbps", [self rateString:rate]]] autorelease];
	}
}

- (unsigned long long)totalDiskSpace
{
	return totalDiskSpace;
}

- (unsigned long long)freeDiskSpace
{
	return freeDiskSpace;
}

- (void)updateDiskSpace
{
	NJB_Get_Disk_Usage(njb, &totalDiskSpace, &freeDiskSpace);
	[statusDisplayer updateDiskSpace:totalDiskSpace withFreeSpace:freeDiskSpace];
}

- (NSCalendarDate *)jukeboxTime
{
	njb_time_t *time = NJB_Get_Time(njb);
	if (time == NULL)
		return nil;
	
	// assume njb time is our timezone
	NSCalendarDate *date = [NSCalendarDate dateWithYear:time->year month:time->month day:time->day hour:time->hours minute:time->minutes second:time->seconds
																						 timeZone:[NSTimeZone localTimeZone]];
	return date;
}

- (BOOL)isProtocol3Device
{
	return (PROTOCOL3_DEVICE(njb->device_type));
}

- (NJBTransactionResult *)setOwnerString:(NSString *)owner
{
	if (NJB_Set_Owner_String (njb, [owner UTF8String]) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)setBitmap:(NSString *)bitmapPath
{
	if (NJB_Set_Bitmap (njb, [bitmapPath UTF8String]) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)setTime:(NSNumber *)timeIntervalSinceNow
{
	njb_time_t time;
	
	NSTimeInterval jukeboxTimeInterval = [timeIntervalSinceNow doubleValue];
	NSCalendarDate *date = [NSCalendarDate dateWithTimeIntervalSinceNow:jukeboxTimeInterval];
	
	time.year = [date yearOfCommonEra];
	time.month = [date monthOfYear];
	time.day = [date dayOfMonth];
	time.weekday = [date dayOfWeek];
	time.hours = [date hourOfDay];
	time.minutes = [date minuteOfHour];
	time.seconds = [date secondOfMinute];
	
	if (NJB_Set_Time(njb, &time) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (MyMutableArray *)playlists
{
	if (![self isConnected])
		return nil;
	
	MyMutableArray *playlists = [[MyMutableArray alloc] init];
	NJB_Reset_Get_Playlist(njb);
	njb_playlist_t *playlist;
	while ((playlist = NJB_Get_Playlist(njb)))
	{
		Playlist *newPlaylist = [[Playlist alloc] init];
				
		const char *str = playlist->name;
		if (str != NULL)
			[newPlaylist setName:[NSString stringWithUTF8String:str]];
		[newPlaylist setPlaylistID:playlist->plid];
		
		njb_playlist_track_t *track;
		NJB_Playlist_Reset_Gettrack(playlist);
		while ((track = NJB_Playlist_Gettrack(playlist)))
		{
			Track *newTrack = [[Track alloc] init];
			[newTrack setTrackID:track->trackid];
			[newPlaylist addTrack:newTrack];
			[newTrack release];
		}
		
		NJB_Playlist_Destroy(playlist);
		
		[newPlaylist setState:NJB_PL_UNCHANGED];
		
		[playlists addObject:newPlaylist];
		[newPlaylist release];
	}
	return [playlists autorelease];
}

/* updates the playlist on the NJB
 * may add a new playlist even if playlist state is not new
 * so the playlist ID may change.
 */
- (NJBTransactionResult *)updatePlaylist:(Playlist *)playlist
{
	[playlist lock];
	// don't do anything if it's unchanged
	if ([playlist state] == NJB_PL_UNCHANGED)
	{
		[playlist unlock];
		return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
	}
	
	// create new njb_playlist_t object
	njb_playlist_t *pl = NJB_Playlist_New();
	NJB_Playlist_Set_Name(pl, [[playlist name] UTF8String]);
	pl->plid = [playlist playlistID];
	NSEnumerator *enumerator = [[playlist tracks] objectEnumerator];
	Track *currentTrack;
	while (currentTrack = [enumerator nextObject])
	{
		njb_playlist_track_t *newTrack = NJB_Playlist_Track_New([currentTrack trackID]);
		NJB_Playlist_Addtrack(pl, newTrack, NJB_PL_END);
	}
	pl->_state = [playlist state];
	// set this now as as far as the caller is concerned the NJB is updated
	[playlist setState:NJB_PL_UNCHANGED];
	[playlist unlock];
	if (NJB_Update_Playlist(njb, pl) == -1)
	{
		NJB_Playlist_Destroy(pl);
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	[playlist lock];
	// update the ID as this is changed when tracks added/removed as a new playlist is created
	[playlist setPlaylistID:pl->plid];
	[playlist unlock];
	NJB_Playlist_Destroy(pl);
	
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)deletePlaylist:(Playlist *)playlist
{
	[playlist lock];
	unsigned plid = [playlist playlistID];
	[playlist unlock];
	if (NJB_Delete_Playlist(njb, plid) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

/* called by the file/track transfer functions of libnjb
 * to tell us the progress
 */
int progress(u_int64_t sent, u_int64_t total, const char* buf, unsigned len, void *data)
{
	double percent = (double)sent*100.0/(double)total;
	[statusDisplayerGlobal updateTaskProgress:percent];
	
	return 0;
}

- (njb_songid_t *)songidStructFromTrack:(Track *)track
{
	njb_songid_t *songid;
	njb_songid_frame_t *frame;
	
	songid = NJB_Songid_New();
	if ([track njbCodec])
	{
		frame = NJB_Songid_Frame_New_Codec([[track njbCodec] UTF8String]);
		NJB_Songid_Addframe(songid, frame);
	}
	// only add file size if non zero
	if ([track filesize] != 0)
	{
		frame = NJB_Songid_Frame_New_Filesize([track filesize]);
		NJB_Songid_Addframe(songid, frame);
	}
	if ([track title])
	{
		frame = NJB_Songid_Frame_New_Title([[track title] UTF8String]);
		NJB_Songid_Addframe(songid, frame);
	}
	if ([track album])
	{
		frame = NJB_Songid_Frame_New_Album([[track album] UTF8String]);
		NJB_Songid_Addframe(songid, frame);
	}
	if ([track artist])
	{
		frame = NJB_Songid_Frame_New_Artist([[track artist] UTF8String]);
		NJB_Songid_Addframe(songid, frame);
	}
	if ([track genre])
	{
		frame = NJB_Songid_Frame_New_Genre([[track genre] UTF8String]);
		NJB_Songid_Addframe(songid, frame);
	}
	frame = NJB_Songid_Frame_New_Year([track year]);
	NJB_Songid_Addframe(songid, frame);
	frame = NJB_Songid_Frame_New_Tracknum([track trackNumber]);
	NJB_Songid_Addframe(songid, frame);
	frame = NJB_Songid_Frame_New_Length([track length]);
	NJB_Songid_Addframe(songid, frame);
	if ([track fullPath] && [[track fullPath] length] > 0)
	{
		frame = NJB_Songid_Frame_New_Filename([[track fullPath] fileSystemRepresentation]);
		NJB_Songid_Addframe(songid, frame);
	}
	
	return songid;
}

- (MyMutableArray *)cachedTrackList
{
	if (cachedTrackList)
	{
		MyMutableArray *tracks = [[MyMutableArray alloc] initWithMyMutableArray:cachedTrackList];
		return [tracks autorelease];
	}
	else
		return nil;
}

/* returns a string rounded to 1dp for
 * the rate given
 */
- (NSString *)rateString:(double) rate
{
	int rate10 = rate * 10;
	if (rate - (double)rate10/10.0 >= 0.05)
		rate10++;
	// the two digits
	int rateA = rate10/10;
	int rateB = rate10-rateA*10;
	return [NSString stringWithFormat:@"%d.%d", rateA, rateB];
}

@end
