//
//  PlaylistsTab.m
//  XNJB
//

/* the playlists tab. Supports creating, removing
 * and modifying playlists.
 */

#import "PlaylistsTab.h"
#import "Playlist.h"
#import "defs.h"

// these correspond to the identifiers in the NIB file
#define COLUMN_TITLE @"Title"
#define COLUMN_ARTIST @"Artist"
#define COLUMN_ALBUM @"Album"
#define COLUMN_GENRE @"Genre"
#define COLUMN_LENGTH @"Length"
#define COLUMN_TRACK_NO @"Track No"
#define COLUMN_CODEC @"Codec"
#define COLUMN_FILESIZE @"Filesize"
#define COLUMN_YEAR @"Year"

// add - (Track *)trackWithID:(unsigned)trackID to MyMutableArray
@interface MyMutableArray (trackWithID)
- (Track *)trackWithID:(unsigned)trackID;
@end

// adds to MyMutableArray a method to return the track with ID trackID
@implementation MyMutableArray (trackWithID)

/* finds a track which has ID trackID
 * assumes all IDs are unique so returns first item only
 * returns nil if none found
 */
- (Track *)trackWithID:(unsigned)trackID
{
	NSEnumerator *enumerator = [objects objectEnumerator];
	Track *currentTrack;
	while (currentTrack = [enumerator nextObject])
	{
		if ([currentTrack trackID] == trackID)
			return currentTrack;
	}
	return nil;
}

@end

// declare the private methods
@interface PlaylistsTab (PrivateAPI)
- (void)fillPlaylistTrackInfo;
- (void)allowNewPlaylist;
- (void)swapPlaylistTrackWithPreviousAtIndex:(unsigned)index forPlaylist:(Playlist *)playlist;
- (void)swapPlaylistTrackWithNextAtIndex:(unsigned)index forPlaylist:(Playlist *)playlist;
@end

@implementation PlaylistsTab

// init/dealloc methods

- (void)dealloc
{
	[playlists release];
	[super dealloc];
}

/**********************/

- (void)onConnectAndActive
{
	[super onConnectAndActive];
	[self loadTracks];
	[self loadPlaylists];
}

- (void)activate
{
	// this must be called first to run onConnectAndActive so we can download the tracks there
	[super activate];
	if (!trackListUpToDate)
	{
		[fullItemArray release];
		fullItemArray = nil;
		[truncatedItemArray release];
		truncatedItemArray = nil;
		[njbTable reloadData];
		[self loadTracks];
	}
}

- (void)awakeFromNib
{
	[super awakeFromNib];
	
	[njbTable setAutosaveName:PREF_KEY_PLAYLISTS_TAB_TABLE];
	[njbTable setAutosaveTableColumns:YES];
	[playlistsTable setAutosaveName:PREF_KEY_PLAYLISTS_TAB_PLAYLISTS_TABLE];
	[playlistsTable setAutosaveTableColumns:YES];
	[playlistsTrackTable setAutosaveName:PREF_KEY_PLAYLISTS_TAB_PLAYLISTS_TRACK_TABLE];
	[playlistsTrackTable setAutosaveTableColumns:YES];
}

- (id)tableView:(NSTableView *)aTableView
   objectValueForTableColumn:(NSTableColumn *)aTableColumn
						row:(int)rowIndex
{	
	if (aTableView == njbTable)
	{
		NSString *ident = [aTableColumn identifier];
		Track *track = (Track *)[[self itemArrayDisplaying] objectAtIndex:rowIndex];
		if ([ident isEqual:COLUMN_TITLE])
			return [track title];
		else if ([ident isEqual:COLUMN_ALBUM])
			return [track album];
		else if ([ident isEqual:COLUMN_ARTIST])
			return [track artist];
		else if ([ident isEqual:COLUMN_GENRE])
			return [track genre];
		else if ([ident isEqual:COLUMN_LENGTH])
			return [track lengthString];
		else if ([ident isEqual:COLUMN_TRACK_NO])
			return [NSString stringWithFormat:@"%d", [track trackNumber]];
		else if ([ident isEqual:COLUMN_CODEC])
			return [track codecString];
		else if ([ident isEqual:COLUMN_FILESIZE])
			return [track filesizeString];
		else if ([ident isEqual:COLUMN_YEAR])
			return [NSString stringWithFormat:@"%d", [track year]];
		else
		{
			NSLog(@"Invalid column tag in PlaylistsTab for njbTable");
			return nil;
		}
	}
	else if (aTableView == playlistsTable)
	{
		Playlist *playlist = (Playlist *)[playlists objectAtIndex:rowIndex];
	  return [playlist name];
	}
	else if (aTableView == playlistsTrackTable)
	{
		if (tracksInCurrentPlaylist != nil)
		{
			NSString *ident = [aTableColumn identifier];
			Track *track = (Track *)[tracksInCurrentPlaylist objectAtIndex:rowIndex];
			if ([ident isEqual:COLUMN_TITLE])
				return [track title];
			else if ([ident isEqual:COLUMN_ARTIST])
				return [track artist];
			else
			{
				NSLog(@"Invalid column tag in PlaylistsTab for playlistsTrackTable");
				return nil;
			}
		}
		else
			return nil;
	}
	NSLog(@"Unknown table %@ in PlaylistsTab tableView:objectValueForTableColumn:row:!", aTableView);
	return nil;
}

- (int)numberOfRowsInTableView:(NSTableView *)aTableView
{
	if (aTableView == njbTable)
		return [super numberOfRowsInTableView:aTableView];
	else if (aTableView == playlistsTable)
		return [playlists count];
	else if (aTableView == playlistsTrackTable)
	{
		if (tracksInCurrentPlaylist)
			return [tracksInCurrentPlaylist count];
		else
			return 0;
	}
	// may get here if any of the table pointers above haven't been set yet
	NSLog(@"Unknown table %@ in PlaylistsTab numberOfRowsInTableView:", aTableView);
	return 0;
}

- (SEL)sortSelectorForColumn:(NSString *)columnID
{
	if ([columnID isEqual:COLUMN_TITLE])
		return [Track compareByTitleSelector];
	else if ([columnID isEqual:COLUMN_ARTIST])
		return [Track compareByArtistSelector];
	else if ([columnID isEqual:COLUMN_ALBUM])
		return [Track compareByAlbumSelector];
	else if ([columnID isEqual:COLUMN_GENRE])
		return [Track compareByGenreSelector];
	else if ([columnID isEqual:COLUMN_LENGTH])
		return [Track compareByLengthSelector];
	else if ([columnID isEqual:COLUMN_TRACK_NO])
		return [Track compareByTrackNumberSelector];
	else if ([columnID isEqual:COLUMN_CODEC])
		return [Track compareByCodecSelector];
	else if ([columnID isEqual:COLUMN_FILESIZE])
		return [Track compareByFilesizeSelector];
	else if ([columnID isEqual:COLUMN_YEAR])
		return [Track compareByYearSelector];
	else
	{
		NSLog(@"Invalid column %@ in sortSelectorForColumn!", columnID);
		return nil;
	}
}

- (void)searchNJBTable
{	
	NSString *searchString = [njbSearchField stringValue];
	// release old track list
	[truncatedItemArray release];
	
	if ([searchString length] == 0)
	{
		truncatedItemArray = nil;
		[njbTable reloadData];
		return;
	}
	
	truncatedItemArray = [fullItemArray searchFor:searchString];
	[truncatedItemArray retain];
	
	[njbTable reloadData];
}

/* add downloadTrackList to the njb queue
 */
- (void)loadTracks
{
	NJBQueueItem *clearSortingArrows = [[NJBQueueItem alloc] initWithTarget:self
																														 withSelector:@selector(clearSortingArrows)
																															 withObject:nil
																											withRunInMainThread:YES];
	[clearSortingArrows setDisplayStatus:NO];
	
	NJBQueueItem *getTracks = [[NJBQueueItem alloc] initWithTarget:self withSelector:@selector(downloadTrackList)];
	[getTracks setStatus:STATUS_DOWNLOADING_TRACK_LIST];
	
	NJBQueueItem *updateTable = [[NJBQueueItem alloc] initWithTarget:self
																											withSelector:@selector(searchNJBTable)
																												withObject:nil
																							 withRunInMainThread:YES];
	[updateTable setDisplayStatus:NO];
	
	NSString *description = @"Getting track list";
	[self addToQueue:getTracks withSubItems:[NSArray arrayWithObjects:clearSortingArrows, updateTable, nil] withDescription:description];
	[getTracks release];
	[clearSortingArrows release];
	[updateTable release];
	// it's not actually yet, but will be in a while, don't want to go copying anything
	trackListUpToDate = YES;
}

/* this is called from the queue consumer 
 * to get the tracks off the NJB and put them in fullItemArray
 * will run in worker thread
 */
- (NJBTransactionResult *)downloadTrackList
{
	NSLog(@"Getting tracks");
	[fullItemArray release];
	fullItemArray = [theNJB tracks];
	
	NJBTransactionResult *result = [[NJBTransactionResult alloc] initWithSuccess:(fullItemArray != nil)];
	
	[fullItemArray retain];
	
	trackListUpToDate = YES;
	
	return [result autorelease];
}

/* add downloadPlaylists to the njb queue
 */
- (void)loadPlaylists
{
	NJBQueueItem *getPlaylists = [[NJBQueueItem alloc] initWithTarget:self withSelector:@selector(downloadPlaylists)];
	[getPlaylists setStatus:STATUS_DOWNLOADING_PLAYLISTS];
	
	NJBQueueItem *fillTrackInfo = [[NJBQueueItem alloc] initWithTarget:self withSelector:@selector(fillPlaylistTrackInfo)
																													withObject:nil withRunInMainThread:YES];
	[fillTrackInfo setDisplayStatus:NO];
	
	NJBQueueItem *updateTable = [[NJBQueueItem alloc] initWithTarget:playlistsTable
																											withSelector:@selector(reloadData)
																												withObject:nil
																							 withRunInMainThread:YES];
	[updateTable setDisplayStatus:NO];
	
	NJBQueueItem *allowNewPlaylist = [[NJBQueueItem alloc] initWithTarget:self
																											withSelector:@selector(allowNewPlaylist)
																												withObject:nil
																							 withRunInMainThread:YES];
	[allowNewPlaylist setDisplayStatus:NO];
	
	NSString *description = @"Getting playlists";
	[self addToQueue:getPlaylists withSubItems:[NSArray arrayWithObjects:fillTrackInfo, updateTable, allowNewPlaylist, nil] withDescription:description];
	[getPlaylists release];
	[fillTrackInfo release];
	[updateTable release];
	[allowNewPlaylist release];
}

/* this is called from the queue consumer 
 * to get the playlists off the NJB and put them in playlists
 * will run in worker thread
 */
- (NJBTransactionResult *)downloadPlaylists
{
	NSLog(@"Getting playlists");
	[playlists release];
	
	playlists = [theNJB playlists];
		
	NJBTransactionResult *result = [[NJBTransactionResult alloc] initWithSuccess:(playlists != nil)];
		
	[playlists retain];
		
	return [result autorelease];
}

/* maps track ids to full tag information from
 * fullItemArray
 */
- (void)fillPlaylistTrackInfo
{
	NSEnumerator *enumerator = [playlists objectEnumerator];
	Playlist *currentPlaylist;
	while (currentPlaylist = [enumerator nextObject])
	{
		[currentPlaylist fillTrackInfoFromList:fullItemArray];
	}
}

/* enable the new playlist button
 */
- (void)allowNewPlaylist
{
	[newPlaylistButton setEnabled:YES];
}

/* called by MusicTab when it has modified the track list
 */
- (void)updateTracks
{
	if (fullItemArray != nil)
	{
		// this doesn't resort, could be annoying if someone is browsing in playlists tab
		// and it gets resorted
		[self clearSortingArrows];
		[fullItemArray release];
		fullItemArray = [[MyMutableArray alloc] initWithMyMutableArray:[musicTab fullItemArray]];
		[self searchNJBTable];
	}
}

/* update the playlist on the jukebox
 */
- (void)updateJukeboxPlaylist:(Playlist *)playlist
{
	NJBQueueItem *updatePlaylist = [[NJBQueueItem alloc] initWithTarget:theNJB withSelector:@selector(updatePlaylist:) withObject:playlist];
	[updatePlaylist setStatus:STATUS_UPDATING_PLAYLIST];
	
	NSString *description = [NSString stringWithFormat:@"Updating playlist %@", [playlist name]];
	[self addToQueue:updatePlaylist withSubItems:nil withDescription:description];
	[updatePlaylist release];
}

/* delete playlist playlist from the NJB
 */
- (void)deleteJukeboxPlaylist:(Playlist *)playlist
{	
	NJBQueueItem *deletePlaylist = [[NJBQueueItem alloc] initWithTarget:theNJB withSelector:@selector(deletePlaylist:) withObject:playlist];
	[deletePlaylist setStatus:STATUS_DELETING_PLAYLIST];
	
	NSString *description = [NSString stringWithFormat:@"Deleting playlist %@", [playlist name]];
	[self addToQueue:deletePlaylist withSubItems:nil withDescription:description];
	[deletePlaylist release];
}

/* add the selected tracks in njbTable to the selected
 * playlist
 */
- (IBAction)addToPlaylist:(id)sender
{
	NSEnumerator *trackEnumerator = [njbTable selectedRowEnumerator];
	NSNumber *index;
	Playlist *playlist = (Playlist *)[playlists objectAtIndex:[playlistsTable selectedRow]];
	[playlist lock];
	while (index = [trackEnumerator nextObject])
	{
		// this is added at the end so don't worry about indexes changing
		Track *track = (Track *)[[self itemArrayDisplaying] objectAtIndex:[index unsignedIntValue]];
		[playlist addTrack:track];
	}
	[playlist unlock];
	// update the jukebox
	NSLog(@"Updating playlist from addToPlaylist:");
	[self updateJukeboxPlaylist:playlist];
	[playlistsTrackTable reloadData];
}

/* delete the selected tracks in njbTable from the selected
 * playlist
 */
- (IBAction)deleteFromPlaylist:(id)sender
{
	NSEnumerator *trackEnumerator = [playlistsTrackTable selectedRowEnumerator];
	NSNumber *index;
	Playlist *playlist = (Playlist *)[playlists objectAtIndex:[playlistsTable selectedRow]];
	NSMutableArray *tracksToDelete = [[NSMutableArray alloc] initWithCapacity:[playlistsTrackTable numberOfSelectedRows]];
	MyMutableArray *tracks = [playlist tracks];
	while (index = [trackEnumerator nextObject])
	{
		[tracksToDelete addObject:[tracks objectAtIndex:[index unsignedIntValue]]];
	}
	[playlist lock];
	[playlist deleteTracks:tracksToDelete];
	[playlist unlock];
	// update the jukebox
	NSLog(@"Updating playlist from deleteFromPlaylist:");
	[self updateJukeboxPlaylist:playlist];
	[tracksToDelete release];
	[playlistsTrackTable reloadData];
}

/* create a new playlist
 */
- (IBAction)newPlaylist:(id)sender
{
	Playlist *playlist = [[Playlist alloc] init];
	// this adds an object to the end
	[playlists addObject:playlist];
	[playlist release];
	[playlistsTable reloadData];
	// so select the last row for editing
	int row = ([playlists count] - 1);
	[playlistsTable selectRow:row byExtendingSelection:NO];
	[playlistsTable editColumn:0 row:row withEvent:nil select:YES];
}

/* delete the selected playlist
 */
- (IBAction)deletePlaylist:(id)sender
{
	int result = NSRunAlertPanel(@"Playlist Deletion", @"Are you sure you want to delete the selected playlist off the NJB?", @"No", @"Yes", nil);
	if (result == NSAlertDefaultReturn)
		return;
	
	int row = [playlistsTable selectedRow];
	// update the jukebox first so not released
	NSLog(@"Updating playlist from deletePlaylist:");
	[self deleteJukeboxPlaylist:(Playlist *)[playlists objectAtIndex:row]];
	[playlists removeObjectAtIndex:row];
	tracksInCurrentPlaylist = nil;
	
	[playlistsTable reloadData];
}

- (void)NJBConnected:(NSNotification *)note
{
	tracksInCurrentPlaylist = nil;
	[playlists release];
	playlists = nil;
	[playlistsTable reloadData];
	[playlistsTrackTable reloadData];
	if ([playlistsTable selectedRow] != -1)
		[deletePlaylistButton setEnabled:YES];
	if ([playlistsTrackTable selectedRow] != -1)
		[deleteFromPlaylistButton setEnabled:YES];
	if ([njbTable selectedRow] != -1 && [playlistsTable selectedRow] != -1)
		[addToPlaylistButton setEnabled:YES];
	[super NJBConnected:note];
}

- (void)NJBDisconnected:(NSNotification *)note
{
	[newPlaylistButton setEnabled:NO];
	[deletePlaylistButton setEnabled:NO];
	[addToPlaylistButton setEnabled:NO];
	[deleteFromPlaylistButton setEnabled:NO];
	[moveTracksUpButton setEnabled:NO];
	[moveTracksDownButton setEnabled:NO];
	[super NJBDisconnected:note];
}

- (void)tableView:(NSTableView *)aTableView setObjectValue:(id)anObject forTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
	if (aTableView != playlistsTable)
		return;
	Playlist *playlist = (Playlist *)[playlists objectAtIndex:rowIndex];
	[playlist lock];
	[playlist setName:anObject];
	[playlist unlock];
	// update the jukebox
	NSLog(@"Updating playlist from tableView:setObjectValue:forTableColumn:row:");
	[self updateJukeboxPlaylist:playlist];
}

- (void)tableViewSelectionDidChange:(NSNotification *)aNotification
{
	// don't necessarily have to disable all buttons if not connected
	// as will have been disabled in NJBDisconnected:
	if ([aNotification object] == njbTable)
	{
		if ([njbTable selectedRow] == -1 || [playlistsTable selectedRow] == -1)
			[addToPlaylistButton setEnabled:NO];
		else if ([theNJB isConnected])
		{
			[addToPlaylistButton setEnabled:YES];
		}
	}
	else if ([aNotification object] == playlistsTable)
	{
		int row = [playlistsTable selectedRow];
		if (row == -1)
		{
			[addToPlaylistButton setEnabled:NO];
			[deletePlaylistButton setEnabled:NO];
			tracksInCurrentPlaylist = nil;
			[playlistsTrackTable reloadData];
		}
		else
		{
		  // update playlistsTrackTable
			// no need to retain this as is not released until playlists is dealloced
			Playlist *playlist = (Playlist *)[playlists objectAtIndex:[playlistsTable selectedRow]];
			tracksInCurrentPlaylist = [playlist tracks];
			[playlistsTrackTable reloadData];
			
			if ([theNJB isConnected])
			{
				[deletePlaylistButton setEnabled:YES];
				if ([njbTable selectedRow] != -1)
					[addToPlaylistButton setEnabled:YES];
			}
		}
	}
	else if ([aNotification object] == playlistsTrackTable)
	{
		if ([playlistsTrackTable selectedRow] == -1)
		{
			[deleteFromPlaylistButton setEnabled:NO];
			[moveTracksUpButton setEnabled:NO];
			[moveTracksDownButton setEnabled:NO];
		}
		else if ([theNJB isConnected])
		{
			[deleteFromPlaylistButton setEnabled:YES];
			[moveTracksUpButton setEnabled:YES];
			[moveTracksDownButton setEnabled:YES];
		}
	}
}

/* shift the selected tracks up in order,
 * if possible
 */
- (IBAction)moveTracksUp:(id)sender
{
	Playlist *playlist = (Playlist *)[playlists objectAtIndex:[playlistsTable selectedRow]];
	
	// nothing to do if all selected
	if ([playlistsTrackTable numberOfSelectedRows] == [playlist trackCount])
		return;
	
	[playlist lock];
	
	NSEnumerator *enumerator = [playlistsTrackTable selectedRowEnumerator];
	NSNumber *currentIndex = nil;
	unsigned previousIndex = 0;
	BOOL selectionIncludesFirstElement = NO;
	unsigned index = 0;
	while (currentIndex = [enumerator nextObject])
	{
		index = [currentIndex unsignedIntValue];
		if (index == 0 || (selectionIncludesFirstElement && previousIndex == index - 1))
		{
			selectionIncludesFirstElement = YES;
			previousIndex = index;
			continue;
		}
		else
			selectionIncludesFirstElement = NO;

		[self swapPlaylistTrackWithPreviousAtIndex:index forPlaylist:playlist];
	}
	
	// scroll to upper most selected item
	// (which is now in new position)
	enumerator = [playlistsTrackTable selectedRowEnumerator];
	currentIndex = [enumerator nextObject];
	[playlistsTrackTable scrollRowToVisible:[currentIndex unsignedIntValue]];
	
	// only update if something has changed
	if (!selectionIncludesFirstElement)
		[self updateJukeboxPlaylist:playlist];
	
	[playlist unlock];
	
	[playlistsTrackTable reloadData];
}

/* shift the selected down up in order,
 * if possible
 */
- (IBAction)moveTracksDown:(id)sender
{
	Playlist *playlist = (Playlist *)[playlists objectAtIndex:[playlistsTable selectedRow]];
	
	// nothing to do if all selected
	if ([playlistsTrackTable numberOfSelectedRows] == [playlist trackCount])
		return;
	
	[playlist lock];
		
	// need to get indexes in reverse order (highest first)
	NSEnumerator *enumerator = [playlistsTrackTable selectedRowEnumerator];
	NSNumber *currentIndex = nil;
	NSMutableArray *indexes = [[NSMutableArray alloc] initWithCapacity:[playlistsTrackTable numberOfSelectedRows]];
	while (currentIndex = [enumerator nextObject])
		[indexes addObject:currentIndex];
	
	enumerator = [indexes reverseObjectEnumerator];
	unsigned previousIndex = 0;
	BOOL selectionIncludesLastElement = NO;
	while (currentIndex = [enumerator nextObject])
	{
		unsigned index = [currentIndex unsignedIntValue];
		if (index == [playlist trackCount] - 1 || (selectionIncludesLastElement && previousIndex == index + 1))
		{
			selectionIncludesLastElement = YES;
			previousIndex = index;
			continue;
		}
		else
			selectionIncludesLastElement = NO;
		
		[self swapPlaylistTrackWithNextAtIndex:index forPlaylist:playlist];
	}
	
	// scroll to lower most selected item
	// which needs to be adjusted to new position
	enumerator = [indexes reverseObjectEnumerator];
	currentIndex = [enumerator nextObject];
	unsigned selectIndex = [currentIndex unsignedIntValue];
	if (selectIndex != [playlist trackCount] - 1)
		selectIndex++;
	[playlistsTrackTable scrollRowToVisible:selectIndex];
	
	// only update if something has changed
	if (!selectionIncludesLastElement)
	[self updateJukeboxPlaylist:playlist];
	
	[playlist unlock];
	
	[playlistsTrackTable reloadData];
	[indexes release];
}

/* swaps the track at index with the track at index-1
 */
- (void)swapPlaylistTrackWithPreviousAtIndex:(unsigned)index forPlaylist:(Playlist *)playlist
{
	[playlist swapTrackWithPreviousAtIndex:index];
	[playlistsTrackTable deselectRow:index];
	[playlistsTrackTable selectRow:(index - 1) byExtendingSelection:YES];
}

/* swaps the track at index with the track at index+1
*/
- (void)swapPlaylistTrackWithNextAtIndex:(unsigned)index forPlaylist:(Playlist *)playlist
{
	[playlist swapTrackWithNextAtIndex:index];
	[playlistsTrackTable deselectRow:index];
	[playlistsTrackTable selectRow:(index + 1) byExtendingSelection:YES];
}

- (void)NJBTrackListModified:(NSNotification *)note
{
	if (!isActive)
	{
		trackListUpToDate = NO;
	}
}

@end
