//
//  MusicTab.m
//  XNJB
//

/* this tab is for the music tab
 * upload/download tracks, delete
 */

#import "MusicTab.h"
#import "Track.h"
#import "ID3Tagger.h"
#import "defs.h"

// these correspond to the identifiers in the NIB file
#define COLUMN_TITLE @"Title"
#define COLUMN_ARTIST @"Artist"
#define COLUMN_ALBUM @"Album"
#define COLUMN_GENRE @"Genre"
#define COLUMN_LENGTH @"Length"
#define COLUMN_TRACK_NO @"Track No"
#define COLUMN_CODEC @"Codec"
#define COLUMN_FILESIZE @"Filesize"
#define COLUMN_YEAR @"Year"

// declare the private methods
@interface MusicTab (PrivateAPI)
- (NSString *)filenameForTrack:(Track *)track;
@end

@implementation MusicTab

// init/dealloc

- (id)init
{
	if (self = [super init])
	{
		
		fileExtensionsToCodecsDictionary = [[NSDictionary alloc] initWithObjectsAndKeys:[NSNumber numberWithInt:CODEC_MP3],
			@"mp3", [NSNumber numberWithInt:CODEC_WMA], @"wma", [NSNumber numberWithInt:CODEC_WAV], @"wav", nil];
	}
	return self;
}

- (void)dealloc
{
	[fileExtensionsToCodecsDictionary release];
	[super dealloc];
}

/**********************/

- (void)awakeFromNib
{
	[super awakeFromNib];
	NSArray *extensions = [NSArray arrayWithObjects:@"mp3", @"wma", @"wav", nil];
	[browser setAllowedExtensions:extensions];
	[browser setBaseLocation:[preferences musicTabDir]];
	
	[njbTable setAllowedExtensions:extensions];
}

- (void)onConnectAndActive
{
	[super onConnectAndActive];
	if (!trackListUpToDate)
		[self loadTracks];
}

/* replaces track variables (%t, %a, ...)
 * with actual values for the track given.
 * does not add extension
 * called when copying file from NJB
 */
- (NSString *)filenameForTrack:(Track *)track
{
	// ignores [track filename] as this is always blank on my NJB3
	NSMutableString *filename = [[NSMutableString alloc] init];
	
	NSString *filenameFormat = [preferences filenameFormat];
	int i = -1;
	BOOL gotPercent = NO;
	for (i = 0; i < [filenameFormat length]; i++)
	{
		unichar currentChar = [filenameFormat characterAtIndex:i];
		if (gotPercent)
		{
			gotPercent = NO;
			switch (currentChar)
			{
				case '%':
					[filename appendString:@"%"];
					break;
					// title
				case 't':
					[filename appendString:[self replaceBadFilenameChars:[track title]]];
					break;
					// artist
				case 'a':
					[filename appendString:[self replaceBadFilenameChars:[track artist]]];
					break;
					// album
				case 'A':
					[filename appendString:[self replaceBadFilenameChars:[track album]]];
					break;
					// genre
				case 'g':
					[filename appendString:[self replaceBadFilenameChars:[track genre]]];
					break;
					// track no
				case 'n':
					[filename appendString:[NSString stringWithFormat:@"%d",[track trackNumber]]];
					break;
					// codec
				case 'c':
					[filename appendString:[track codecString]];
					break;
				default:
					// incorrect variable name so print nothing
					break;
			}
		}
		else if (currentChar == '%')
			gotPercent = YES;
		else
			[filename appendString:[NSString stringWithCharacters:&currentChar length:1]];
	}
	NSString *f = [NSString stringWithString:filename];
	[filename release];
	return f;
}

/* copy the item at index in itemArrayDisplaying
 * from the NJB
 */
- (void)copyFileFromNJB:(MyItem *)item
{
	Track *track = (Track *)item;
	NSFileManager *manager = [NSFileManager defaultManager];
	
	NSString *filename = [self filenameForTrack:track];
	NSString *directory = [[browser directory] stringByAppendingString:@"/"];
	
	NSArray *components = [filename pathComponents];
	NSMutableString *dirsSoFar = [[NSMutableString alloc] initWithString:directory];
	int i = -1;
	for (i = 0; i < [components count]; i++)
	{
		NSString *component = [components objectAtIndex:i];
		
		if ([component isEqual:@"/"])
		{
			// the format string starts with /, this will get ignored when prefixing it with directory
			// so don't do anything now
		}
		else if (i != [components count] - 1)
		{
			// this is not the filename, need to make a directory
			[dirsSoFar appendString:component];
			[dirsSoFar appendString:@"/"];
			// create the directory, will fail if already made, don't worry
			[manager createDirectoryAtPath:dirsSoFar attributes:nil];
		}
	}
	
	NSString *extension = [[NSString stringWithString:@"."] stringByAppendingString:[[track codecString] lowercaseString]];
	NSString *fullPath = [directory stringByAppendingString:filename];
	
	fullPath = [fullPath stringByAppendingString:extension];
	
	[track setFullPath:fullPath];
		
	[self copyTrackFromNJB:track];
}

/* actually add the track to the queue
 * to be copied
 */
- (void)copyTrackFromNJB:(Track *)track
{
	if (![theNJB isConnected])
		return;
		
	NJBQueueItem *copyTrack = [[NJBQueueItem alloc] initWithTarget:theNJB withSelector:@selector(downloadTrack:) withObject:track];
	[copyTrack setStatus:STATUS_DOWNLOADING_TRACK];
	NSString *description = [NSString stringWithFormat:@"Downloading track '%@'", [track title]];
	
	// could update browser here, but causes browser to lose selection so annoying
	
	// write track tag to file
	if ([preferences writeTagAfterCopy] && [track codec] == CODEC_MP3)
	{
		NJBQueueItem *writeTag = [[NJBQueueItem alloc] initWithTarget:self withSelector:@selector(writeTagToFile:) withObject:track];
		[writeTag setRunInMainThread:YES];
		[writeTag setDisplayStatus:NO];

		[self addToQueue:copyTrack withSubItems:[NSArray arrayWithObjects:writeTag, nil] withDescription:description];
		[writeTag release];
	}
	else
		[self addToQueue:copyTrack withSubItems:nil withDescription:description];
	
	[copyTrack release];
}

/* copy the file at filename to
 * the NJB.
 */
- (void)copyFileToNJB:(NSString *)filename
{
	Track *track = [[Track alloc] init];
	[track setFullPath:filename];
	[self copyTrackToNJB:track];
	[track release];
}

/* actually add the track to the queue
 * to be copied
 */
- (void)copyTrackToNJB:(Track *)track
{
	if (![theNJB isConnected])
		return;
	NJBQueueItem *copyTrack = [[NJBQueueItem alloc] initWithTarget:self withSelector:@selector(uploadTrack:) withObject:track];
	[copyTrack setStatus:STATUS_UPLOADING_TRACK];
	
	NJBQueueItem *addToList = [[NJBQueueItem alloc] initWithTarget:self withSelector:@selector(addToArray:) withObject:track withRunInMainThread:YES];
	[addToList setDisplayStatus:NO];
	
	[copyTrack cancelItemIfFail:addToList];
	
	NSString *description = [NSString stringWithFormat:@"Uploading track '%@'", [track fullPath]];
	[self addToQueue:copyTrack withSubItems:[NSArray arrayWithObjects:addToList, nil] withDescription:description];
	
	[copyTrack release];
	[addToList release];
}

/* uploads the track to the jukebox, run in worker thread
 */
- (NJBTransactionResult *)uploadTrack:(Track *)track
{
	ID3Tagger *tagger = [[ID3Tagger alloc] init];
	[tagger setFileExtensionsToCodecsDictionary:fileExtensionsToCodecsDictionary];
	[tagger setFilename:[track fullPath]];
	Track *trackWithTag = [tagger readTrack];
	[track setValuesToTrack:trackWithTag];
	[tagger release];
	
	return [theNJB uploadTrack:track];
}

/* write the info in track to the tag in the file
 */
- (void)writeTagToFile:(Track *)track
{
	ID3Tagger *tagger = [[ID3Tagger alloc] init];
	[tagger writeTrack:track];
	[tagger release];
}

/* update the drawer when someone selects a different
 * track
 */
- (void)tableViewSelectionDidChange:(NSNotification *)aNotification
{
	// we might be called from code changing the selection
	if (activeObject != OBJECT_NJB_TABLE)
		return;
	[super tableViewSelectionDidChange:aNotification];
	// unchanged tag edit will be lost
	
	if ([njbTable selectedRow] != -1)
		objectToSaveTrack = OBJECT_NJB_TABLE;
}

/* called by MainController when we must write
 * a track to a file/njb
 */
- (void)drawerWriteTag:(Track *)modifiedTrack
{
	switch (objectToSaveTrack)
	{
		case OBJECT_NJB_TABLE:
			// none selected, shouldn't be here anyway
			if ([njbTable selectedRow] == -1)
				return;	
			
			Track *oldTrack = (Track *)[[arrayController arrangedObjects] objectAtIndex:[njbTable selectedRow]];
			
			NJBQueueItem *changeTrackTag = [[NJBQueueItem alloc] initWithTarget:theNJB withSelector:@selector(changeTrackTagTo:from:) withObject:modifiedTrack];
			[changeTrackTag setObject2:oldTrack];
			[changeTrackTag setStatus:STATUS_UPDATING_TRACK_TAG];
			
			NJBQueueItem *updateTrackList = [[NJBQueueItem alloc] initWithTarget:self withSelector:@selector(replaceTrack:withTrack:)
																																withObject:oldTrack withRunInMainThread:YES];
			[updateTrackList setObject2:modifiedTrack];
			[updateTrackList setDisplayStatus:NO];
			
			[changeTrackTag cancelItemIfFail:updateTrackList];
			
			NSString *description = [NSString stringWithFormat:@"Changing tag for track '%@'", [oldTrack title]];
			[self addToQueue:changeTrackTag withSubItems:[NSArray arrayWithObjects:updateTrackList, nil] withDescription:description];

			[changeTrackTag release];
			[updateTrackList release];
			
			break;
		case OBJECT_FILE_BROWSER:
			[self writeTagToFile:modifiedTrack];
			break;
		default:
			NSLog(@"Drawer write tag without njbTable/fileBrowser selected!");
	}
}

- (void)fileSystemBrowserFileClick:(NSString *)path
{
	[super fileSystemBrowserFileClick:path];
	objectToSaveTrack = OBJECT_FILE_BROWSER;
	// unchanged tag edit will be lost
	ID3Tagger *tagger = [[ID3Tagger alloc] initWithFilename:path];
	[tagger setFileExtensionsToCodecsDictionary:fileExtensionsToCodecsDictionary];
	[drawerController enableAll];
	Track *track = [tagger readTrack];
	[drawerController showTrack:track];
	[tagger release];
	if ([track codec] != CODEC_MP3)
		[drawerController disableWrite];
}

- (void)fileSystemBrowserDirectoryClick:(NSString *)path
{
	[super fileSystemBrowserDirectoryClick:path];
	// unchanged tag edit will be lost
	[drawerController disableAll];
}

- (void)onFirstResponderChange:(NSResponder *)aResponder
{
	[super onFirstResponderChange:(NSResponder *)aResponder];
	if (aResponder == njbTable)
	{
		objectToSaveTrack = OBJECT_NJB_TABLE;
	}
	else if ([aResponder class] == [NSMatrix class])
		objectToSaveTrack = OBJECT_FILE_BROWSER;
	// used to be [aResponder class] == [NSSearchField class]
	else if (aResponder == njbSearchField)
	{
		[drawerController disableAll];
	}
}

/* add downloadTrackList to the njb queue
 */
- (void)loadTracks
{
	if (trackListUpToDate)
	{
		NSLog(@"loadTracks called in Music tab when track list up to date");
		return;
	}
	NSMutableArray *cachedTrackList = [theNJB cachedTrackList];
	if (cachedTrackList != nil)
	{
		[tempArray release];
		tempArray = cachedTrackList;
		[tempArray retain];
		[self showNewArray];
		
		trackListUpToDate = YES;
		return;
	}
	
	NJBQueueItem *getTracks = [[NJBQueueItem alloc] initWithTarget:self withSelector:@selector(downloadTrackList)];
	[getTracks setStatus:STATUS_DOWNLOADING_TRACK_LIST];

	NJBQueueItem *updateTable = [[NJBQueueItem alloc] initWithTarget:self
																											withSelector:@selector(showNewArray)
																												withObject:nil
																							 withRunInMainThread:YES];
	[updateTable setDisplayStatus:NO];
	
	[getTracks cancelItemIfFail:updateTable];
	
	NSString *description = @"Getting track list";
	[self addToQueue:getTracks withSubItems:[NSArray arrayWithObjects:updateTable, nil] withDescription:description];
	[getTracks release];
	[updateTable release];
	// it's not actually yet, but will be in a while, don't want to go copying anything
	trackListUpToDate = YES;
}

/* this is called from the queue consumer 
 * to get the tracks off the NJB and put them in fullItemArray
 * will run in worker thread
 */
- (NJBTransactionResult *)downloadTrackList
{
	[tempArray release];
	tempArray = [theNJB tracks];
	[tempArray retain];
	
	NJBTransactionResult *result = [[NJBTransactionResult alloc] initWithSuccess:(tempArray != nil)];
	
	trackListUpToDate = YES;
	
	return [result autorelease];
}

- (void)deleteFromNJB:(MyItem *)item
{
	Track *track = (Track *)item;

	NJBQueueItem *deleteTrack = [[NJBQueueItem alloc] initWithTarget:theNJB withSelector:@selector(deleteTrack:)
																												withObject:track];
	[deleteTrack setStatus:STATUS_DELETING_TRACK];
		
	NJBQueueItem *removeFromAllTracks = [[NJBQueueItem alloc] initWithTarget:self withSelector:@selector(removeFromArray:)
																																	withObject:track];
	[removeFromAllTracks setRunInMainThread:YES];
	[removeFromAllTracks setDisplayStatus:NO];
	
	[deleteTrack cancelItemIfFail:removeFromAllTracks];
	
	NSString *description = [NSString stringWithFormat:@"Deleting track '%@'", [track title]];
	[self addToQueue:deleteTrack withSubItems:[NSArray arrayWithObjects:removeFromAllTracks, nil] withDescription:description];

	[deleteTrack release];
	[removeFromAllTracks release];
}

- (void)replaceTrack:(Track *)replace withTrack:(Track *)new
{
	[arrayController replaceObject:replace withObject:new];
//	[arrayController rearrangeObjects];
}

- (void)NJBConnected:(NSNotification *)note
{
	trackListUpToDate = NO;
	
	[super NJBConnected:note];
}

- (void)NJBDisconnected:(NSNotification *)note
{
	trackListUpToDate = NO;
	
	[super NJBDisconnected:note];
	
	if (isActive && activeObject == OBJECT_NJB_TABLE)
		[drawerController disableWrite];
}

- (void)activate
{
	[super activate];
	if ([theNJB isConnected] && !trackListUpToDate)
	{
		[self loadTracks];
	}
	else if (activeObject == OBJECT_FILE_BROWSER)
	{
		if (![browser isDirectory])
		{
			ID3Tagger *tagger = [[ID3Tagger alloc] initWithFilename:[browser path]];
			[tagger setFileExtensionsToCodecsDictionary:fileExtensionsToCodecsDictionary];
			[drawerController enableAll];
			[drawerController showTrack:[tagger readTrack]];
			[tagger release];
		}
	}
}

- (void)applicationTerminating:(NSNotification *)note
{
	[preferences setLastUsedMusicTabDir:[self browserDirectory]];
	[super applicationTerminating:note];
}

- (void)NJBTrackListModified:(NSNotification *)note
{
	if (!isActive)
	{
		trackListUpToDate = NO;
	}
}

- (void)showDrawerInfo
{
	// update display to show new track
	if ([njbTable selectedRow] == -1)
	{
		[drawerController clearAll];
		[drawerController disableAll];
	}
	else
	{
		[drawerController showTrack:(Track *)[[arrayController arrangedObjects] objectAtIndex:[njbTable selectedRow]]];
		if (![theNJB isConnected])
			[drawerController disableWrite];
	}
}

@end