//
//  ID3Tagger.m
//  id3libtest
//
//  Created by Richard Low on Mon Jul 26 2004.
//

/* this class provides a wrapper for libid3tag
 * to read/write id3 tags
 */

#include "wmaread.h"
#import "ID3Tagger.h"
#import "WMATagger.h"
#import "UnicodeWrapper.h"
#import "MP3Length.h"
// for stat
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#import "TagAPI.h"

@interface ID3Tagger (PrivateAPI)
- (NSString *)trimTagString:(NSString *)string;
- (unsigned)wavFileLength:(NSString *)path;
@end

@implementation ID3Tagger

// init/dealloc methods

- (id)init
{
	return [self initWithFilename:nil];
}

- (id)initWithFilename:(NSString *)newFilename
{
	if (self = [super init])
	{
		[self setFilename:newFilename];
	}
	return self;
}

- (void)dealloc
{
	[filename release];
	[fileExtensionsToCodecsDictionary release];
	[super dealloc];
}

// accessor methods

- (void)setFilename:(NSString *)newFilename
{
	[newFilename retain];
	[filename release];
	filename = newFilename;
}

- (void)setFileExtensionsToCodecsDictionary:(NSDictionary *)newFileExtensionsToCodecsDictionary
{
	[newFileExtensionsToCodecsDictionary retain];
	[fileExtensionsToCodecsDictionary release];
	fileExtensionsToCodecsDictionary = newFileExtensionsToCodecsDictionary;
}

/**********************/

/* returns the codec for the current file
 * based on extension
 */
- (codecTypes)codec
{
	NSEnumerator *enumerator = [fileExtensionsToCodecsDictionary keyEnumerator];
	NSString *key;
	
	while (key = [enumerator nextObject]) {
		if ([[filename pathExtension] compare:key options:NSCaseInsensitiveSearch] == NSOrderedSame)
		{
			switch ([[fileExtensionsToCodecsDictionary objectForKey:key] intValue])
			{
				case CODEC_MP3:
					return CODEC_MP3;
				case CODEC_WAV:
					return CODEC_WAV;
				case CODEC_WMA:
					return CODEC_WMA;
				default:
					return CODEC_UNDEFINED;
			}
		}
	}
	return CODEC_UNDEFINED;
}

/* read the id3 tag and return it as
 * a track object
 * will pass on to WMATagger if a wma file
 */
- (Track *)readTrack
{	
	// if is WMA file, we can't do anything here, pass on to WMATagger
	if ([self codec] == CODEC_WMA)
	{
		return [WMATagger readTrack:filename];
	}
	
	struct stat sb;
	// check file exists
	if (stat([filename fileSystemRepresentation], &sb) == -1)
		return nil;
	
	// N.B. cannot use NSFileManager as this is not thread safe
		
	Track *track = [[Track alloc] init];
	
	[track setFilesize:sb.st_size];
	[track setCodec:[self codec]];
	[track setFullPath:filename];
	
  if ([self codec] == CODEC_MP3)
	{
		NSMutableDictionary* GenreDictionary = [[NSMutableDictionary alloc] initWithContentsOfFile:@"~/GenreDictionary.plist"];
		TagAPI *tag = [[TagAPI alloc] initWithGenreList: GenreDictionary];
		[tag examineFile:filename];
		
		[track setTitle:[self trimTagString:[tag getTitle]]];
		[track setArtist:[self trimTagString:[tag getArtist]]];
		[track setAlbum:[self trimTagString:[tag getAlbum]]];
		[track setGenre:[self trimTagString:[[tag getGenreNames] objectAtIndex:0]]];
		[track setTrackNumber:[tag getTrack]];
		[track setYear:[tag getYear]];
		
		[tag release];
		
		// read the length from the header (use this as it is faster than id3.framework
		[track setLength:[MP3Length length:filename]];
	}
	else if ([self codec] == CODEC_WAV)
	{
		// use filename as title, but strip extension
		NSString *title = [[filename lastPathComponent] stringByDeletingPathExtension];
		[track setTitle:title];
		[track setLength:[self wavFileLength:filename]];
	}
	
	return [track autorelease];
}

/* write the tag in track to the file
 * in [track fullPath]
 */
- (void)writeTrack:(Track *)track
{
	NSMutableDictionary * GenreDictionary = [[NSMutableDictionary alloc] initWithContentsOfFile: @"~/GenreDictionary.plist"];
	TagAPI *tag = [[TagAPI alloc] initWithGenreList: GenreDictionary];
	[tag examineFile:[track fullPath]];

	[tag setTitle:[track title]];
	[tag setArtist:[track artist]];
	[tag setGenreName:[NSArray arrayWithObject:[track genre]]];
	int totalTracks = [tag getTotalNumberTracks];
	if (totalTracks != -1)
		[tag setTrack:[track trackNumber] totalTracks:[tag getTotalNumberTracks]];
	else
		[tag setTrack:[track trackNumber] totalTracks:0];
	
	[tag setYear:[track year]];
	
	[tag updateFile];
	
	[tag release];	
}

/* get the length of the wav file at path
 * see http://ccrma.stanford.edu/courses/422/projects/WaveFormat/ for wave format spec
 */
- (unsigned)wavFileLength:(NSString *)path
{
	// open the file handle for the specified path
  NSFileHandle *file = [NSFileHandle fileHandleForReadingAtPath:path];
  if (file == nil)
  {
		NSLog(@"Cannot open file :%@", path);
    return 0;
	}
	
	[file seekToFileOffset:28];
	
	NSData *buffer = [file readDataOfLength:4];
	const unsigned char *bytesPerSecondChar = (const unsigned char*)[buffer bytes];
	unsigned int bytesPerSecond = bytesPerSecondChar[0] + (bytesPerSecondChar[1] << 8) + (bytesPerSecondChar[2] << 16) + (bytesPerSecondChar[3] << 24);
	
	[file seekToFileOffset:40];
	
	buffer = [file readDataOfLength:4];
	const unsigned char *bytesLongChar = (const unsigned char*)[buffer bytes];
	unsigned int bytesLong = bytesLongChar[0] + (bytesLongChar[1] << 8) + (bytesLongChar[2] << 16) + (bytesLongChar[3] << 24);
	
	[file closeFile];
	
	if (bytesPerSecond == 0)
	{
		NSLog(@"bytesPerSecond == 0 in file %@", path);
		return 0;
	}
	// round up to be on the safe side
	if (bytesLong % bytesPerSecond == 0)
		return (bytesLong / bytesPerSecond);
	else
		return (bytesLong / bytesPerSecond) + 1;
}

- (NSString *)trimTagString:(NSString *)string
{
	if (string == nil || [string length] < 1)
		return string;
	
	while ([string length] > 0 && [string characterAtIndex:([string length] - 1)] == 0)
	{
		string = [string substringToIndex:([string length] - 1)];
	}
	
	return string;
}

@end
