//
//  NJB.m
//  XNJB
//
//  Created by Richard Low on Wed Jul 21 2004.
//

// todo: can we find out how many tracks there are before we download for progress bar & set capacity in NSMutableArray?

/* An obj-c wrapper for libnjb
 * most methods return an NJBTransactionResult
 * to indicate success/failure
 */

#import "NJB.h"
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#import "Track.h"
#include "string.h"
#include "base.h"
#include "njb_error.h"
#import "defs.h"
#include <sys/stat.h>
#include <unistd.h>
#import "UnicodeWrapper.h"

// declared so can be accessed from C function progress
// set to be the outlet statusDisplayer in awakeFromNib
StatusDisplayer *statusDisplayerGlobal;

// declare the private methods
@interface NJB (PrivateAPI)
- (void)updateDiskSpace;
- (njb_songid_t *)songidStructFromTrack:(Track *)track;
- (NSString *)njbErrorString;
- (NSString *)rateString:(double) rate;
- (NSString *)uniqueFilename:(NSString *)path;
// we make this private so that the user never changes the setting directly
// so we only change it before a transfer, so it is not changed during a transfer
// which could mess things up
- (void)enableTurbo;

- (BOOL)setMTPTrackTag:(Track *)track;
@end

@implementation NJB

// init/dealloc methods

- (id)init
{
	if (self = [super init])
	{
		njb = NULL;
		// 15 is max
		[self setDebug:0];
		[self setTurbo:YES];
		// this defined for the non MTP devices
		batteryLevelMax = 100;
		storageID = 0;
	}
	return self;
}

- (void)dealloc
{
	[statusDisplayerGlobal release];
	[cachedTrackList release];
	
	[deviceString release];
	[firmwareVersionString release];
	[deviceIDString release];
	[deviceVersionString release];
	
	[super dealloc];
}

// accessor methods

- (void)setDebug:(unsigned)newDebug
{
	// see libnjb.h for the Debug flags
	debug = newDebug;
}

- (statusTypes)status
{
	return status;
}

- (void)setStatus:(statusTypes)newStatus
{
	status = newStatus;
	[statusDisplayer setStatus:status];
}

- (BOOL)isConnected
{
	return connected;
}

- (NSString *)deviceString
{
	return deviceString;
}

- (NSString *)firmwareVersionString
{
	return firmwareVersionString;
}

- (NSString *)deviceIDString
{
	return deviceIDString;
}

- (NSString *)deviceVersionString
{
	return deviceVersionString;
}

- (void)setTurbo:(BOOL)newTurbo
{
	turbo = newTurbo;
}

- (int)batteryLevelMax
{
	return batteryLevelMax;
}

/**********************/

- (void)awakeFromNib
{
	statusDisplayerGlobal = [statusDisplayer retain];
}

- (NSString *)njbErrorString
{
	const char *sp;
	NSMutableString *errorString = [[NSMutableString alloc] init];
	NJB_Error_Reset_Geterror(njb);
  while ((sp = NJB_Error_Geterror(njb)) != NULL)
	{
		[errorString appendString:[NSString stringWithFormat:@"%s, ", sp]];
  }
  njb_error_clear(njb);
	
	return [errorString autorelease];
}

/* connect to the NJB
 */
- (NJBTransactionResult *)connect
{	
	int n;
	
	connected = NO;
	NSString *errorString = nil;
	
	// try for PDE devices first
	
	if (debug)
		NJB_Set_Debug(debug);
	
	if (NJB_Discover(njbs, 0, &n) == -1)
	{
		[self setStatus:STATUS_NO_NJB];
		errorString = NSLocalizedString(@"Could not discover any jukeboxes", nil);
	}
	else
	{
		if (n == 0)
		{
			[self setStatus:STATUS_NO_NJB];
			errorString = NSLocalizedString(@"Could not locate any jukeboxes", nil);
		}
		else
		{
			njb = &njbs[0];
	
			if (NJB_Open(njb) == -1)
			{
				[self setStatus:STATUS_COULD_NOT_OPEN];
				errorString = [self njbErrorString];
			}
			else
			{
				if (NJB_Capture(njb) == -1)
				{
					[self setStatus:STATUS_COULD_NOT_CAPTURE];
					errorString = [self njbErrorString];
					[self disconnect];
				}
				else
				{
					mtpDevice = NO;
					connected = YES;
					[self setStatus:STATUS_CONNECTED];
				}
			}
		}
	}

	if (!connected)
	{
		mtpDevice = YES;
	  
		uint16_t ret = connect_first_device(&params, &ptp_usb, &interfaceNumber);
		
		switch (ret)
		{
			case PTP_CD_RC_CONNECTED:
				connected = YES;
				[self setStatus:STATUS_CONNECTED];
				break;
			case PTP_CD_RC_NO_DEVICES:
				connected = NO;
				errorString = NSLocalizedString(@"Could not locate any jukeboxes", nil);
				[self setStatus:STATUS_NO_NJB];
				break;
			case PTP_CD_RC_ERROR_CONNECTING:
				connected = NO;
				errorString = NSLocalizedString(@"Could not talk to jukebox", nil);
				[self setStatus:STATUS_COULD_NOT_OPEN];
				break;
		}
	}
	
	if (!connected)
	{
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:errorString] autorelease]; 
	}
	
	if (mtpDevice)
	{
		// get storage ID
		PTPStorageIDs storageIDs;
		if (ptp_getstorageids (&params, &storageIDs) == PTP_RC_OK)
		{
			if (storageIDs.n > 0)
				storageID = storageIDs.Storage[0];
			free(storageIDs.Storage);
		}
	}
	
	[self updateDiskSpace];
	
	// enable UTF8
	if (!mtpDevice)
		NJB_Set_Unicode(NJB_UC_UTF8);
	
	// check we have reset the cache, should have been done on disconnect
	[cachedTrackList release];
	cachedTrackList = nil;
	
	if (mtpDevice)
	{
		if (ptp_getdeviceinfo(&params,&(params.deviceinfo)) == PTP_RC_OK)
		{
			deviceString = [NSString stringWithCString:params.deviceinfo.Model];
			deviceIDString = [NSString stringWithCString:params.deviceinfo.SerialNumber];
			
			// the version string is of the form firmware_device
			NSString *versionString = [NSString stringWithCString:params.deviceinfo.DeviceVersion];
			NSArray *versions = [versionString componentsSeparatedByString:@"_"];
			if ([versions count] != 2)
			{
				deviceVersionString = versionString;
				firmwareVersionString = @"";
			}
			else
			{
				deviceVersionString = [versions objectAtIndex:1];
				firmwareVersionString = [versions objectAtIndex:0];
			}
		}
		else
		{
			deviceString = @"";
			deviceVersionString = @"";
			deviceIDString = @"";
			firmwareVersionString = @"";
		}
	
		// get max battery level
		PTPDevicePropDesc dpd;
		
		if (ptp_getdevicepropdesc(&params,PTP_DPC_BatteryLevel,&dpd) != PTP_RC_OK)
		{
			NSLog(@"Could not get battery level property description");
			return 0;
		}
		
		// if is NULL, just leave as default
		if (dpd.FORM.Range.MaximumValue != NULL)
			batteryLevelMax = *(uint8_t *)dpd.FORM.Range.MaximumValue;	
		
		ptp_free_devicepropdesc(&dpd);
	}
	else
	{
		[self storeDeviceString];
		[self storeFirmwareVersionString];
		[self storeDeviceIDString];
		[self storeDeviceVersionString];
	}

	// send notification, make sure this is last so all variables are set above
	[statusDisplayer postNotificationName:NOTIFICATION_CONNECTED object:self];
		
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

/* disconnect from the NJB
 */
- (void)disconnect
{
	if ([self isConnected])
	{
		if (!mtpDevice)
		{
			NJB_Release(njb);
			NJB_Close(njb);
		}
		else
		{
			close_device(&ptp_usb, &params, interfaceNumber);
			ptp_free_deviceinfo(&params.deviceinfo);
		}
		connected = NO;
		[self setStatus:STATUS_DISCONNECTED];
		
		// lose the cache
		[cachedTrackList release];
		cachedTrackList = nil;
		
		// send notification
		[statusDisplayer postNotificationName:NOTIFICATION_DISCONNECTED object:self];
	}
}

- (NSString *)productName
{
	if (njb == NULL)
		return nil;
	const char *name = NJB_Get_Device_Name(njb, 1);
	if (name != NULL)
		return [NSString stringWithCString:name];
	else
		return nil;
}

- (NSString *)ownerString
{
	if (!mtpDevice)
	{
		char *ownerString = NJB_Get_Owner_String(njb);
		if (ownerString == NULL)
			return nil;
		NSString *nsOwnerString = [[NSString alloc] initWithUTF8String:ownerString];
		free(ownerString);
		return [nsOwnerString autorelease];
	}
	else
	{
		unsigned short *unistring = NULL;
		if (ptp_getdevicepropvalue(&params, PTP_DPC_DeviceFriendlyName, (void **)&unistring, PTP_DTC_UNISTR) == PTP_RC_OK)
		{
			NSString *deviceFriendlyName = [UnicodeWrapper stringFromUTF16:unistring];
			free(unistring);
			return deviceFriendlyName;
		}
		else
			return @"";
	}
}

- (NSMutableArray *)tracks
{
	if (![self isConnected])
		return nil;
	if (cachedTrackList)
	{
		return [self cachedTrackList];
	}
	
	downloadingTracks = YES;
	cachedTrackList = [[NSMutableArray alloc] init];
	if (!mtpDevice)
	{
		NJB_Reset_Get_Track_Tag(njb);	
		njb_songid_t *songtag;
		njb_songid_frame_t *frame;
		while ((songtag = NJB_Get_Track_Tag(njb))) {
			Track *track = [[Track alloc] init];
			
			frame = NJB_Songid_Findframe(songtag, FR_TITLE);
			if (frame != NULL && frame->data.strval != NULL)
				[track setTitle:[NSString stringWithUTF8String:frame->data.strval]];
			
			frame = NJB_Songid_Findframe(songtag, FR_ALBUM);
			if (frame != NULL && frame->data.strval != NULL)
				[track setAlbum:[NSString stringWithUTF8String:frame->data.strval]];
			
			frame = NJB_Songid_Findframe(songtag, FR_ARTIST);
			if (frame != NULL && frame->data.strval != NULL)
				[track setArtist:[NSString stringWithUTF8String:frame->data.strval]];
			
			frame = NJB_Songid_Findframe(songtag, FR_GENRE);
			if (frame != NULL && frame->data.strval != NULL)
				[track setGenre:[NSString stringWithUTF8String:frame->data.strval]];
			
			// this is not used: we don't get extended track info from njb3
			// njb1 gets it but ignored
			frame = NJB_Songid_Findframe(songtag, FR_FNAME);
			if (frame != NULL && frame->data.strval != NULL)
				[track setFilename:[NSString stringWithUTF8String:frame->data.strval]];
			
			frame = NJB_Songid_Findframe(songtag, FR_SIZE);
			if (frame != NULL)
			{
				if (frame->type == NJB_TYPE_UINT16)
					[track setFilesize:frame->data.u_int16_val];
				else
					[track setFilesize:frame->data.u_int32_val];
			}
			frame = NJB_Songid_Findframe(songtag, FR_LENGTH);
			if (frame != NULL)
			{
				if (frame->type == NJB_TYPE_UINT16)
					[track setLength:frame->data.u_int16_val];
				else
					[track setLength:frame->data.u_int32_val];
			}
			frame = NJB_Songid_Findframe(songtag, FR_TRACK);
			if (frame != NULL)
			{
				if (frame->type == NJB_TYPE_UINT16)
					[track setTrackNumber:frame->data.u_int16_val];
				else if (frame->type == NJB_TYPE_UINT32)
					[track setTrackNumber:frame->data.u_int32_val];
				// in case it's a string
				else if (frame->type == NJB_TYPE_STRING && frame->data.strval != NULL)
				{
					NSString *trackNumber = [NSString stringWithCString:frame->data.strval];
					[track setTrackNumber:(unsigned)[trackNumber intValue]];
				}
				else
					NSLog(@"type not expected for FR_TRACK field %d", frame->type);
			}
			frame = NJB_Songid_Findframe(songtag, FR_CODEC);
			if (frame != NULL)
			{
				if (frame->data.strval == NULL)
					[track setCodec:CODEC_UNDEFINED];
				else
				{
					if (strcmp(frame->data.strval, NJB_CODEC_MP3) == 0)
						[track setCodec:CODEC_MP3];
					else if (strcmp(frame->data.strval, NJB_CODEC_WMA) == 0)
						[track setCodec:CODEC_WMA];
					else if (strcmp(frame->data.strval, NJB_CODEC_WAV) == 0)
						[track setCodec:CODEC_WAV];
					else if (strcmp(frame->data.strval, NJB_CODEC_AA) == 0)
						[track setCodec:CODEC_AA];
					else
						[track setCodec:CODEC_UNDEFINED];
				}
			}
			
			[track setItemID:songtag->trid];

			frame = NJB_Songid_Findframe(songtag, FR_YEAR);
			if (frame != NULL)
			{
				if (frame->type == NJB_TYPE_UINT16)
					[track setYear:frame->data.u_int16_val];
				else if (frame->type == NJB_TYPE_UINT32)
					[track setYear:frame->data.u_int32_val];
				// strings on NJB1
				else if (frame->type == NJB_TYPE_STRING && frame->data.strval != NULL)
				{
					NSString *year = [NSString stringWithCString:frame->data.strval];
					[track setYear:(unsigned)[year intValue]];
				}
				else
					NSLog(@"type not expected for FR_YEAR field %d", frame->type);
			}
			else
				[track setYear:0];
			
			[cachedTrackList addObject:track];
			[track release];
			
			NJB_Songid_Destroy(songtag);
		}
	}
	else
	{		
		PTPObjectInfo oi;
		unsigned short *unicodevalue = NULL;
		uint16_t *uint16value = NULL;
		uint32_t *uint32value = NULL;
		int ret;
		
		if (ptp_getobjecthandles(&params,PTP_GOH_ALL_STORAGE, PTP_GOH_ALL_FORMATS, PTP_GOH_ALL_ASSOCS, &params.handles) != PTP_RC_OK)
		{
			// todo: print error
			NSLog(@"Could not get object handles...");
			return nil;
		}
		int i = 0;
		for (i = 0; i < params.handles.n; i++) {
			if (ptp_getobjectinfo(&params, params.handles.Handler[i], &oi) == PTP_RC_OK)
			{
				/*
				if (oi.Filename != NULL)
					NSLog(@"handle: 0x%08.x, filename: %s", params.handles.Handler[i], oi.Filename); */
				
				if (oi.ObjectFormat == PTP_OFC_Association || 
						(oi.ObjectFormat != PTP_OFC_WAV && oi.ObjectFormat != PTP_OFC_MP3 && oi.ObjectFormat != PTP_OFC_WMA))
					continue;
				
				Track *track = [[Track alloc] init];
			
				ret = ptp_getobjectpropvalue(&params, PTP_OPC_Name, params.handles.Handler[i], (void**)&unicodevalue, PTP_DTC_UNISTR);
				if (ret == PTP_RC_OK && unicodevalue != NULL)
				{
					[track setTitle:[UnicodeWrapper stringFromUTF16:unicodevalue]];
					free(unicodevalue);
					unicodevalue = NULL;
				}

				switch (oi.ObjectFormat)
				{
					case PTP_OFC_WAV:
						[track setCodec:CODEC_WAV];
						break;
					case PTP_OFC_MP3:
						[track setCodec:CODEC_MP3];
						break;
					case PTP_OFC_WMA:
						[track setCodec:CODEC_WMA];
						break;
					default:
						[track setCodec:CODEC_UNDEFINED];
				}
				
				[track setFilesize:oi.ObjectCompressedSize];
				if (oi.Filename != NULL)
					[track setFilename:[NSString stringWithUTF8String:oi.Filename]];
				
				ret = ptp_getobjectpropvalue(&params, PTP_OPC_Artist, params.handles.Handler[i], (void**)&unicodevalue, PTP_DTC_UNISTR);
				if (ret == PTP_RC_OK && unicodevalue != NULL)
				{
					[track setArtist:[UnicodeWrapper stringFromUTF16:unicodevalue]];
					free(unicodevalue);
					unicodevalue = NULL;
				}
				
				ret = ptp_getobjectpropvalue(&params, PTP_OPC_Duration, params.handles.Handler[i], (void**)&uint32value, PTP_DTC_UINT32);
				if (ret == PTP_RC_OK && uint32value != NULL)
				{
					// duration is in milliseconds, we want seconds
					[track setLength:(*uint32value)/1000];
					free(uint32value);
					uint32value = NULL;
				}
				
				ret = ptp_getobjectpropvalue(&params, PTP_OPC_Track, params.handles.Handler[i], (void**)&uint16value, PTP_DTC_UINT16);
				if (ret == PTP_RC_OK && uint16value != NULL)
				{
					[track setTrackNumber:*uint16value];
					free(uint16value);
					uint16value = NULL;
				}
				
				ret = ptp_getobjectpropvalue(&params, PTP_OPC_Genre, params.handles.Handler[i], (void**)&unicodevalue, PTP_DTC_UNISTR);
				if (ret == PTP_RC_OK && unicodevalue != NULL)
				{
					[track setGenre:[UnicodeWrapper stringFromUTF16:unicodevalue]];
					free(unicodevalue);
					unicodevalue = NULL;
				}
				
				ret = ptp_getobjectpropvalue(&params, PTP_OPC_AlbumName, params.handles.Handler[i], (void**)&unicodevalue, PTP_DTC_UNISTR);
				if (ret == PTP_RC_OK && unicodevalue != NULL)
				{
					[track setAlbum:[UnicodeWrapper stringFromUTF16:unicodevalue]];
					free(unicodevalue);
					unicodevalue = NULL;
				}
				
				char *timeString= NULL;
				ret = ptp_getobjectpropvalue(&params, PTP_OPC_OriginalReleaseDate, params.handles.Handler[i], (void**)&timeString, PTP_DTC_STR);
				if (ret == PTP_RC_OK && timeString != NULL)
				{
					NSString *nsTimeString = [NSString stringWithCString:timeString];
					free(timeString);
					timeString = NULL;
					// the Creative devices don't send seconds (but they do send tenths of a second!)
					NSCalendarDate *date = [[NSCalendarDate alloc] initWithString:nsTimeString calendarFormat:@"%Y%m%dT%H%M"];
					
					[track setYear:[date yearOfCommonEra]];
				}
				
				[track setItemID:params.handles.Handler[i]];
				
				[cachedTrackList addObject:track];
				[track release];
			}
		}
	}
	
	downloadingTracks = NO;
	return [self cachedTrackList];
}

/* uploads the track and sets the track id
 */
- (NJBTransactionResult *)uploadTrack:(Track *)track
{
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:NSLocalizedString(@"Not connected", nil)] autorelease];
	
	[self setStatus:STATUS_UPLOADING_TRACK];
	
	NSDate *date = [NSDate date];
	
	if (!mtpDevice)
	{
		// set turbo on/off
		[self enableTurbo];
		
		unsigned int trackid;
		njb_songid_t *songid = [self songidStructFromTrack:track];
		if (NJB_Send_Track(njb, [[track fullPath] fileSystemRepresentation], songid, progress, NULL, &trackid) == -1 ) {
			NSString *error = [self njbErrorString];
			NSLog(error);
			NJB_Songid_Destroy(songid);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		} else {
			[track setItemID:trackid];
			
			NJB_Songid_Destroy(songid);
		}
	}
	else
	{
		uint32_t handle = 0;
		uint16_t format = 0;
			
		switch ([track codec])
		{
			case CODEC_MP3:
				format = PTP_OFC_MP3;
				break;
			case CODEC_WAV:
				format = PTP_OFC_WAV;
				break;
			case CODEC_WMA:
				format = PTP_OFC_WMA;
				break;
			default:
				format = PTP_OFC_Undefined;
				break;
		}
		
		if (send_file(&params, [[track fullPath] fileSystemRepresentation], [[[track fullPath] lastPathComponent] fileSystemRepresentation],
									format, progressPTP, &handle) != PTP_RC_OK)
		{
			NSString *error = @"Error uploading file";
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
		
		[track setItemID:handle];
	
		if (![self setMTPTrackTag:track])
		{
			NSString *error = @"Error setting track info for file, but was uploaded correctly";
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	
	double time = -[date timeIntervalSinceNow];
	// rate in Mbps
	double rate = ((double)[track filesize] / time) * 8.0 / (1024.0 * 1024.0);
	
	[self updateDiskSpace];
	
	// update my cache
	[cachedTrackList addObject:track];
	// tell others the track list has changed
	[statusDisplayer postNotificationName:NOTIFICATION_TRACK_LIST_MODIFIED object:self];
	
	return [[[NJBTransactionResult alloc] initWithSuccess:YES
																					 resultString:[NSString stringWithFormat:NSLocalizedString(@"Speed %@ Mbps", nil), [self rateString:rate]]] autorelease];
		
}

- (NJBTransactionResult *)deleteTrack:(Track *)track
{
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	
	if (!mtpDevice)
	{
		if (NJB_Delete_Track(njb, [track itemID]) == -1)
		{
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	else
	{
		int ret = ptp_deleteobject(&params, [track itemID],0);
		if (ret != PTP_RC_OK)
		{
			ptp_perror(&params, ret);
			NSString *error = @"error deleting file";
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	
	[self updateDiskSpace];
	// update my cache
	[cachedTrackList removeObject:track];
	// tell others the track list has changed
	[statusDisplayer postNotificationName:NOTIFICATION_TRACK_LIST_MODIFIED object:self];
		
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)deleteFile:(NSNumber *)fileID
{
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	
	if (!mtpDevice)
	{
		if (NJB_Delete_Datafile(njb, [fileID unsignedIntValue]) == -1)
		{
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	else
	{
		int ret = ptp_deleteobject(&params, [fileID unsignedIntValue], 0);
		if (ret != PTP_RC_OK)
		{
			ptp_perror(&params, ret);
			NSString *error = @"error deleting file";
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	
	[self updateDiskSpace];
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)downloadTrack:(Track *)track
{
	// check to see we're not going to overwrite anything
	NSString *fullPath = [track fullPath];
	fullPath = [self uniqueFilename:fullPath];
	[track setFullPath:fullPath];
	
	NSDate *date = [NSDate date];
	
	if (!mtpDevice)
	{
		// set turbo on/off
		[self enableTurbo];
	
		if (NJB_Get_Track(njb, [track itemID], [track filesize], [[track fullPath] fileSystemRepresentation], progress, NULL) == -1)
		{
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	else
	{
		int ret = get_file(&params, [track itemID], [[track fullPath] fileSystemRepresentation], &progressPTP);
		if (ret != 0)
		{
			NSString *error = @"Error downloading file";
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	
	double time = -[date timeIntervalSinceNow];
	// rate in Mbps
	double rate = ((double)[track filesize] / time) * 8.0 / (1024.0 * 1024.0);
	
	return [[[NJBTransactionResult alloc] initWithSuccess:YES
																					 resultString:[NSString stringWithFormat:NSLocalizedString(@"Speed %@ Mbps", nil), [self rateString:rate]]] autorelease];
}

- (NSString *)uniqueFilename:(NSString *)path
{
	struct stat sb;
	while (stat([path fileSystemRepresentation], &sb) != -1)
	{
		NSString *extension = [path pathExtension];
		NSString *pathWithoutExtension = [path stringByDeletingPathExtension];
		pathWithoutExtension = [pathWithoutExtension stringByAppendingString:@"_"];
		path = [pathWithoutExtension stringByAppendingPathExtension:extension];
	}
	return path;
}

- (NJBTransactionResult *)changeTrackTagTo:(Track *)newTrack from:(Track *)oldTrack
{
	if (!mtpDevice)
	{
		njb_songid_t *songid = [self songidStructFromTrack:newTrack];
		
		if (NJB_Replace_Track_Tag(njb, [newTrack itemID], songid) == -1)
		{
			NSString *error = [self njbErrorString];
			NSLog(error);
			NJB_Songid_Destroy(songid);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
		else
		{
			NJB_Songid_Destroy(songid);
		}
	}
	else
	{
		if (![self setMTPTrackTag:newTrack])
		{
			NSString *error = @"Could not update track info";
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	
	[self updateDiskSpace];
	
	// update my cache
	[cachedTrackList replaceObject:oldTrack withObject:newTrack];
	
	// tell others the track list has changed
	[statusDisplayer postNotificationName:NOTIFICATION_TRACK_LIST_MODIFIED object:self];
	
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (BOOL)setMTPTrackTag:(Track *)track
{
	if (!connected || !mtpDevice)
		return NO;
	
	int ret = 0;
	// todo: check all return codes
	unsigned short *unicode = NULL;
	unicode = [UnicodeWrapper UTF16FromString:[track title]];
	ret = ptp_setobjectpropvalue(&params, PTP_OPC_Name, [track itemID], unicode, PTP_DTC_UNISTR);
	free(unicode);
	unicode = [UnicodeWrapper UTF16FromString:[track album]];
	ret = ptp_setobjectpropvalue(&params, PTP_OPC_AlbumName, [track itemID], unicode, PTP_DTC_UNISTR);
	free(unicode);
	unicode = [UnicodeWrapper UTF16FromString:[track artist]];
	ret = ptp_setobjectpropvalue(&params, PTP_OPC_Artist, [track itemID], unicode, PTP_DTC_UNISTR);
	free(unicode);
	unicode = [UnicodeWrapper UTF16FromString:[track genre]];
	ret = ptp_setobjectpropvalue(&params, PTP_OPC_Genre, [track itemID], unicode, PTP_DTC_UNISTR);
	free(unicode);
	uint32_t durationMS = [track length] * 1000;
	ret = ptp_setobjectpropvalue(&params, PTP_OPC_Duration, [track itemID], &durationMS, PTP_DTC_UINT32);
	uint16_t trackNumber = [track trackNumber];
	ret = ptp_setobjectpropvalue(&params, PTP_OPC_Track, [track itemID], &trackNumber, PTP_DTC_UINT16);

	NSString *nsTimeString = [NSString stringWithFormat:@"%.04d0101T000000.0", [track year]];
	const char *timeString = [nsTimeString cString];
	ret = ptp_setobjectpropvalue(&params, PTP_OPC_OriginalReleaseDate, [track itemID], (void*)timeString, PTP_DTC_STR);
	
	if (ret != PTP_RC_OK)
		return NO;
	else
		return YES;
}

- (Directory *)dataFiles
{
	if (![self isConnected])
		return nil;
	
	if (!mtpDevice)
	{
		Directory *baseDir = [[Directory alloc] init];
		NJB_Reset_Get_Datafile_Tag(njb);
		njb_datafile_t *filetag;
		while (filetag = NJB_Get_Datafile_Tag(njb))
		{
			NSString *filename = @"";
			if (filetag->filename != NULL)
				filename = [NSString stringWithCString:filetag->filename];
			NSArray *pathArray = nil;
			if (filetag->folder != NULL)
			{
				NSString *folder = [NSString stringWithCString:filetag->folder];
				// this has \\ instead of /
				pathArray = [folder componentsSeparatedByString:@"\\"];
				// the / on the beginning and end gives us extra empty objects
				NSMutableArray *newPathArray = [NSMutableArray arrayWithArray:pathArray];
				if ([[newPathArray objectAtIndex:[newPathArray count] - 1] length] == 0)
					[newPathArray removeLastObject];
				if ([[newPathArray objectAtIndex:0] length] == 0)
					[newPathArray removeObjectAtIndex:0];
				pathArray = newPathArray;
			}

			if ([filename isEqualToString:@"."])
			{
				// this is an empty folder
				NSMutableArray *parentPathArray = [NSMutableArray arrayWithArray:pathArray];
				[parentPathArray removeLastObject];
				
				NSString *name = [pathArray objectAtIndex:[pathArray count] - 1];

				Directory *parentDir = [baseDir itemWithPath:parentPathArray];
				if (![parentDir containsItemWithName:name])
				{
					Directory *newDir = [[Directory alloc] initWithName:name];
					[newDir setItemID:filetag->dfid];
				
					[baseDir addItem:newDir toDir:parentPathArray];
				
					[newDir release];
				}
			}
			else
			{
				// this is a file
				DataFile *dataFile = [[DataFile alloc] init];
				[dataFile setFilename:filename];
				[dataFile setSize:filetag->filesize];
				// [dataFile setTimestampSince1970:filetag->timestamp];
				[dataFile setItemID:filetag->dfid];
				
				[baseDir addItem:dataFile toDir:pathArray];
				
				[dataFile release];
			}
			
			NJB_Datafile_Destroy(filetag);
		}
		
		return [baseDir autorelease];
	}
	else
	{
		PTPObjectInfo oi;
		
		Directory *baseDir = [[Directory alloc] init];
		
		if (ptp_getobjecthandles(&params,PTP_GOH_ALL_STORAGE, PTP_GOH_ALL_FORMATS, PTP_GOH_ALL_ASSOCS, &params.handles) != PTP_RC_OK)
		{
			// todo: print error
			NSLog(@"Could not get object handles...");
			return nil;
		}
		int i = 0;
		for (i = 0; i < params.handles.n; i++) {
			if (ptp_getobjectinfo(&params,params.handles.Handler[i], &oi) == PTP_RC_OK)
			{
				if (oi.ObjectFormat == PTP_OFC_Association ||	oi.ObjectFormat == PTP_OFC_WAV || oi.ObjectFormat == PTP_OFC_MP3 || oi.ObjectFormat == PTP_OFC_WMA)
					continue;
				
				DataFile *dataFile = [[DataFile alloc] init];
				if (oi.Filename != NULL)
					[dataFile setFilename:[NSString stringWithUTF8String:oi.Filename]];
				[dataFile setSize:oi.ObjectCompressedSize];
				[dataFile setItemID:params.handles.Handler[i]];
				
				[baseDir addItem:dataFile toDir:nil];
				
				[dataFile release];
			}
		}
		
		return [baseDir autorelease];
	}
}

- (NJBTransactionResult *)downloadFile:(DataFile *)dataFile
{
	// check to see we're not going to overwrite anything
	NSString *fullPath = [dataFile fullPath];
	fullPath = [self uniqueFilename:fullPath];
	[dataFile setFullPath:fullPath];
		
	NSDate *date = [NSDate date];
		
	if (!mtpDevice)
	{
		// set turbo on/off
		[self enableTurbo];

		if (NJB_Get_File(njb, [dataFile itemID], [dataFile size], [[dataFile fullPath] fileSystemRepresentation], progress, NULL) == -1)
		{
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	else
	{
		int ret = get_file(&params, [dataFile itemID], [[dataFile fullPath] fileSystemRepresentation], &progressPTP);
		if (ret != 0)
		{
			NSString *error = @"Error downloading file";
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	
	double time = -[date timeIntervalSinceNow];
	// rate in Mbps
	double rate = ((double)[dataFile size] / time) * 8.0 / (1024.0 * 1024.0);
		
	return [[[NJBTransactionResult alloc] initWithSuccess:YES
																					 resultString:[NSString stringWithFormat:NSLocalizedString(@"Speed %@ Mbps", nil), [self rateString:rate]]] autorelease];
}

- (NJBTransactionResult *)uploadFile:(DataFile *)dataFile toFolder:(NSString *)path
{
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	
	NSDate *date = [NSDate date];
	[self setStatus:STATUS_UPLOADING_FILE];

	if (!mtpDevice)
	{
		// set turbo on/off
		[self enableTurbo];
	
		unsigned int fileid;
		if (NJB_Send_File(njb, [[dataFile fullPath] fileSystemRepresentation], [[[dataFile fullPath] lastPathComponent] UTF8String], [path UTF8String], progress, NULL, &fileid) == -1 ) {
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		} else {
			[dataFile setItemID:fileid];
		}
	}
	else
	{
		uint16_t objectFormat = PTP_OFC_Undefined;
		
		// a test to see how the player works with video & images
		NSString *extension = [[[dataFile fullPath] pathExtension] lowercaseString];
		if ([extension isEqualToString:@"wmv"])
			objectFormat = PTP_OFC_WMV;
		else if ([extension isEqualToString:@"avi"])
			objectFormat = PTP_OFC_AVI;
		else if ([extension isEqualToString:@"mpg"] || [extension isEqualToString:@"mpeg"])
			objectFormat = PTP_OFC_MPEG;
		else if ([extension isEqualToString:@"asf"])
			objectFormat = PTP_OFC_ASF;
		else if ([extension isEqualToString:@"qt"])
			objectFormat = PTP_OFC_QT;
		else if ([extension isEqualToString:@"jpg"] || [extension isEqualToString:@"jpeg"])
			objectFormat = PTP_OFC_JFIF; // or should this be PTP_OFC_EXIF_JPEG?
		else if ([extension isEqualToString:@"tif"] || [extension isEqualToString:@"tiff"])
			objectFormat = PTP_OFC_TIFF;
		else if ([extension isEqualToString:@"bmp"])
			objectFormat = PTP_OFC_BMP;
		else if ([extension isEqualToString:@"gif"])
			objectFormat = PTP_OFC_GIF;
		else if ([extension isEqualToString:@"pict"])
			objectFormat = PTP_OFC_PICT;
		else if ([extension isEqualToString:@"png"])
			objectFormat = PTP_OFC_PNG;
		
		NSLog(@"uploading file with object format 0x%.02x", objectFormat);
		
		uint32_t handle = 0;
		uint16_t ret = send_file(&params, [[dataFile fullPath] fileSystemRepresentation], [[[dataFile fullPath] lastPathComponent] fileSystemRepresentation],
														 objectFormat, progressPTP, &handle);
		if (ret != PTP_RC_OK)
		{
			/* we might get an (MTP error) object too large here */
			NSString *error = @"Error uploading file";
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
		
		[dataFile setItemID:handle];
	}
	
	[self updateDiskSpace];
		
	double time = -[date timeIntervalSinceNow];
	// rate in Mbps
	double rate = ((double)[dataFile size] / time) * 8.0 / (1024.0 * 1024.0);
	return [[[NJBTransactionResult alloc] initWithSuccess:YES
																					 resultString:[NSString stringWithFormat:NSLocalizedString(@"Speed %@ Mbps", nil), [self rateString:rate]]] autorelease];	
}

- (unsigned long long)totalDiskSpace
{
	return totalDiskSpace;
}

- (unsigned long long)freeDiskSpace
{
	return freeDiskSpace;
}

- (void)updateDiskSpace
{
	if (!mtpDevice)
	{
		NJB_Get_Disk_Usage(njb, &totalDiskSpace, &freeDiskSpace);
		[statusDisplayer updateDiskSpace:totalDiskSpace withFreeSpace:freeDiskSpace];
	}
	else
	{
		PTPStorageInfo storageInfo;
		if (ptp_getstorageinfo(&params, storageID, &storageInfo) == PTP_RC_OK)
		{
			freeDiskSpace = storageInfo.FreeSpaceInBytes;
			totalDiskSpace = storageInfo.MaxCapability;
			
			if (storageInfo.StorageDescription != NULL)
				free(storageInfo.StorageDescription);
			if (storageInfo.VolumeLabel != NULL)
				free(storageInfo.VolumeLabel);
			
			[statusDisplayer updateDiskSpace:totalDiskSpace withFreeSpace:freeDiskSpace];
		}
	}
}

- (NSCalendarDate *)jukeboxTime
{
	if (!mtpDevice)
	{
		njb_time_t *time = NJB_Get_Time(njb);
		if (time == NULL)
			return nil;
	
		// assume njb time is our timezone
		NSCalendarDate *date = [NSCalendarDate dateWithYear:time->year month:time->month day:time->day hour:time->hours minute:time->minutes second:time->seconds
																							 timeZone:[NSTimeZone localTimeZone]];
		return date;
	}
	else
	{
		// note Creative devices seem not to support this
		
		if (!ptp_property_issupported(&params, PTP_DPC_DateTime))
			return nil;
		
		char *timeString = NULL;
		if (ptp_getdevicepropvalue(&params, PTP_DPC_DateTime, (void **)&timeString, PTP_DTC_STR) != PTP_RC_OK)
			return nil;
		
		if (timeString != NULL)
		{
			NSString *nsTimeString = [NSString stringWithCString:timeString];
			free(timeString);
			NSCalendarDate *date = [[NSCalendarDate alloc] initWithString:nsTimeString calendarFormat:@"%Y%m%dT%H%M%S%z"];
		
			return [date autorelease];
		}
		else
			return nil;
	}
}

- (BOOL)isProtocol3Device
{
	if (!mtpDevice)
		return (PDE_PROTOCOL_DEVICE(njb));
	else
		return NO;
}

- (NJBTransactionResult *)setOwnerString:(NSString *)owner
{
	if (!mtpDevice)
	{
		if (NJB_Set_Owner_String (njb, [owner UTF8String]) == -1)
		{
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	else
	{
		unsigned short *unistring = [UnicodeWrapper UTF16FromString:owner];
		uint16_t ret = ptp_setdevicepropvalue(&params, PTP_DPC_DeviceFriendlyName, unistring, PTP_DTC_UNISTR);
		free(unistring);
		if (ret != PTP_RC_OK)
		{
			NSString *error = [NSString stringWithFormat:@"Could not set device friendly name, return value 0x%04.x", ret];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)setBitmap:(NSString *)bitmapPath
{
	if (mtpDevice)
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	
	if (NJB_Set_Bitmap(njb, [bitmapPath UTF8String]) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)setTime:(NSNumber *)timeIntervalSinceNow
{
	if (mtpDevice)
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	
	njb_time_t time;
	
	NSTimeInterval jukeboxTimeInterval = [timeIntervalSinceNow doubleValue];
	NSCalendarDate *date = [NSCalendarDate dateWithTimeIntervalSinceNow:jukeboxTimeInterval];
	
	time.year = [date yearOfCommonEra];
	time.month = [date monthOfYear];
	time.day = [date dayOfMonth];
	time.weekday = [date dayOfWeek];
	time.hours = [date hourOfDay];
	time.minutes = [date minuteOfHour];
	time.seconds = [date secondOfMinute];
	
	if (NJB_Set_Time(njb, &time) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NSMutableArray *)playlists
{
	if (![self isConnected])
		return nil;
	
	NSMutableArray *playlists = [[NSMutableArray alloc] init];
	
	if (!mtpDevice)
	{
		NJB_Reset_Get_Playlist(njb);
		njb_playlist_t *playlist;
		while ((playlist = NJB_Get_Playlist(njb)))
		{
			Playlist *newPlaylist = [[Playlist alloc] init];
					
			const char *str = playlist->name;
			if (str != NULL)
				[newPlaylist setName:[NSString stringWithUTF8String:str]];
			[newPlaylist setPlaylistID:playlist->plid];
			
			njb_playlist_track_t *track;
			NJB_Playlist_Reset_Gettrack(playlist);
			while ((track = NJB_Playlist_Gettrack(playlist)))
			{
				Track *newTrack = [[Track alloc] init];
				[newTrack setItemID:track->trackid];
				[newPlaylist addTrack:newTrack];
				[newTrack release];
			}
			
			NJB_Playlist_Destroy(playlist);
			
			[newPlaylist setState:NJB_PL_UNCHANGED];
			
			[playlists addObject:newPlaylist];
			[newPlaylist release];
		}
	}
	else
	{
		PTPObjectInfo oi;
		
		// todo: can we just ask for PTP_OFC_AbstractAudioVideoPlaylist type?
		if (ptp_getobjecthandles(&params,PTP_GOH_ALL_STORAGE, PTP_GOH_ALL_FORMATS, PTP_GOH_ALL_ASSOCS, &params.handles) != PTP_RC_OK)
		{
			// todo: print error
			NSLog(@"Could not get playlist object handles...");
			return nil;
		}
		int i = 0;
		for (i = 0; i < params.handles.n; i++) {
			if (ptp_getobjectinfo(&params,params.handles.Handler[i], &oi) == PTP_RC_OK)
			{
				if (oi.ObjectFormat != PTP_OFC_AbstractAudioVideoPlaylist)
					continue;
				
				Playlist *newPlaylist = [[Playlist alloc] init];
				
				unsigned short *unicodevalue = NULL;
				uint16_t ret = 0;
				ret = ptp_getobjectpropvalue(&params, PTP_OPC_Name, params.handles.Handler[i], (void**)&unicodevalue, PTP_DTC_UNISTR);
				if (ret == PTP_RC_OK && unicodevalue != NULL)
				{
					[newPlaylist setName:[UnicodeWrapper stringFromUTF16:unicodevalue]];
					free(unicodevalue);
					unicodevalue = NULL;
				}
				
				uint32_t* ohArray = NULL;
				uint32_t arrayLen = 0;
				if (ptp_getobjectreferences (&params, params.handles.Handler[i], &ohArray, &arrayLen) == PTP_RC_OK)
				{
					int j = 0;
					for (j = 0; j<arrayLen; j++)
					{
						Track *newTrack = [[Track alloc] init];
						[newTrack setItemID:ohArray[j]];
						[newPlaylist addTrack:newTrack];
						[newTrack release];
					}
					free(ohArray);
				}
				else
				{
					NSLog(@"Could not get playlist tracks for playlist handle 0x%08.x", params.handles.Handler[i]);
				}
				
				[newPlaylist setPlaylistID:params.handles.Handler[i]];
				
				// todo: worry about playlist statuses
				[newPlaylist setState:NJB_PL_UNCHANGED];
				
				[playlists addObject:newPlaylist];
				[newPlaylist release];
			}
		}
	}
	return [playlists autorelease];
}

/* updates the playlist on the NJB
 * may add a new playlist even if playlist state is not new
 * so the playlist ID may change.
 */
- (NJBTransactionResult *)updatePlaylist:(Playlist *)playlist
{
	[playlist lock];
	// don't do anything if it's unchanged
	if ([playlist state] == NJB_PL_UNCHANGED)
	{
		[playlist unlock];
		return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
	}
	
	// the playlist is locked here
	if (!mtpDevice)
	{
		// create new njb_playlist_t object
		njb_playlist_t *pl = NJB_Playlist_New();
		NJB_Playlist_Set_Name(pl, [[playlist name] UTF8String]);
		pl->plid = [playlist playlistID];
		NSEnumerator *enumerator = [[playlist tracks] objectEnumerator];
		Track *currentTrack;
		while (currentTrack = [enumerator nextObject])
		{
			njb_playlist_track_t *newTrack = NJB_Playlist_Track_New([currentTrack itemID]);
			NJB_Playlist_Addtrack(pl, newTrack, NJB_PL_END);
		}
		pl->_state = [playlist state];
		// set this now as as far as the caller is concerned the NJB is updated
		[playlist setState:NJB_PL_UNCHANGED];
		[playlist unlock];
		if (NJB_Update_Playlist(njb, pl) == -1)
		{
			NJB_Playlist_Destroy(pl);
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
		[playlist lock];
		// update the ID as this is changed when tracks added/removed as a new playlist is created
		[playlist setPlaylistID:pl->plid];
		[playlist unlock];
		NJB_Playlist_Destroy(pl);
	}
	else
	{
		NSArray *tracks = [playlist tracks];
		uint32_t size = [tracks count];
		uint32_t *trackArray = (uint32_t*)malloc(sizeof(uint32_t)*size);
		uint32_t i = 0;
		uint16_t ret = 0;
		for (i = 0; i < size; i++)
			trackArray[i] = [[tracks objectAtIndex:i] itemID];
		// store the ID so we can unlock
		uint32_t playlistID = [playlist playlistID];
		
		int playlistState = [playlist state];
		[playlist setState:NJB_PL_UNCHANGED];
		[playlist unlock];
		
		if (playlistState == NJB_PL_NEW)
		{
			/* note that we really want to create a new playlist not by using
			SendObjectInfo but with the enhanced SendObjectPropList. Then we can send
			filesize 0 (we still have to send object, but with no data). Sending filesize
			0 with SendObjectInfo gives error PTP_RC_StoreFull. So hopefully we can persuade
			Microsoft to let us use SendObjectPropList. */
			
			PTPObjectInfo newInfo;
			memset(&newInfo, 0, sizeof(newInfo));
			[playlist lock];
			newInfo.Filename = [[NSString stringWithFormat:@"%@.zpl", [playlist name]] cString];
			[playlist unlock];
			newInfo.ObjectFormat = PTP_OFC_AbstractAudioVideoPlaylist;
			newInfo.ObjectCompressedSize = 1;
			
			uint32_t store = 0;
			uint32_t parenthandle = 0;
			
			ret = ptp_sendobjectinfo(&params, &store, &parenthandle, &playlistID, &newInfo);
			if (ret != PTP_RC_OK)
			{
				free(trackArray);
				return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:
					[NSString stringWithFormat:@"Could not create new playlist, error 0x%.04x", ret]] autorelease];
			}
			char *data='\0';
			ret = ptp_sendobject(&params, data, 1);
			if (ret != PTP_RC_OK)
			{
				free(trackArray);
				return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:
					[NSString stringWithFormat:@"Could not create new playlist data, error 0x%.04x", ret]] autorelease];
			}
			
			[playlist lock];
			[playlist setPlaylistID:playlistID];
			[playlist unlock];
		}
		
		if (playlistState == NJB_PL_NEW || playlistState == NJB_PL_CHNAME)
		{
			unsigned short *unicode = NULL;
			[playlist lock];
			unicode = [UnicodeWrapper UTF16FromString:[playlist name]];
			[playlist unlock];
			ret = ptp_setobjectpropvalue(&params, PTP_OPC_Name, playlistID, unicode, PTP_DTC_UNISTR);
			free(unicode);
			if (ret != PTP_RC_OK)
			{
				free(trackArray);
				return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:
					[NSString stringWithFormat:@"Could not set name for new playlist, error 0x%.04x", ret]] autorelease];
			}
		}
		
		ret = ptp_setobjectreferences(&params, playlistID, trackArray, size);
		if (ret != PTP_RC_OK)
		{
			free(trackArray);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:
				[NSString stringWithFormat:@"Could not update playlist, error 0x%.04x", ret]] autorelease];
		}
		free(trackArray);
	}
	
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)deletePlaylist:(Playlist *)playlist
{	
	[playlist lock];
	unsigned plid = [playlist playlistID];
	[playlist unlock];
	
	if (!mtpDevice)
	{
		if (NJB_Delete_Playlist(njb, plid) == -1)
		{
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	else
	{
		int ret = ptp_deleteobject(&params, [playlist playlistID],0);
		if (ret != PTP_RC_OK)
		{
			NSString *error = @"error deleting playlist";
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

/* called by the file/track transfer functions of libnjb
 * to tell us the progress
 */
int progress(u_int64_t sent, u_int64_t total, const char* buf, unsigned len, void *data)
{
	double percent = (double)sent*100.0/(double)total;
	[statusDisplayerGlobal updateTaskProgress:percent];
	
	return 0;
}

/* called by the file/track transfer functions of libptp
 * to tell us the progress
 */
int progressPTP(u_int32_t sent, u_int32_t total)
{
	double percent = (double)sent*100.0/(double)total;
	[statusDisplayerGlobal updateTaskProgress:percent];
	
	return 0;
}

- (njb_songid_t *)songidStructFromTrack:(Track *)track
{
	njb_songid_t *songid;
	njb_songid_frame_t *frame;
	
	songid = NJB_Songid_New();
	if ([track njbCodec])
	{
		frame = NJB_Songid_Frame_New_Codec([[track njbCodec] UTF8String]);
		NJB_Songid_Addframe(songid, frame);
	}
	// only add file size if non zero
	if ([track filesize] != 0)
	{
		frame = NJB_Songid_Frame_New_Filesize([track filesize]);
		NJB_Songid_Addframe(songid, frame);
	}
	if ([track title])
	{
		frame = NJB_Songid_Frame_New_Title([[track title] UTF8String]);
		NJB_Songid_Addframe(songid, frame);
	}
	if ([track album])
	{
		frame = NJB_Songid_Frame_New_Album([[track album] UTF8String]);
		NJB_Songid_Addframe(songid, frame);
	}
	if ([track artist])
	{
		frame = NJB_Songid_Frame_New_Artist([[track artist] UTF8String]);
		NJB_Songid_Addframe(songid, frame);
	}
	if ([track genre])
	{
		frame = NJB_Songid_Frame_New_Genre([[track genre] UTF8String]);
		NJB_Songid_Addframe(songid, frame);
	}
	frame = NJB_Songid_Frame_New_Year([track year]);
	NJB_Songid_Addframe(songid, frame);
	frame = NJB_Songid_Frame_New_Tracknum([track trackNumber]);
	NJB_Songid_Addframe(songid, frame);
	frame = NJB_Songid_Frame_New_Length([track length]);
	NJB_Songid_Addframe(songid, frame);
	if ([track fullPath] && [[track fullPath] length] > 0)
	{
		frame = NJB_Songid_Frame_New_Filename([[track fullPath] fileSystemRepresentation]);
		NJB_Songid_Addframe(songid, frame);
	}
	
	return songid;
}

- (NSMutableArray *)cachedTrackList
{
	if (cachedTrackList && !downloadingTracks)
	{
		NSMutableArray *tracks = [[NSMutableArray alloc] initWithArray:cachedTrackList];
		return [tracks autorelease];
	}
	else
		return nil;
}

/* returns a string rounded to 1dp for
 * the rate given
 */
- (NSString *)rateString:(double) rate
{
	int rate10 = rate * 10;
	if (rate - (double)rate10/10.0 >= 0.05)
		rate10++;
	// the two digits
	int rateA = rate10/10;
	int rateB = rate10-rateA*10;
	return [NSString stringWithFormat:@"%d.%d", rateA, rateB];
}

- (NJBTransactionResult *)createFolder:(Directory *)dir inDir:(NSString *)path
{
	if (mtpDevice)
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	
	unsigned int folderID;
	if (NJB_Create_Folder(njb, [[NSString stringWithFormat:@"%@%@", path, [dir name]] UTF8String], &folderID) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	[dir setItemID:folderID];
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (void)storeDeviceString
{
	if (!mtpDevice)
	{
		const char *name = NJB_Get_Device_Name(njb, 0);
		if (name != NULL)
			deviceString = [[NSString alloc] initWithCString:name];
		else
			deviceString = nil;
	}
	else
		deviceString = @"";
}

- (void)storeFirmwareVersionString
{
	if (!mtpDevice)
	{
		u_int8_t major, minor, release;
			
		int ret = NJB_Get_Firmware_Revision(njb, &major, &minor, &release);
		if (ret == 0)
			firmwareVersionString = [[NSString alloc] initWithString:[NSString stringWithFormat:@"%u.%u.%u", major, minor, release]];
		else
			deviceVersionString = nil;
	}
	else
		firmwareVersionString = @"";
}

- (void)storeDeviceIDString
{	
	if (!mtpDevice)
	{
		NSMutableString *idString = [[NSMutableString alloc] initWithCapacity:32];
		int j;
		
		u_int8_t sdmiid[16];
		int result;
		result = NJB_Get_SDMI_ID(njb, &(sdmiid[0]));
		if (result != 0)
			idString = @"";
		
		for (j = 0; j < 16; j++)
		{
			if (sdmiid[j] > 15)
				[idString appendString:[NSString stringWithFormat:@"%X", sdmiid[j]]];
			else
				[idString appendString:[NSString stringWithFormat:@"0%X", sdmiid[j]]];
		}
		
		deviceIDString = [[NSString alloc] initWithString:[NSString stringWithString:idString]];
		[idString release];
	}
	else
		deviceIDString = @"";
}

- (void)storeDeviceVersionString
{
	if (!mtpDevice)
	{
		u_int8_t major, minor, release;
			
		int ret = NJB_Get_Hardware_Revision(njb, &major, &minor, &release);
		if (ret == 0)
			deviceVersionString = [[NSString alloc] initWithString:[NSString stringWithFormat:@"%u.%u.%u", major, minor, release]];
		else
			deviceVersionString = nil;
	}
	else
		deviceVersionString = @"";
}

- (int)batteryLevel
{
	if (connected == NO)
		return 0;
	if (!mtpDevice)
	{
		return NJB_Get_Battery_Level(njb);
	}
	else
	{				
		uint8_t *value = NULL;
		if (ptp_getdevicepropvalue(&params,PTP_DPC_BatteryLevel,(void**)&value,PTP_DTC_UINT8) != PTP_RC_OK)
		{
			NSLog(@"Could not get battery level property value");
			return 0;
		}
				
		if (value != NULL)
		{
			int level = *value;
			free(value);
			return level;
		}
		else
			return 0;
	}
}

- (NSString *)batteryStatus
{
	if (!mtpDevice)
	{
		BOOL charging = (NJB_Get_Battery_Charging(njb) == 1);
		BOOL powerConnected = (NJB_Get_Auxpower(njb) == 1);
		
		if (charging)
			return NSLocalizedString(@"Charging", nil);
		if (powerConnected)
			return NSLocalizedString(@"Battery charged or not present", nil);
		return NSLocalizedString(@"Running off battery", nil);
	}
	else
	{
		// they always charge...
		return @"Charging";
	}
}

- (void)enableTurbo
{
	//NSLog(@"enableTurbo : %d", turbo);

	if (turbo)
		NJB_Set_Turbo_Mode(njb, NJB_TURBO_ON);
	else
		NJB_Set_Turbo_Mode(njb, NJB_TURBO_OFF);
}

@end
