//
//  NJB.m
//  XNJB
//
//  Created by Richard Low on Wed Jul 21 2004.
//

// todo: can we find out how many tracks there are before we download for progress bar & set capacity in NSMutableArray?

/* An obj-c wrapper for libnjb
 * most methods return an NJBTransactionResult
 * to indicate success/failure
 */

#import "NJB.h"
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#import "Track.h"
#include "string.h"
#include "base.h"
#include "njb_error.h"
#import "defs.h"
#include <sys/stat.h>
#include <unistd.h>
#import "UnicodeWrapper.h"

// number of tracks in tracklist downloaded in between progress bar updates (MTP only)
#define TRACKLIST_PROGRESS_UPDATE_INTERVAL	50

// declared so can be accessed from C function progress
// set to be the outlet statusDisplayer in awakeFromNib
StatusDisplayer *statusDisplayerGlobal;

// declare the private methods
@interface NJB (PrivateAPI)
- (void)updateDiskSpace;
- (njb_songid_t *)songidStructFromTrack:(Track *)track;
- (NSString *)njbErrorString;
- (NSString *)rateString:(double) rate;
- (NSString *)uniqueFilename:(NSString *)path;
- (LIBMTP_track_t *)libmtpTrack_tFromTrack:(Track *)track;
// we make this private so that the user never changes the setting directly
// so we only change it before a transfer, so it is not changed during a transfer
// which could mess things up
- (void)enableTurbo;

@end

@implementation NJB

// init/dealloc methods

- (id)init
{
	if (self = [super init])
	{
		njb = NULL;
		// 15 is max (this is for libnjb devices)
		[self setDebug:0];
		[self setTurbo:YES];
		// this defined here for the non MTP devices
		batteryLevelMax = 100;
		
		downloadingTracks = NO;
		downloadingFiles = NO;
	}
	return self;
}

- (void)dealloc
{
	[statusDisplayerGlobal release];
	[cachedTrackList release];
	[cachedDataFileList release];
	
	[deviceString release];
	[firmwareVersionString release];
	[deviceIDString release];
	[deviceVersionString release];
	
	[super dealloc];
}

// accessor methods

- (void)setDebug:(unsigned)newDebug
{
	// see libnjb.h for the Debug flags
	debug = newDebug;
}

- (statusTypes)status
{
	return status;
}

- (void)setStatus:(statusTypes)newStatus
{
	status = newStatus;
	[statusDisplayer setStatus:status];
}

- (BOOL)isConnected
{
	return connected;
}

- (NSString *)deviceString
{
	return deviceString;
}

- (NSString *)firmwareVersionString
{
	return firmwareVersionString;
}

- (NSString *)deviceIDString
{
	return deviceIDString;
}

- (NSString *)deviceVersionString
{
	return deviceVersionString;
}

- (void)setTurbo:(BOOL)newTurbo
{
	turbo = newTurbo;
}

- (int)batteryLevelMax
{
	return batteryLevelMax;
}

- (uint32_t)playlistsAssoc
{
	if (device != NULL)
		return device->default_playlist_folder;
	else
		return 0;
}

- (uint32_t)musicAssoc
{
	if (device != NULL)
		return device->default_music_folder;
	else
		return 0;
}

- (uint32_t)picturesAssoc
{
	if (device != NULL)
		return device->default_picture_folder;
	else
		return 0;
}

- (uint32_t)videoAssoc
{
	if (device != NULL)
		return device->default_video_folder;
	else
		return 0;
}

- (uint32_t)zencastAssoc
{
	if (device != NULL)
		return device->default_zencast_folder;
	else
		return 0;
}

/**********************/

- (void)awakeFromNib
{
	statusDisplayerGlobal = [statusDisplayer retain];
}

- (NSString *)njbErrorString
{
	const char *sp;
	NSMutableString *errorString = [[NSMutableString alloc] init];
	NJB_Error_Reset_Geterror(njb);
  while ((sp = NJB_Error_Geterror(njb)) != NULL)
	{
		[errorString appendString:[NSString stringWithFormat:@"%s, ", sp]];
  }
  njb_error_clear(njb);
	
	return [errorString autorelease];
}

/* connect to the NJB
 */
- (NJBTransactionResult *)connect
{	
	int n;
	
	connected = NO;
	NSString *errorString = nil;
	
	// try for PDE devices first
	
	if (debug)
		NJB_Set_Debug(debug);
	
	if (NJB_Discover(njbs, 0, &n) == -1)
	{
		[self setStatus:STATUS_NO_NJB];
		errorString = NSLocalizedString(@"Could not discover any jukeboxes", nil);
	}
	else
	{
		if (n == 0)
		{
			[self setStatus:STATUS_NO_NJB];
			errorString = NSLocalizedString(@"Could not locate any jukeboxes", nil);
		}
		else
		{
			njb = &njbs[0];
	
			if (NJB_Open(njb) == -1)
			{
				[self setStatus:STATUS_COULD_NOT_OPEN];
				errorString = [self njbErrorString];
			}
			else
			{
				if (NJB_Capture(njb) == -1)
				{
					[self setStatus:STATUS_COULD_NOT_CAPTURE];
					errorString = [self njbErrorString];
					[self disconnect];
				}
				else
				{
					mtpDevice = NO;
					connected = YES;
					[self setStatus:STATUS_CONNECTED];
				}
			}
		}
	}

	if (!connected)
	{
		mtpDevice = YES;
	  
		LIBMTP_Init();
		device = LIBMTP_Get_First_Device();
		
		if (device == NULL)
		{
			connected = NO;
			errorString = NSLocalizedString(@"Could not locate any jukeboxes", nil);
			[self setStatus:STATUS_NO_NJB];
			
			// what about couldn't connect?
			/*
			 errorString = NSLocalizedString(@"Could not talk to jukebox", nil);
			 [self setStatus:STATUS_COULD_NOT_OPEN];
			 */
		}
		else
		{
			connected = YES;
			[self setStatus:STATUS_CONNECTED];
		}
	}
	
	if (!connected)
	{
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:errorString] autorelease]; 
	}
	
	[self updateDiskSpace];
	
	// enable UTF8
	if (!mtpDevice)
		NJB_Set_Unicode(NJB_UC_UTF8);
	
	// check we have reset the cache, should have been done on disconnect
	[cachedTrackList release];
	cachedTrackList = nil;
	[cachedDataFileList release];
	cachedDataFileList = nil;
	
	if (mtpDevice)
	{
		char *tempString =  LIBMTP_Get_Modelname(device);
		deviceString = [NSString stringWithUTF8String:tempString];
		free(tempString);

		tempString = LIBMTP_Get_Serialnumber(device);
		deviceIDString = [NSString stringWithUTF8String:tempString];
		free(tempString);

		tempString = LIBMTP_Get_Deviceversion(device);
		NSString *versionString = [NSString stringWithUTF8String:tempString];
		free(tempString);
		
		NSArray *versions = [versionString componentsSeparatedByString:@"_"];
		if ([versions count] != 2)
		{
			deviceVersionString = versionString;
			firmwareVersionString = @"";
		}
		else
		{
			deviceVersionString = [versions objectAtIndex:1];
			firmwareVersionString = [versions objectAtIndex:0];
		}
		
		// unneeded
		uint8_t currentLevel = 0;
		if (LIBMTP_Get_Batterylevel(device, &batteryLevelMax, &currentLevel) != 0)
		{
			NSLog(@"Could not get battery level property description");
			return 0;
		}
		
		// download the data files to the cache
		//[self dataFiles];
	}
	else
	{
		[self storeDeviceString];
		[self storeFirmwareVersionString];
		[self storeDeviceIDString];
		[self storeDeviceVersionString];
	}

	// send notification, make sure this is last so all variables are set above
	[statusDisplayer postNotificationName:NOTIFICATION_CONNECTED object:self];
		
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

/* disconnect from the NJB
 */
- (void)disconnect
{
	if ([self isConnected])
	{
		if (!mtpDevice)
		{
			NJB_Release(njb);
			NJB_Close(njb);
		}
		else
		{
			LIBMTP_Release_Device(device);
		}
		connected = NO;
		[self setStatus:STATUS_DISCONNECTED];
		
		// lose the cache
		[cachedTrackList release];
		cachedTrackList = nil;
		[cachedDataFileList release];
		cachedDataFileList = nil;
		
		// send notification
		[statusDisplayer postNotificationName:NOTIFICATION_DISCONNECTED object:self];
	}
}

- (NSString *)productName
{
	if (njb == NULL)
		return nil;
	const char *name = NJB_Get_Device_Name(njb, 1);
	if (name != NULL)
		return [NSString stringWithCString:name];
	else
		return nil;
}

- (NSString *)ownerString
{
	if (!mtpDevice)
	{
		char *ownerString = NJB_Get_Owner_String(njb);
		if (ownerString == NULL)
			return nil;
		NSString *nsOwnerString = [[NSString alloc] initWithUTF8String:ownerString];
		free(ownerString);
		return [nsOwnerString autorelease];
	}
	else
	{
		char *friendlyName = LIBMTP_Get_Friendlyname(device);
		if (friendlyName != NULL)
		{
			NSString *deviceFriendlyName = [NSString stringWithUTF8String:friendlyName];
			free(friendlyName);
			return deviceFriendlyName;
		}
		else
			return @"";
	}
}

- (NSMutableArray *)tracks
{
	if (![self isConnected])
		return nil;
	if (cachedTrackList)
	{
		return [self cachedTrackList];
	}
	
	downloadingTracks = YES;
	cachedTrackList = [[NSMutableArray alloc] init];
	if (!mtpDevice)
	{
		NJB_Reset_Get_Track_Tag(njb);	
		njb_songid_t *songtag;
		njb_songid_frame_t *frame;
		while ((songtag = NJB_Get_Track_Tag(njb))) {
			Track *track = [[Track alloc] init];
			
			frame = NJB_Songid_Findframe(songtag, FR_TITLE);
			if (frame != NULL && frame->data.strval != NULL)
				[track setTitle:[NSString stringWithUTF8String:frame->data.strval]];
			
			frame = NJB_Songid_Findframe(songtag, FR_ALBUM);
			if (frame != NULL && frame->data.strval != NULL)
				[track setAlbum:[NSString stringWithUTF8String:frame->data.strval]];
			
			frame = NJB_Songid_Findframe(songtag, FR_ARTIST);
			if (frame != NULL && frame->data.strval != NULL)
				[track setArtist:[NSString stringWithUTF8String:frame->data.strval]];
			
			frame = NJB_Songid_Findframe(songtag, FR_GENRE);
			if (frame != NULL && frame->data.strval != NULL)
				[track setGenre:[NSString stringWithUTF8String:frame->data.strval]];
			
			// this is not used: we don't get extended track info from njb3
			// njb1 gets it but ignored
			frame = NJB_Songid_Findframe(songtag, FR_FNAME);
			if (frame != NULL && frame->data.strval != NULL)
				[track setFilename:[NSString stringWithUTF8String:frame->data.strval]];
			
			frame = NJB_Songid_Findframe(songtag, FR_SIZE);
			if (frame != NULL)
			{
				if (frame->type == NJB_TYPE_UINT16)
					[track setFilesize:frame->data.u_int16_val];
				else
					[track setFilesize:frame->data.u_int32_val];
			}
			frame = NJB_Songid_Findframe(songtag, FR_LENGTH);
			if (frame != NULL)
			{
				if (frame->type == NJB_TYPE_UINT16)
					[track setLength:frame->data.u_int16_val];
				else
					[track setLength:frame->data.u_int32_val];
			}
			frame = NJB_Songid_Findframe(songtag, FR_TRACK);
			if (frame != NULL)
			{
				if (frame->type == NJB_TYPE_UINT16)
					[track setTrackNumber:frame->data.u_int16_val];
				else if (frame->type == NJB_TYPE_UINT32)
					[track setTrackNumber:frame->data.u_int32_val];
				// in case it's a string
				else if (frame->type == NJB_TYPE_STRING && frame->data.strval != NULL)
				{
					NSString *trackNumber = [NSString stringWithCString:frame->data.strval];
					[track setTrackNumber:(unsigned)[trackNumber intValue]];
				}
				else
					NSLog(@"type not expected for FR_TRACK field %d", frame->type);
			}
			frame = NJB_Songid_Findframe(songtag, FR_CODEC);
			if (frame != NULL)
			{
				if (frame->data.strval == NULL)
					[track setCodec:CODEC_UNDEFINED];
				else
				{
					if (strcmp(frame->data.strval, NJB_CODEC_MP3) == 0)
						[track setCodec:CODEC_MP3];
					else if (strcmp(frame->data.strval, NJB_CODEC_WMA) == 0)
						[track setCodec:CODEC_WMA];
					else if (strcmp(frame->data.strval, NJB_CODEC_WAV) == 0)
						[track setCodec:CODEC_WAV];
					else if (strcmp(frame->data.strval, NJB_CODEC_AA) == 0)
						[track setCodec:CODEC_AA];
					else
						[track setCodec:CODEC_UNDEFINED];
				}
			}
			
			[track setItemID:songtag->trid];

			frame = NJB_Songid_Findframe(songtag, FR_YEAR);
			if (frame != NULL)
			{
				if (frame->type == NJB_TYPE_UINT16)
					[track setYear:frame->data.u_int16_val];
				else if (frame->type == NJB_TYPE_UINT32)
					[track setYear:frame->data.u_int32_val];
				// strings on NJB1
				else if (frame->type == NJB_TYPE_STRING && frame->data.strval != NULL)
				{
					NSString *year = [NSString stringWithCString:frame->data.strval];
					[track setYear:(unsigned)[year intValue]];
				}
				else
					NSLog(@"type not expected for FR_YEAR field %d", frame->type);
			}
			else
				[track setYear:0];
			
			[cachedTrackList addObject:track];
			[track release];
			
			NJB_Songid_Destroy(songtag);
		}
	}
	else
	{		
		NSDate *date = [NSDate date];
				
		LIBMTP_track_t *tracks = LIBMTP_Get_Tracklisting(device);
		if (tracks == NULL)
		{
			downloadingTracks = NO;
			return [self cachedTrackList];
		}
		
		LIBMTP_track_t *track, *tmp;
		track = tracks;
		while (track != NULL)
		{
			Track *myTrack = [[Track alloc] init];
			[myTrack setItemID:track->item_id];
			if (track->title != NULL)
				[myTrack setTitle:[NSString stringWithUTF8String:track->title]];
			if (track->artist != NULL)
				[myTrack setArtist:[NSString stringWithUTF8String:track->artist]];
			if (track->genre != NULL)
				[myTrack setGenre:[NSString stringWithUTF8String:track->genre]];
			if (track->album != NULL)
				[myTrack setAlbum:[NSString stringWithUTF8String:track->album]];
			
			switch (track->filetype)
			{
				case LIBMTP_FILETYPE_WAV:
					[myTrack setCodec:CODEC_WAV];
					break;
				case LIBMTP_FILETYPE_MP3:
					[myTrack setCodec:CODEC_MP3];
					break;
				case LIBMTP_FILETYPE_WMA:
					[myTrack setCodec:CODEC_WMA];
					break;
				default:
					[myTrack setCodec:CODEC_UNDEFINED];
					break;
			}
			
			[myTrack setFilesize:track->filesize];
			if (track->filename != NULL)
				[myTrack setFilename:[NSString stringWithUTF8String:track->filename]];
			// duration is in milliseconds
			[myTrack setLength:track->duration/1000];
			[myTrack setTrackNumber:track->tracknumber];
			
			if (track->date != NULL)
			{
				NSString *nsTimeString = [NSString stringWithUTF8String:track->date];
				// the Creative devices don't send seconds (but they do send tenths of a second!)
				NSCalendarDate *date = [[NSCalendarDate alloc] initWithString:nsTimeString calendarFormat:@"%Y%m%dT%H%M"];
			
				[myTrack setYear:[date yearOfCommonEra]];
			}
			
			[cachedTrackList addObject:myTrack];
			[myTrack release];
			
			// update progress bar
			// TODO: libmtp does not return track count
			/*
			if (i % TRACKLIST_PROGRESS_UPDATE_INTERVAL == 0)
			{
				double progressPercent = (double)i*(double)100.0 / (double)params.handles.n;
				
				[statusDisplayerGlobal updateTaskProgress:progressPercent];
			}*/
			
			tmp = track;
			track = track->next;
			LIBMTP_destroy_track_t(tmp);
		}
		
		[statusDisplayerGlobal updateTaskProgress:100.0];
		
		double time = -[date timeIntervalSinceNow];
		
		NSLog(@"getting tracklist took %f seconds", time);
	}
	
	downloadingTracks = NO;
	return [self cachedTrackList];
}

/* uploads the track and sets the track id
 */
- (NJBTransactionResult *)uploadTrack:(Track *)track
{
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:NSLocalizedString(@"Not connected", nil)] autorelease];
	
	if ([[track fullPath] length] == 0)
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:NSLocalizedString(@"Filename empty", nil)] autorelease];
	
	[self setStatus:STATUS_UPLOADING_TRACK];
	
	NSDate *date = [NSDate date];
	
	char* filename = NULL;
	char* lastPathComponent = NULL;
	
	NS_DURING
		filename = (char *) [[track fullPath] fileSystemRepresentation];
		lastPathComponent = (char *) [[[track fullPath] lastPathComponent] fileSystemRepresentation];
	NS_HANDLER
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:NSLocalizedString(@"Could not understand path %@", filename)] autorelease];
	NS_ENDHANDLER
	
	if (!mtpDevice)
	{
		// set turbo on/off
		[self enableTurbo];
		
		unsigned int trackid;
		njb_songid_t *songid = [self songidStructFromTrack:track];
		if (NJB_Send_Track(njb, filename, songid, progress, NULL, &trackid) == -1 ) {
			NSString *error = [self njbErrorString];
			NSLog(error);
			NJB_Songid_Destroy(songid);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		} else {
			[track setItemID:trackid];
			
			NJB_Songid_Destroy(songid);
		}
	}
	else
	{
		LIBMTP_track_t *trackmeta = [self libmtpTrack_tFromTrack:track];
		if (trackmeta->filename != NULL)
			free(trackmeta->filename);
		trackmeta->filename = strdup(filename);
		
		if (LIBMTP_Send_Track_From_File(device, filename, trackmeta, mtp_progress, NULL, [self musicAssoc]) != 0)
		{
			NSString *error = @"Error uploading file";
			NSLog(error);
			LIBMTP_destroy_track_t(trackmeta);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
		
		[track setItemID:trackmeta->item_id];
		
		 LIBMTP_destroy_track_t(trackmeta);
	}
	
	double time = -[date timeIntervalSinceNow];
	// rate in Mbps
	double rate = ((double)[track filesize] / time) * 8.0 / (1024.0 * 1024.0);
	
	[self updateDiskSpace];
	
	// update my cache
	[cachedTrackList addObject:track];
	// tell others the track list has changed
	[statusDisplayer postNotificationName:NOTIFICATION_TRACK_LIST_MODIFIED object:self];
	
	return [[[NJBTransactionResult alloc] initWithSuccess:YES
																					 resultString:[NSString stringWithFormat:NSLocalizedString(@"Speed %@ Mbps", nil), [self rateString:rate]]] autorelease];
		
}

- (NJBTransactionResult *)deleteTrack:(Track *)track
{
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	
	if (!mtpDevice)
	{
		if (NJB_Delete_Track(njb, [track itemID]) == -1)
		{
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	else
	{
		if (LIBMTP_Delete_Object(device, [track itemID]) != 0)
		{
			NSString *error = @"error deleting track";
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	
	[self updateDiskSpace];
	// update my cache
	[cachedTrackList removeObject:track];
	// tell others the track list has changed
	[statusDisplayer postNotificationName:NOTIFICATION_TRACK_LIST_MODIFIED object:self];
		
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)deleteFile:(NSNumber *)fileID
{
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	
	if (!mtpDevice)
	{
		if (NJB_Delete_Datafile(njb, [fileID unsignedIntValue]) == -1)
		{
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	else
	{
		if (LIBMTP_Delete_Object(device, [fileID unsignedIntValue]) != 0)
		{
			NSString *error = @"error deleting file";
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	
	[self updateDiskSpace];
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)downloadTrack:(Track *)track
{
	// check to see we're not going to overwrite anything
	NSString *fullPath = [track fullPath];
	fullPath = [self uniqueFilename:fullPath];
	[track setFullPath:fullPath];
	
	NSDate *date = [NSDate date];
	
	// todo: is fileSystemRepresentation safe here??
	
	if (!mtpDevice)
	{
		// set turbo on/off
		[self enableTurbo];
	
		if (NJB_Get_Track(njb, [track itemID], [track filesize], [[track fullPath] fileSystemRepresentation], progress, NULL) == -1)
		{
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	else
	{
		if (LIBMTP_Get_File_To_File(device, [track itemID], [[track fullPath] fileSystemRepresentation], mtp_progress, NULL) != 0)
		{
			NSString *error = @"Error downloading file";
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	
	double time = -[date timeIntervalSinceNow];
	// rate in Mbps
	double rate = ((double)[track filesize] / time) * 8.0 / (1024.0 * 1024.0);
	
	return [[[NJBTransactionResult alloc] initWithSuccess:YES
																					 resultString:[NSString stringWithFormat:NSLocalizedString(@"Speed %@ Mbps", nil), [self rateString:rate]]] autorelease];
}

- (NSString *)uniqueFilename:(NSString *)path
{
	struct stat sb;
	while (stat([path fileSystemRepresentation], &sb) != -1)
	{
		NSString *extension = [path pathExtension];
		NSString *pathWithoutExtension = [path stringByDeletingPathExtension];
		pathWithoutExtension = [pathWithoutExtension stringByAppendingString:@"_"];
		path = [pathWithoutExtension stringByAppendingPathExtension:extension];
	}
	return path;
}

- (NJBTransactionResult *)changeTrackTagTo:(Track *)newTrack from:(Track *)oldTrack
{
	if (!mtpDevice)
	{
		njb_songid_t *songid = [self songidStructFromTrack:newTrack];
		
		if (NJB_Replace_Track_Tag(njb, [newTrack itemID], songid) == -1)
		{
			NSString *error = [self njbErrorString];
			NSLog(error);
			NJB_Songid_Destroy(songid);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
		else
		{
			NJB_Songid_Destroy(songid);
		}
	}
	else
	{
		LIBMTP_track_t *libmtp_track = [self libmtpTrack_tFromTrack:newTrack];
		if (LIBMTP_Update_Track_Metadata(device, libmtp_track) != 0)
		{
			NSString *error = @"Could not update track info";
			NSLog(error);
			LIBMTP_destroy_track_t(libmtp_track);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
		LIBMTP_destroy_track_t(libmtp_track);
	}
	
	[self updateDiskSpace];
	
	// update my cache
	[cachedTrackList replaceObject:oldTrack withObject:newTrack];
	
	// tell others the track list has changed
	[statusDisplayer postNotificationName:NOTIFICATION_TRACK_LIST_MODIFIED object:self];
	
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (Directory *)dataFiles
{
	if (![self isConnected])
		return nil;
	if (cachedDataFileList)
	{
		return [self cachedDataFileList];
	}
	
	downloadingFiles = YES;
	
	cachedDataFileList = [[Directory alloc] init];
	// a better name for here...
	Directory *baseDir = cachedDataFileList;
	
	if (!mtpDevice)
	{
		NJB_Reset_Get_Datafile_Tag(njb);
		njb_datafile_t *filetag;
		while (filetag = NJB_Get_Datafile_Tag(njb))
		{
			NSString *filename = @"";
			if (filetag->filename != NULL)
				filename = [NSString stringWithCString:filetag->filename];
			NSArray *pathArray = nil;
			if (filetag->folder != NULL)
			{
				NSString *folder = [NSString stringWithCString:filetag->folder];
				// this has \\ instead of /
				pathArray = [folder componentsSeparatedByString:@"\\"];
				// the / on the beginning and end gives us extra empty objects
				NSMutableArray *newPathArray = [NSMutableArray arrayWithArray:pathArray];
				if ([[newPathArray objectAtIndex:[newPathArray count] - 1] length] == 0)
					[newPathArray removeLastObject];
				if ([[newPathArray objectAtIndex:0] length] == 0)
					[newPathArray removeObjectAtIndex:0];
				pathArray = newPathArray;
			}

			if ([filename isEqualToString:@"."])
			{
				// this is an empty folder
				NSMutableArray *parentPathArray = [NSMutableArray arrayWithArray:pathArray];
				[parentPathArray removeLastObject];
				
				NSString *name = [pathArray objectAtIndex:[pathArray count] - 1];

				Directory *parentDir = [baseDir itemWithPath:parentPathArray];
				if (![parentDir containsItemWithName:name])
				{
					Directory *newDir = [[Directory alloc] initWithName:name];
					[newDir setItemID:filetag->dfid];
				
					[baseDir addItem:newDir toDir:parentPathArray];
				
					[newDir release];
				}
			}
			else
			{
				// this is a file
				DataFile *dataFile = [[DataFile alloc] init];
				[dataFile setFilename:filename];
				[dataFile setSize:filetag->filesize];
				// [dataFile setTimestampSince1970:filetag->timestamp];
				[dataFile setItemID:filetag->dfid];
				
				[baseDir addItem:dataFile toDir:pathArray];
				
				[dataFile release];
			}
			
			NJB_Datafile_Destroy(filetag);
		}
	}
	else
	{
		LIBMTP_file_t *files;
		
		files = LIBMTP_Get_Filelisting(device);
		if (files == NULL)
		{
			downloadingFiles = NO;
			return nil;
		}
		
		LIBMTP_file_t *file, *tmp;
		file = files;
		while (file != NULL)
		{
			// TODO: use Get_Folder_List for folders first; then get files. we will only get files here
			
			if (![self isAudioType:file->filetype])
			{
				DataFile *dataFile = [[DataFile alloc] init];
				if (file->filename != NULL)
					[dataFile setFilename:[NSString stringWithUTF8String:file->filename]];
				[dataFile setSize:file->filesize];
				[dataFile setItemID:file->item_id];
				
				[baseDir addItem:dataFile toDir:nil];
				
				[dataFile release];
			}
			
			tmp = file;
			file = file->next;
			LIBMTP_destroy_file_t(tmp);
		}
	}
	
	downloadingFiles = NO;
	return [self cachedDataFileList];
}

- (NJBTransactionResult *)downloadFile:(DataFile *)dataFile
{
	// check to see we're not going to overwrite anything
	NSString *fullPath = [dataFile fullPath];
	fullPath = [self uniqueFilename:fullPath];
	[dataFile setFullPath:fullPath];
		
	NSDate *date = [NSDate date];
		
	if (!mtpDevice)
	{
		// set turbo on/off
		[self enableTurbo];

		if (NJB_Get_File(njb, [dataFile itemID], [dataFile size], [[dataFile fullPath] fileSystemRepresentation], progress, NULL) == -1)
		{
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	else
	{
		if (LIBMTP_Get_File_To_File(device, [dataFile itemID], [[dataFile fullPath] fileSystemRepresentation], mtp_progress, NULL) != 0)
		{
			NSString *error = @"Error downloading file";
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	
	double time = -[date timeIntervalSinceNow];
	// rate in Mbps
	double rate = ((double)[dataFile size] / time) * 8.0 / (1024.0 * 1024.0);
		
	return [[[NJBTransactionResult alloc] initWithSuccess:YES
																					 resultString:[NSString stringWithFormat:NSLocalizedString(@"Speed %@ Mbps", nil), [self rateString:rate]]] autorelease];
}

- (NJBTransactionResult *)uploadFile:(DataFile *)dataFile toFolder:(NSString *)path
{
	if (![self isConnected])
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	
	NSDate *date = [NSDate date];
	[self setStatus:STATUS_UPLOADING_FILE];

	if (!mtpDevice)
	{
		// set turbo on/off
		[self enableTurbo];
	
		unsigned int fileid;
		if (NJB_Send_File(njb, [[dataFile fullPath] fileSystemRepresentation], [[[dataFile fullPath] lastPathComponent] UTF8String], [path UTF8String], progress, NULL, &fileid) == -1 ) {
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		} else {
			[dataFile setItemID:fileid];
		}
	}
	else
	{
		uint16_t objectFormat = LIBMTP_FILETYPE_UNKNOWN;
		
		uint32_t parenthandle = 0;
		
		// a test to see how the player works with video & images. We really should
		// implement the full directory structure for MTP and just upload to the folder
		// we've been told to here.
		NSString *extension = [[[dataFile fullPath] pathExtension] lowercaseString];
		if ([extension isEqualToString:@"wmv"])
		{
			objectFormat = LIBMTP_FILETYPE_WMV;
			parenthandle = [self videoAssoc];
		}
		else if ([extension isEqualToString:@"avi"])
		{
			objectFormat = LIBMTP_FILETYPE_AVI;
			parenthandle = [self videoAssoc];
		}
		else if ([extension isEqualToString:@"mpg"] || [extension isEqualToString:@"mpeg"])
		{
			objectFormat = LIBMTP_FILETYPE_MPEG;
			parenthandle = [self videoAssoc];
		}
		else if ([extension isEqualToString:@"asf"])
		{
			objectFormat = LIBMTP_FILETYPE_ASF;
			parenthandle = [self videoAssoc];
		}
		else if ([extension isEqualToString:@"qt"])
		{
			objectFormat = LIBMTP_FILETYPE_QT;
			parenthandle = [self videoAssoc];
		}
		else if ([extension isEqualToString:@"jpg"] || [extension isEqualToString:@"jpeg"])
		{
			objectFormat = LIBMTP_FILETYPE_JPEG;
			parenthandle = [self picturesAssoc];
		}
		else if ([extension isEqualToString:@"tif"] || [extension isEqualToString:@"tiff"])
		{
			objectFormat = LIBMTP_FILETYPE_TIFF;
			parenthandle = [self picturesAssoc];
		}
		else if ([extension isEqualToString:@"bmp"])
		{
			objectFormat = LIBMTP_FILETYPE_BMP;
			parenthandle = [self picturesAssoc];
		}
		else if ([extension isEqualToString:@"gif"])
		{
			objectFormat = LIBMTP_FILETYPE_GIF;
			parenthandle = [self picturesAssoc];
		}
		else if ([extension isEqualToString:@"pict"])
		{
			objectFormat = LIBMTP_FILETYPE_PICT;
			parenthandle = [self picturesAssoc];
		}
		else if ([extension isEqualToString:@"png"])
		{
			objectFormat = LIBMTP_FILETYPE_PNG;
			parenthandle = [self picturesAssoc];
		}
		
		NSLog(@"uploading file with object format 0x%.02x", objectFormat);
		
		LIBMTP_file_t *genfile = LIBMTP_new_file_t();
		genfile->filesize = [dataFile size];
		genfile->filetype = objectFormat;
		// todo: safe?
		genfile->filename = strdup([[[dataFile fullPath] lastPathComponent] fileSystemRepresentation]);
		
		if (LIBMTP_Send_File_From_File(device, [[dataFile fullPath] fileSystemRepresentation], genfile, mtp_progress, NULL, parenthandle) != 0)
		{
			/* we might get an (MTP error) object too large here */
			NSString *error = @"Error uploading file";
			NSLog(error);
			LIBMTP_destroy_file_t(genfile);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}

		[dataFile setItemID:genfile->item_id];
		LIBMTP_destroy_file_t(genfile);
	}
	
	[self updateDiskSpace];
		
	double time = -[date timeIntervalSinceNow];
	// rate in Mbps
	double rate = ((double)[dataFile size] / time) * 8.0 / (1024.0 * 1024.0);
	return [[[NJBTransactionResult alloc] initWithSuccess:YES
																					 resultString:[NSString stringWithFormat:NSLocalizedString(@"Speed %@ Mbps", nil), [self rateString:rate]]] autorelease];	
}

- (unsigned long long)totalDiskSpace
{
	return totalDiskSpace;
}

- (unsigned long long)freeDiskSpace
{
	return freeDiskSpace;
}

- (void)updateDiskSpace
{
	if (!mtpDevice)
	{
		NJB_Get_Disk_Usage(njb, &totalDiskSpace, &freeDiskSpace);
		[statusDisplayer updateDiskSpace:totalDiskSpace withFreeSpace:freeDiskSpace];
	}
	else
	{
		char *storage_description = NULL;
		char *volume_label = NULL;
		if (LIBMTP_Get_Storageinfo(device, &totalDiskSpace, &freeDiskSpace, &storage_description, &volume_label) == 0)
		{
			free(storage_description);
			free(volume_label);
			
			[statusDisplayer updateDiskSpace:totalDiskSpace withFreeSpace:freeDiskSpace];
		}
	}
}

- (NSCalendarDate *)jukeboxTime
{
	if (!mtpDevice)
	{
		njb_time_t *time = NJB_Get_Time(njb);
		if (time == NULL)
			return nil;
	
		// assume njb time is our timezone
		NSCalendarDate *date = [NSCalendarDate dateWithYear:time->year month:time->month day:time->day hour:time->hours minute:time->minutes second:time->seconds
																							 timeZone:[NSTimeZone localTimeZone]];
		return date;
	}
	else
	{
		/*char *secureTime = NULL;
		if (LIBMTP_Get_Secure_Time(device, &secureTime) != 0)
			return nil;
		
		if (secureTime == NULL)
			return nil;
		
		NSLog(@"Secure time: %s", secureTime);
		
		free(secureTime);*/
		// TODO
		/*
		if (!ptp_property_issupported(&params, PTP_DPC_DateTime))
			return nil;
		
		char *timeString = NULL;
		if (ptp_getdevicepropvalue(&params, PTP_DPC_DateTime, (void **)&timeString, PTP_DTC_STR) != PTP_RC_OK)
			return nil;
		
		if (timeString != NULL)
		{
			NSString *nsTimeString = [NSString stringWithCString:timeString];
			free(timeString);
			NSCalendarDate *date = [[NSCalendarDate alloc] initWithString:nsTimeString calendarFormat:@"%Y%m%dT%H%M%S%z"];
		
			return [date autorelease];
		}
		else*/
			return nil;
	}
}

- (BOOL)isProtocol3Device
{
	if (!mtpDevice)
		return (PDE_PROTOCOL_DEVICE(njb));
	else
		return NO;
}

- (NJBTransactionResult *)setOwnerString:(NSString *)owner
{
	if (!mtpDevice)
	{
		if (NJB_Set_Owner_String (njb, [owner UTF8String]) == -1)
		{
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	else
	{
		if (LIBMTP_Set_Friendlyname(device, [owner UTF8String]) != 0)
		{
			NSString *error = [NSString stringWithFormat:@"Could not set device friendly name"];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)setBitmap:(NSString *)bitmapPath
{
	if (mtpDevice)
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	
	if (NJB_Set_Bitmap(njb, [bitmapPath UTF8String]) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)setTime:(NSNumber *)timeIntervalSinceNow
{
	if (mtpDevice)
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	
	njb_time_t time;
	
	NSTimeInterval jukeboxTimeInterval = [timeIntervalSinceNow doubleValue];
	NSCalendarDate *date = [NSCalendarDate dateWithTimeIntervalSinceNow:jukeboxTimeInterval];
	
	time.year = [date yearOfCommonEra];
	time.month = [date monthOfYear];
	time.day = [date dayOfMonth];
	time.weekday = [date dayOfWeek];
	time.hours = [date hourOfDay];
	time.minutes = [date minuteOfHour];
	time.seconds = [date secondOfMinute];
	
	if (NJB_Set_Time(njb, &time) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NSMutableArray *)playlists
{
	if (![self isConnected])
		return nil;
	
	NSMutableArray *playlists = [[NSMutableArray alloc] init];
	
	if (!mtpDevice)
	{
		NJB_Reset_Get_Playlist(njb);
		njb_playlist_t *playlist;
		while ((playlist = NJB_Get_Playlist(njb)))
		{
			Playlist *newPlaylist = [[Playlist alloc] init];
					
			const char *str = playlist->name;
			if (str != NULL)
				[newPlaylist setName:[NSString stringWithUTF8String:str]];
			[newPlaylist setPlaylistID:playlist->plid];
			
			njb_playlist_track_t *track;
			NJB_Playlist_Reset_Gettrack(playlist);
			while ((track = NJB_Playlist_Gettrack(playlist)))
			{
				Track *newTrack = [[Track alloc] init];
				[newTrack setItemID:track->trackid];
				[newPlaylist addTrack:newTrack];
				[newTrack release];
			}
			
			NJB_Playlist_Destroy(playlist);
			
			[newPlaylist setState:NJB_PL_UNCHANGED];
			
			[playlists addObject:newPlaylist];
			[newPlaylist release];
		}
	}
	else
	{
		LIBMTP_playlist_t *libmtpPlaylists = LIBMTP_Get_Playlist_List(device);
		if (libmtpPlaylists == NULL)
		{
			// no playlists
			return [playlists autorelease];
		}
		LIBMTP_playlist_t *pl, *tmp;
    pl = libmtpPlaylists;
    while (pl != NULL) {
			Playlist *newPlaylist = [[Playlist alloc] init];
			
			[newPlaylist setPlaylistID:pl->playlist_id];
			[newPlaylist setName:[NSString stringWithUTF8String:pl->name]];
			int i = 0;
			for (i = 0; i < pl->no_tracks; i++)
			{
				Track *newTrack = [[Track alloc] init];
				[newTrack setItemID:pl->tracks[i]];
				[newPlaylist addTrack:newTrack];
				[newTrack release];
			}
			
			[newPlaylist setState:NJB_PL_UNCHANGED];
			
			[playlists addObject:newPlaylist];
			[newPlaylist release];

      tmp = pl;
      pl = pl->next;
      LIBMTP_destroy_playlist_t(tmp);
    }
	}
	return [playlists autorelease];
}

/* updates the playlist on the NJB
 * may add a new playlist even if playlist state is not new
 * so the playlist ID may change.
 */
- (NJBTransactionResult *)updatePlaylist:(Playlist *)playlist
{
	[playlist lock];
	// don't do anything if it's unchanged
	if ([playlist state] == NJB_PL_UNCHANGED)
	{
		[playlist unlock];
		return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
	}
	
	// the playlist is locked here
	if (!mtpDevice)
	{
		// create new njb_playlist_t object
		njb_playlist_t *pl = NJB_Playlist_New();
		NJB_Playlist_Set_Name(pl, [[playlist name] UTF8String]);
		pl->plid = [playlist playlistID];
		NSEnumerator *enumerator = [[playlist tracks] objectEnumerator];
		Track *currentTrack;
		while (currentTrack = [enumerator nextObject])
		{
			njb_playlist_track_t *newTrack = NJB_Playlist_Track_New([currentTrack itemID]);
			NJB_Playlist_Addtrack(pl, newTrack, NJB_PL_END);
		}
		pl->_state = [playlist state];
		// set this now as as far as the caller is concerned the NJB is updated
		[playlist setState:NJB_PL_UNCHANGED];
		[playlist unlock];
		if (NJB_Update_Playlist(njb, pl) == -1)
		{
			NJB_Playlist_Destroy(pl);
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
		[playlist lock];
		// update the ID as this is changed when tracks added/removed as a new playlist is created
		[playlist setPlaylistID:pl->plid];
		[playlist unlock];
		NJB_Playlist_Destroy(pl);
	}
	else
	{
		NSArray *tracks = [playlist tracks];
		uint32_t size = [tracks count];
		uint32_t *trackArray = (uint32_t*)malloc(sizeof(uint32_t)*size);
		uint32_t i = 0;
		for (i = 0; i < size; i++)
			trackArray[i] = [[tracks objectAtIndex:i] itemID];
		
		LIBMTP_playlist_t *libmtpPlaylist = LIBMTP_new_playlist_t();
		libmtpPlaylist->name = strdup([[playlist name] UTF8String]);
		libmtpPlaylist->playlist_id = [playlist playlistID];
		libmtpPlaylist->no_tracks = size;
		libmtpPlaylist->tracks = trackArray;
		
		int playlistState = [playlist state];
		[playlist setState:NJB_PL_UNCHANGED];
		[playlist unlock];
		
		if (playlistState == NJB_PL_NEW)
		{
			if (LIBMTP_Create_New_Playlist(device, libmtpPlaylist, device->default_playlist_folder) != 0)
			{
				LIBMTP_destroy_playlist_t(libmtpPlaylist);
				return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:
					[NSString stringWithFormat:@"Could not create new playlist"]] autorelease];
			}
			[playlist lock];
			[playlist setPlaylistID:libmtpPlaylist->playlist_id];
			[playlist unlock];
		}
		else
		{
			if (LIBMTP_Update_Playlist(device, libmtpPlaylist) != 0)
			{
				LIBMTP_destroy_playlist_t(libmtpPlaylist);
				return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:
					[NSString stringWithFormat:@"Could not update playlist"]] autorelease];
			}
			[playlist lock];
			// it doesn't change the playlist now but put it in to be
			// on the safe side
			[playlist setPlaylistID:libmtpPlaylist->playlist_id];
			[playlist unlock];
		}
		LIBMTP_destroy_playlist_t(libmtpPlaylist);
	}
	
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (NJBTransactionResult *)deletePlaylist:(Playlist *)playlist
{	
	[playlist lock];
	unsigned plid = [playlist playlistID];
	[playlist unlock];
	
	if (!mtpDevice)
	{
		if (NJB_Delete_Playlist(njb, plid) == -1)
		{
			NSString *error = [self njbErrorString];
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	else
	{
		if (LIBMTP_Delete_Object(device, [playlist playlistID]) != 0)
		{
			NSString *error = @"error deleting playlist";
			NSLog(error);
			return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
		}
	}
	
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

/* called by the file/track transfer functions of libnjb
 * to tell us the progress
 */
int progress(u_int64_t sent, u_int64_t total, const char* buf, unsigned len, void *data)
{
	double percent = (double)sent*100.0/(double)total;
	[statusDisplayerGlobal updateTaskProgress:percent];
	
	return 0;
}

int mtp_progress (uint64_t const sent, uint64_t const total, void const * const data)
{
	double percent = (double)sent*100.0/(double)total;
	[statusDisplayerGlobal updateTaskProgress:percent];
	
	return 0;
}

- (njb_songid_t *)songidStructFromTrack:(Track *)track
{
	njb_songid_t *songid;
	njb_songid_frame_t *frame;
	
	songid = NJB_Songid_New();
	if ([track njbCodec])
	{
		frame = NJB_Songid_Frame_New_Codec([[track njbCodec] UTF8String]);
		NJB_Songid_Addframe(songid, frame);
	}
	// only add file size if non zero
	if ([track filesize] != 0)
	{
		frame = NJB_Songid_Frame_New_Filesize([track filesize]);
		NJB_Songid_Addframe(songid, frame);
	}
	if ([track title])
	{
		frame = NJB_Songid_Frame_New_Title([[track title] UTF8String]);
		NJB_Songid_Addframe(songid, frame);
	}
	if ([track album])
	{
		frame = NJB_Songid_Frame_New_Album([[track album] UTF8String]);
		NJB_Songid_Addframe(songid, frame);
	}
	if ([track artist])
	{
		frame = NJB_Songid_Frame_New_Artist([[track artist] UTF8String]);
		NJB_Songid_Addframe(songid, frame);
	}
	if ([track genre])
	{
		frame = NJB_Songid_Frame_New_Genre([[track genre] UTF8String]);
		NJB_Songid_Addframe(songid, frame);
	}
	frame = NJB_Songid_Frame_New_Year([track year]);
	NJB_Songid_Addframe(songid, frame);
	frame = NJB_Songid_Frame_New_Tracknum([track trackNumber]);
	NJB_Songid_Addframe(songid, frame);
	frame = NJB_Songid_Frame_New_Length([track length]);
	NJB_Songid_Addframe(songid, frame);
	if ([track fullPath] && [[track fullPath] length] > 0)
	{
		frame = NJB_Songid_Frame_New_Filename([[track fullPath] fileSystemRepresentation]);
		NJB_Songid_Addframe(songid, frame);
	}
	
	return songid;
}

- (NSMutableArray *)cachedTrackList
{
	if (cachedTrackList && !downloadingTracks)
	{
		NSMutableArray *tracks = [[NSMutableArray alloc] initWithArray:cachedTrackList];
		return [tracks autorelease];
	}
	else
		return nil;
}

- (Directory *)cachedDataFileList
{
	if (cachedDataFileList && !downloadingFiles)
	{
		return [cachedDataFileList copy];
	}
	else
		return nil;
}

/* returns a string rounded to 1dp for
 * the rate given
 */
- (NSString *)rateString:(double) rate
{
	int rate10 = rate * 10;
	if (rate - (double)rate10/10.0 >= 0.05)
		rate10++;
	// the two digits
	int rateA = rate10/10;
	int rateB = rate10-rateA*10;
	return [NSString stringWithFormat:@"%d.%d", rateA, rateB];
}

- (NJBTransactionResult *)createFolder:(Directory *)dir inDir:(NSString *)path
{
	if (mtpDevice)
		return [[[NJBTransactionResult alloc] initWithSuccess:NO] autorelease];
	
	unsigned int folderID;
	if (NJB_Create_Folder(njb, [[NSString stringWithFormat:@"%@%@", path, [dir name]] UTF8String], &folderID) == -1)
	{
		NSString *error = [self njbErrorString];
		NSLog(error);
		return [[[NJBTransactionResult alloc] initWithSuccess:NO resultString:error] autorelease];
	}
	[dir setItemID:folderID];
	return [[[NJBTransactionResult alloc] initWithSuccess:YES] autorelease];
}

- (void)storeDeviceString
{
	if (!mtpDevice)
	{
		const char *name = NJB_Get_Device_Name(njb, 0);
		if (name != NULL)
			deviceString = [[NSString alloc] initWithCString:name];
		else
			deviceString = nil;
	}
	else
		deviceString = @"";
}

- (void)storeFirmwareVersionString
{
	if (!mtpDevice)
	{
		u_int8_t major, minor, release;
			
		int ret = NJB_Get_Firmware_Revision(njb, &major, &minor, &release);
		if (ret == 0)
			firmwareVersionString = [[NSString alloc] initWithString:[NSString stringWithFormat:@"%u.%u.%u", major, minor, release]];
		else
			deviceVersionString = nil;
	}
	else
		firmwareVersionString = @"";
}

- (void)storeDeviceIDString
{	
	if (!mtpDevice)
	{
		NSMutableString *idString = [[NSMutableString alloc] initWithCapacity:32];
		int j;
		
		u_int8_t sdmiid[16];
		int result;
		result = NJB_Get_SDMI_ID(njb, &(sdmiid[0]));
		if (result != 0)
			idString = @"";
		
		for (j = 0; j < 16; j++)
		{
			if (sdmiid[j] > 15)
				[idString appendString:[NSString stringWithFormat:@"%X", sdmiid[j]]];
			else
				[idString appendString:[NSString stringWithFormat:@"0%X", sdmiid[j]]];
		}
		
		deviceIDString = [[NSString alloc] initWithString:[NSString stringWithString:idString]];
		[idString release];
	}
	else
		deviceIDString = @"";
}

- (void)storeDeviceVersionString
{
	if (!mtpDevice)
	{
		u_int8_t major, minor, release;
			
		int ret = NJB_Get_Hardware_Revision(njb, &major, &minor, &release);
		if (ret == 0)
			deviceVersionString = [[NSString alloc] initWithString:[NSString stringWithFormat:@"%u.%u.%u", major, minor, release]];
		else
			deviceVersionString = nil;
	}
	else
		deviceVersionString = @"";
}

- (int)batteryLevel
{
	if (connected == NO)
		return 0;
	if (!mtpDevice)
	{
		return NJB_Get_Battery_Level(njb);
	}
	else
	{				
		uint8_t value = 0;
		if (LIBMTP_Get_Batterylevel(device, &batteryLevelMax, &value) != 0)
		{
			NSLog(@"Could not get battery level property value");
			return 0;
		}
				
		return value;
	}
}

- (NSString *)batteryStatus
{
	if (!mtpDevice)
	{
		BOOL charging = (NJB_Get_Battery_Charging(njb) == 1);
		BOOL powerConnected = (NJB_Get_Auxpower(njb) == 1);
		
		if (charging)
			return NSLocalizedString(@"Charging", nil);
		if (powerConnected)
			return NSLocalizedString(@"Battery charged or not present", nil);
		return NSLocalizedString(@"Running off battery", nil);
	}
	else
	{
		// they always charge...
		// TODO: they don't - they can be fully charged. Can we see that?
		return @"Charging";
	}
}

- (void)enableTurbo
{
	//NSLog(@"enableTurbo : %d", turbo);

	if (turbo)
		NJB_Set_Turbo_Mode(njb, NJB_TURBO_ON);
	else
		NJB_Set_Turbo_Mode(njb, NJB_TURBO_OFF);
}

// TODO: libmtp has this in LIBMTP_Get_Tracklisting - they should be the same function
- (BOOL)isAudioType:(LIBMTP_filetype_t) filetype
{
	if (filetype == LIBMTP_FILETYPE_WAV ||
			filetype == LIBMTP_FILETYPE_MP3 ||
			filetype == LIBMTP_FILETYPE_WMA ||
			filetype == LIBMTP_FILETYPE_OGG ||
			filetype == LIBMTP_FILETYPE_AUDIBLE ||
			filetype == LIBMTP_FILETYPE_MP4 ||
			filetype == LIBMTP_FILETYPE_UNDEF_AUDIO
			)
		return YES;
	else
		return NO;
}

// return value must be freed (LIBMTP_destroy_track_t) by caller
- (LIBMTP_track_t *)libmtpTrack_tFromTrack:(Track *)track
{
	LIBMTP_track_t *libmtp_track = LIBMTP_new_track_t();
	
	switch ([track codec])
	{
		case CODEC_MP3:
			libmtp_track->filetype = LIBMTP_FILETYPE_MP3;
			break;
		case CODEC_WAV:
			libmtp_track->filetype = LIBMTP_FILETYPE_WAV;
			break;
		case CODEC_WMA:
			libmtp_track->filetype = LIBMTP_FILETYPE_WMA;
			break;
		default:
			libmtp_track->filetype = LIBMTP_FILETYPE_UNDEF_AUDIO;
			break;
	}
		
	libmtp_track->title = strdup([[track title] UTF8String]);
	libmtp_track->album = strdup([[track album] UTF8String]);
	libmtp_track->artist = strdup([[track artist] UTF8String]);
	libmtp_track->genre = strdup([[track genre] UTF8String]);
	NSString *nsTimeString = [NSString stringWithFormat:@"%.04d0101T000000.0", [track year]];
	libmtp_track->date = strdup([nsTimeString UTF8String]);
	libmtp_track->tracknumber = [track trackNumber];
	libmtp_track->duration = [track length] * 1000;
	libmtp_track->filesize = [track filesize];
	libmtp_track->filename = strdup([[track filename] UTF8String]);
	libmtp_track->item_id = [track itemID];
		
	return libmtp_track;
}

@end
