/**
 * \file libmtp.h
 *
 * Interface to the Media Transfer Protocol library.
 *
 * <code>
 * #include <libmtp.h>
 * </code>
 */
#ifndef LIBMTP_H_INCLUSION_GUARD
#define LIBMTP_H_INCLUSION_GUARD

#define LIBMTP_VERSION 0.0.22

/* This handles MSVC pecularities */
#ifdef _MSC_VER
#include <windows.h>
#define __WIN32__
#define snprintf _snprintf
#define ssize_t SSIZE_T
#endif

#include <stdio.h>
#include <usb.h>
#include <stdint.h>

#ifdef __WIN32__
/*
 * Windows specific code, types that do not exist in Windows
 * sys/types.h
 */
typedef char int8_t;
typedef unsigned char uint8_t;
typedef __int16 int16_t;
typedef unsigned __int16 uint16_t;
typedef __int32 int32_t;
typedef unsigned __int32 uint32_t;
typedef unsigned __int64 uint64_t;
#endif

/**
 * @defgroup types libmtp global type definitions
 * @{
 * The filetypes defined here are the external types used
 * by the libmtp library interface. The types used internally
 * as PTP-defined enumerator types is something different.
 */
typedef enum {
  LIBMTP_FILETYPE_WAV,
  LIBMTP_FILETYPE_MP3,
  LIBMTP_FILETYPE_WMA,
  LIBMTP_FILETYPE_OGG,
  LIBMTP_FILETYPE_AUDIBLE,
  LIBMTP_FILETYPE_MP4,
  LIBMTP_FILETYPE_UNDEF_AUDIO,
  LIBMTP_FILETYPE_WMV,
  LIBMTP_FILETYPE_AVI,
  LIBMTP_FILETYPE_MPEG,
  LIBMTP_FILETYPE_ASF,
  LIBMTP_FILETYPE_QT,
  LIBMTP_FILETYPE_UNDEF_VIDEO,
  LIBMTP_FILETYPE_JPEG,
  LIBMTP_FILETYPE_JFIF,
  LIBMTP_FILETYPE_TIFF,
  LIBMTP_FILETYPE_BMP,
  LIBMTP_FILETYPE_GIF,
  LIBMTP_FILETYPE_PICT,
  LIBMTP_FILETYPE_PNG,
  LIBMTP_FILETYPE_VCALENDAR1,
  LIBMTP_FILETYPE_VCALENDAR2,
  LIBMTP_FILETYPE_VCARD2,
  LIBMTP_FILETYPE_VCARD3,
  LIBMTP_FILETYPE_WINDOWSIMAGEFORMAT,
  LIBMTP_FILETYPE_WINEXEC,
  LIBMTP_FILETYPE_TEXT,
  LIBMTP_FILETYPE_HTML,
  LIBMTP_FILETYPE_UNKNOWN
} LIBMTP_filetype_t;
typedef struct LIBMTP_device_entry_struct LIBMTP_device_entry_t; /**< @see LIBMTP_device_entry_struct */
typedef struct LIBMTP_mtpdevice_struct LIBMTP_mtpdevice_t; /**< @see LIBMTP_mtpdevice_struct */
typedef struct LIBMTP_file_struct LIBMTP_file_t; /**< @see LIBMTP_file_struct */
typedef struct LIBMTP_track_struct LIBMTP_track_t; /**< @see LIBMTP_track_struct */
typedef struct LIBMTP_playlist_struct LIBMTP_playlist_t; /**< @see LIBMTP_playlist_struct */
typedef struct LIBMTP_album_struct LIBMTP_album_t; /**< @see LIBMTP_album_struct */
typedef struct LIBMTP_folder_struct LIBMTP_folder_t; /**< @see LIBMTP_folder_t */
typedef struct LIBMTP_object_struct LIBMTP_object_t; /**< @see LIBMTP_object_t */

/**
 * The callback type definition. Notice that a progress percentage ratio
 * is easy to calculate by dividing <code>sent</code> by
 * <code>total</code>.
 * @param sent the number of bytes sent so far
 * @param total the total number of bytes to send
 * @param data a user-defined dereferencable pointer
 * @return if anything else than 0 is returned, the current transfer will be
 *         interrupted / cancelled.
 */
typedef int (* LIBMTP_progressfunc_t) (uint64_t const sent, uint64_t const total,
                		void const * const data);

/**
 * @}
 * @defgroup structar libmtp data structures
 * @{
 */

/**
 * A data structure to hold MTP device entries
 */
struct LIBMTP_device_entry_struct {
  char *name; /**< The descriptive name of this device */
  uint16_t vendor_id; /**< Vendor ID for this device */
  uint16_t product_id; /**< Product ID for this device */
  uint32_t device_flags; /**< Bugs, device specifics etc */
};

/**
 * Main MTP device object struct
 */
struct LIBMTP_mtpdevice_struct {
  /** Interface number of this device */
  uint8_t interface_number;
  /**
   * Parameters for this device, must be cast into
   * \c (PTPParams*) before internal use.
   */
  void *params;
  /**
   * USB device for this device, must be cast into
   * \c (PTP_USB*) before internal use.
   */
  void *usbinfo;
  /** The storage ID for this device */
  uint32_t storage_id;
  /** The maximum battery level for this device */
  uint8_t maximum_battery_level;
  /** Default music folder */
  uint32_t default_music_folder;
  /** Default playlist folder */
  uint32_t default_playlist_folder;
  /** Default picture folder */
  uint32_t default_picture_folder;
  /** Default video folder */
  uint32_t default_video_folder;
  /** Default organizer folder */
  uint32_t default_organizer_folder;
  /** Default ZENcast folder (only Creative devices...) */
  uint32_t default_zencast_folder;
  /** Default Album folder */
  uint32_t default_album_folder;
  /** Per device iconv() converters, only used internally */
  void *cd;
};

/**
 * MTP file struct
 */
struct LIBMTP_file_struct {
  uint32_t item_id; /**< Unique item ID */
  uint32_t parent_id; /**< ID of parent folder */
  char *filename; /**< Filename of this file */
  uint64_t filesize; /**< Size of file in bytes */
  LIBMTP_filetype_t filetype; /**< Filetype used for the current file */
  LIBMTP_file_t *next; /**< Next file in list or NULL if last file */
};

/**
 * MTP track struct
 */
struct LIBMTP_track_struct {
  uint32_t item_id; /**< Unique item ID */
  uint32_t parent_id; /**< ID of parent folder */
  char *title; /**< Track title */
  char *artist; /**< Name of recording artist */
  char *genre; /**< Genre name for track */
  char *album; /**< Album name for track */
  char *date; /**< Date of original recording as a string */
  char *filename; /**< Original filename of this track */
  uint16_t tracknumber; /**< Track number (in sequence on recording) */
  uint32_t duration; /**< Duration in milliseconds */
  uint32_t samplerate; /**< Sample rate of original file, min 0x1f80 max 0xbb80 */
  uint16_t nochannels; /**< Number of channels in this recording 0 = unknown, 1 or 2 */
  uint32_t wavecodec; /**< FourCC wave codec name */
  uint32_t bitrate; /**< (Average) bitrate for this file min=1 max=0x16e360 */
  uint16_t bitratetype; /**< 0 = unused, 1 = constant, 2 = VBR, 3 = free */
  uint16_t rating; /**< User rating 0-100 (0x00-0x64) */
  uint32_t usecount; /**< Number of times used/played */
  uint64_t filesize; /**< Size of track file in bytes */
  LIBMTP_filetype_t filetype; /**< Filetype used for the current track */
  LIBMTP_track_t *next; /**< Next track in list or NULL if last track */
};

/**
 * MTP Playlist structure
 */
struct LIBMTP_playlist_struct {
  uint32_t playlist_id; /**< Unique playlist ID */
  char *name; /**< Name of playlist */
  uint32_t *tracks; /**< The tracks in this playlist */
  uint32_t no_tracks; /**< The number of tracks in this playlist */
  LIBMTP_playlist_t *next; /**< Next playlist or NULL if last playlist */
};

/**
 * MTP Album structure
 */
struct LIBMTP_album_struct {
  uint32_t album_id; /**< Unique playlist ID */
  char *name; /**< Name of album */
  uint32_t *tracks; /**< The tracks in this album */
  uint32_t no_tracks; /**< The number of tracks in this album */
  LIBMTP_album_t *next; /**< Next album or NULL if last album */
};

/**
 * MTP Folder structure
 */
struct LIBMTP_folder_struct {
  uint32_t folder_id; /**< Unique folder ID */
  uint32_t parent_id; /**< ID of parent folder */
  char *name; /**< Name of folder */
  LIBMTP_folder_t *sibling; /**< Next folder at same level or NULL if no more */
  LIBMTP_folder_t *child; /**< Child folder or NULL if no children */
};

/**
 * LIBMTP Object Structure
 */
struct LIBMTP_object_struct {
  uint32_t id;
  uint32_t parent;
  uint32_t type;
  uint32_t size;
  char *name;
  void *data;
  LIBMTP_object_t *sibling;
  LIBMTP_object_t *child;
};

/** @} */

/* Make functions available for C++ */
#ifdef __cplusplus
extern "C" {
#endif

/**
 * @defgroup internals The libmtp internals API.
 * @{
 */
void LIBMTP_Init(void);
int LIBMTP_Get_Supported_Devices_List(LIBMTP_device_entry_t ** const, int * const);
/**
 * @}
 * @defgroup basic The basic device management API.
 * @{
 */
int LIBMTP_Detect_Descriptor(uint16_t*,uint16_t*);
LIBMTP_mtpdevice_t *LIBMTP_Get_First_Device(void);
void LIBMTP_Release_Device(LIBMTP_mtpdevice_t*);
void LIBMTP_Dump_Device_Info(LIBMTP_mtpdevice_t*);
char *LIBMTP_Get_Modelname(LIBMTP_mtpdevice_t*);
char *LIBMTP_Get_Serialnumber(LIBMTP_mtpdevice_t*);
char *LIBMTP_Get_Deviceversion(LIBMTP_mtpdevice_t*);
char *LIBMTP_Get_Friendlyname(LIBMTP_mtpdevice_t*);
int LIBMTP_Set_Friendlyname(LIBMTP_mtpdevice_t*, char const * const);
char *LIBMTP_Get_Syncpartner(LIBMTP_mtpdevice_t*);
int LIBMTP_Set_Syncpartner(LIBMTP_mtpdevice_t*, char const * const);
int LIBMTP_Get_Storageinfo(LIBMTP_mtpdevice_t *, uint64_t * const,
			uint64_t * const, char ** const storage_description,
			char ** const volume_label);
int LIBMTP_Format_Storage(LIBMTP_mtpdevice_t *);
int LIBMTP_Get_Batterylevel(LIBMTP_mtpdevice_t *,
			    uint8_t * const,
			    uint8_t * const);
int LIBMTP_Get_Secure_Time(LIBMTP_mtpdevice_t *, char ** const);
int LIBMTP_Get_Device_Certificate(LIBMTP_mtpdevice_t *, char ** const);
int LIBMTP_Get_Supported_Filetypes(LIBMTP_mtpdevice_t *, uint16_t ** const, uint16_t * const);

/**
 * @}
 * @defgroup files The file management API.
 * @{
 */
LIBMTP_file_t *LIBMTP_new_file_t(void);
void LIBMTP_destroy_file_t(LIBMTP_file_t*);
char const * LIBMTP_Get_Filetype_Description(LIBMTP_filetype_t);
LIBMTP_file_t *LIBMTP_Get_Filelisting(LIBMTP_mtpdevice_t *);
LIBMTP_file_t *LIBMTP_Get_Filelisting_With_Callback(LIBMTP_mtpdevice_t *,
      LIBMTP_progressfunc_t const, void const * const);
LIBMTP_file_t *LIBMTP_Get_Filemetadata(LIBMTP_mtpdevice_t *, uint32_t const);
int LIBMTP_Get_File_To_File(LIBMTP_mtpdevice_t*, uint32_t, char const * const,
			LIBMTP_progressfunc_t const, void const * const);
int LIBMTP_Get_File_To_File_Descriptor(LIBMTP_mtpdevice_t*, uint32_t const, int const,
			LIBMTP_progressfunc_t const, void const * const);
int LIBMTP_Send_File_From_File(LIBMTP_mtpdevice_t *, char const * const,
	                 LIBMTP_file_t * const, LIBMTP_progressfunc_t const,
			 void const * const, uint32_t const);
int LIBMTP_Send_File_From_File_Descriptor(LIBMTP_mtpdevice_t *, int const,
	                LIBMTP_file_t * const, LIBMTP_progressfunc_t const,
			void const * const, uint32_t const);
/**
 * @}
 * @defgroup tracks The track management API.
 * @{
 */
LIBMTP_track_t *LIBMTP_new_track_t(void);
void LIBMTP_destroy_track_t(LIBMTP_track_t*);
LIBMTP_track_t *LIBMTP_Get_Tracklisting(LIBMTP_mtpdevice_t*);
LIBMTP_track_t *LIBMTP_Get_Tracklisting_With_Callback(LIBMTP_mtpdevice_t*,
      LIBMTP_progressfunc_t const, void const * const);
LIBMTP_track_t *LIBMTP_Get_Trackmetadata(LIBMTP_mtpdevice_t*, uint32_t const);
int LIBMTP_Get_Track_To_File(LIBMTP_mtpdevice_t*, uint32_t, char const * const,
			LIBMTP_progressfunc_t const, void const * const);
int LIBMTP_Get_Track_To_File_Descriptor(LIBMTP_mtpdevice_t*, uint32_t const, int const,
			LIBMTP_progressfunc_t const, void const * const);
int LIBMTP_Send_Track_From_File(LIBMTP_mtpdevice_t *,
			 char const * const, LIBMTP_track_t * const,
                         LIBMTP_progressfunc_t const,
			 void const * const, uint32_t const);
int LIBMTP_Send_Track_From_File_Descriptor(LIBMTP_mtpdevice_t *,
			 int const, LIBMTP_track_t * const,
                         LIBMTP_progressfunc_t const,
			 void const * const, uint32_t const);
int LIBMTP_Update_Track_Metadata(LIBMTP_mtpdevice_t *,
			LIBMTP_track_t const * const);
int LIBMTP_Track_Exists(LIBMTP_mtpdevice_t *, uint32_t);
/** @} */

/**
 * @}
 * @defgroup folders The folder management API.
 * @{
 */
LIBMTP_folder_t *LIBMTP_new_folder_t(void);
void LIBMTP_destroy_folder_t(LIBMTP_folder_t*);
LIBMTP_folder_t *LIBMTP_Get_Folder_List(LIBMTP_mtpdevice_t*);
LIBMTP_folder_t *LIBMTP_Find_Folder(LIBMTP_folder_t*, uint32_t const);
uint32_t LIBMTP_Create_Folder(LIBMTP_mtpdevice_t*, char *, uint32_t);
/** @} */


/**
 * @}
 * @defgroup playlists The audio/video playlist management API.
 * @{
 */
LIBMTP_playlist_t *LIBMTP_new_playlist_t(void);
void LIBMTP_destroy_playlist_t(LIBMTP_playlist_t *);
LIBMTP_playlist_t *LIBMTP_Get_Playlist_List(LIBMTP_mtpdevice_t *);
LIBMTP_playlist_t *LIBMTP_Get_Playlist(LIBMTP_mtpdevice_t *, uint32_t const);
int LIBMTP_Create_New_Playlist(LIBMTP_mtpdevice_t *, LIBMTP_playlist_t * const, uint32_t const);
int LIBMTP_Update_Playlist(LIBMTP_mtpdevice_t *, LIBMTP_playlist_t const * const);

/**
 * @}
 * @defgroup albums The audio/video album management API.
 * @{
 */
LIBMTP_album_t *LIBMTP_new_album_t(void);
void LIBMTP_destroy_album_t(LIBMTP_album_t *);
LIBMTP_album_t *LIBMTP_Get_Album_List(LIBMTP_mtpdevice_t *);
LIBMTP_album_t *LIBMTP_Get_Album(LIBMTP_mtpdevice_t *, uint32_t const);
int LIBMTP_Create_New_Album(LIBMTP_mtpdevice_t *, LIBMTP_album_t * const, uint32_t const);
int LIBMTP_Update_Album(LIBMTP_mtpdevice_t *, LIBMTP_album_t const * const);
int LIBMTP_Send_Album_Art(LIBMTP_mtpdevice_t *, uint32_t const, uint8_t * const, uint32_t const);

/**
 * @}
 * @defgroup objects The object management API.
 * @{
 */
char *LIBMTP_Get_String_From_Object(LIBMTP_mtpdevice_t *,const uint32_t, const uint16_t);
uint32_t LIBMTP_Get_U32_From_Object(LIBMTP_mtpdevice_t *,const uint32_t, const uint16_t, const uint32_t);
uint16_t LIBMTP_Get_U16_From_Object(LIBMTP_mtpdevice_t *,const uint32_t, const uint16_t, const uint16_t);
uint8_t LIBMTP_Get_U8_From_Object(LIBMTP_mtpdevice_t *,const uint32_t, const uint16_t, const uint8_t);
int LIBMTP_Set_Object_String(LIBMTP_mtpdevice_t *,const uint32_t,const uint16_t,
                             char const * const);
int LIBMTP_Set_Object_U32(LIBMTP_mtpdevice_t *,const uint32_t, const uint16_t, const uint32_t);
int LIBMTP_Set_Object_U16(LIBMTP_mtpdevice_t *,const uint32_t, const uint16_t, const uint16_t);
int LIBMTP_Set_Object_U8(LIBMTP_mtpdevice_t *,const uint32_t, const uint16_t, const uint8_t);
int LIBMTP_Get_Object_References(LIBMTP_mtpdevice_t *, const uint32_t, uint32_t **, uint32_t *);
int LIBMTP_Set_Object_References(LIBMTP_mtpdevice_t *, const uint32_t, uint32_t const * const, const uint32_t);
LIBMTP_object_t *LIBMTP_Make_List(LIBMTP_mtpdevice_t *, uint32_t *, uint32_t, uint32_t *, uint32_t);
LIBMTP_object_t *LIBMTP_Find_Object(LIBMTP_object_t *, const uint32_t);
void LIBMTP_Dump_List(LIBMTP_object_t *);
LIBMTP_object_t *LIBMTP_new_object_t(void);
void LIBMTP_destroy_object_t(LIBMTP_object_t *, uint32_t);
int LIBMTP_Delete_Object(LIBMTP_mtpdevice_t *, uint32_t);
/** @} */

/**
 * @}
 * @defgroup File mapping The file mapping API
 * @{
 */
int LIBMTP_Register_Filetype(char const * const, LIBMTP_filetype_t const,
			     uint16_t const, void const * const,
			     void const * const, void const * const);
int LIBMTP_Set_Filetype_Description(LIBMTP_filetype_t, char const * const);
int LIBMTP_Set_Constructor(LIBMTP_filetype_t, void const * const);
int LIBMTP_Set_Destructor(LIBMTP_filetype_t, void const * const);
int LIBMTP_Set_Datafunc(LIBMTP_filetype_t, void const * const);
/** @} */


/* End of C++ exports */
#ifdef __cplusplus
}
#endif

#endif /* LIBMTP_H_INCLUSION_GUARD */

